*Ŀ
*RRCnfInfo()     Parse interesting info from a RR.cnf file                  
*                                                                           
*Syntax:                                                                    
* RRCnfInfo( <cCnfFile> )                                                   
*                                                                           
*Parameters:                                                                
* <cCnfFile>     Name of the R&R configuration file to be processed         
*                                                                           
*Returns:                                                                   
* <aCnfInfo>     Array with configuration info:                             
*                [1,*]  Directory information                               
*                       [1, 1]  Default data directory                      
*                       [1, 2]  Default library directory                   
*                       [1, 3]  Default work directory                      
*                       [1, 4]  Default index directory                     
*                       [1, 5]  Default soft font directory                 
*                                                                           
*                [2,*]  Default settings                                    
*                       [2, 1]  Default index file extension                
*                       [2, 2]  Default memo file extension                 
*                       [2, 3]  Default FIF file                            
*                       [2, 4]  Dos exit allowed (.T., .F.)                 
*                                                                           
*                [3,*]  Printer 1 settings                                  
*                       [3, 1]  Printer name                                
*                       [3, 2]  Printer type                                
*                       [3, 3]  Start printer                               
*                       [3, 4]  End printer                                 
*                       [3, 5]  Bold ON                                     
*                       [3, 6]  Bold OFF                                    
*                       [3, 7]  Underline ON                                
*                       [3, 8]  Underline OFF                               
*                       [3, 9]  Italic ON                                   
*                       [3,10]  Italic OFF                                  
*                       [3,11]  10 CPI                                      
*                       [3,12]  12 CPI                                      
*                       [3,13]  Compressed                                  
*                       [3,14]  Landscape mode                              
*                       [3,15]  Portrait mode                               
*                                                                           
*                [4,*]  Printer 2 etc.                                      
*                                                                           
*                                                                           
* Developed by Ernst Peter Tamminga, CIS 100042,1760                        
*

#include "fileio.ch"              // We need this for low level IO
#define RECIDLEN    4             // Each record has a sort of header


function RRCnfInfo( cCnfFile )

local nHandle                     // For reading the .cnf file
local lIsEof   := .F.             // Don' try to read after the EOF
local cRecId                      // The record identification + lenght
local nRecType                    // Recordtype indication
local nReclen                     // Data length of the record
local cRecCont                    // Contents of the data part
local nPrinter := 0               // Printer number offset in result
local nOffset                     // Offset in resulting array

                                  // The resulting array with contents
local aCnfInfo := { array(5), array(4) }


                                       // Open .cnf file in shared/read mode
if (( nHandle := fopen(cCnfFile, FO_READ + FO_SHARED)) >= 0 )

     do while !lIsEof                  // W'll handle all records

        cRecId := space( RECIDLEN )    // Get the record header

        fread( nHandle, @cRecId, RECIDLEN )

                                       // 2 bytes is for ID#
                                       // 2 bytes for data length
        nRecType := bin2I( substr( cRecId, 1, 2 ) )
        nRecLen  := bin2I( substr( cRecId, 3, 2 ) )

        cRecCont := space( nRecLen )   // Read the record contents

        fread( nHandle, @cRecCont, nRecLen )

        do case                        // Dispatch interesting record types

        case (nRecType == 2 )          // This record type if EOF
           lIsEof := .T.

                                       // First directory settings
        case (nOffset := ascan( {11, 12, 13, 31, 20}, nRecType ) ) > 0
           aCnfInfo[1, nOffSet] := left( cRecCont, nRecLen-1 )

                                       // Default settings
        case (nOffset := ascan( {14, 17, 18}, nRecType ) ) > 0
           aCnfInfo[2, nOffSet] := left( cRecCont, nRecLen-1 )

        case nRecType == 35            // Dos exit allowed
           aCnfInfo[2, 4] := ( left(cRecCont, 1) == chr(0) )

        case nRecType == 256           // Start new printer
           aadd( aCnfInfo, array(15) )
           nPrinter := len( aCnfInfo )

                                       // Printer settings
        case (nOffset := ascan( {260, 261, 262, 263, 273, ;
                                 274, 275, 276, 277, 278, ;
                                 268, 269, 270, 271, 272}, nRecType ) ) > 0
           aCnfInfo[nPrinter, nOffset] := substr( cRecCont, 2 )

        endcase

     enddo

     fclose( nHandle )            // You're done

endif

return ( aCnfInfo )               // You want some? You'll get it

