#define VERSION "v1.2 2/7/94"

static char what[] = "@(#) rose.c " VERSION " By Alex Howansky";

/*
 *  rose.c
 *
 *  By Alex Howansky
 *     261 Boght Road
 *     Watervliet, NY 12189-1121
 *
 *  Please feel free to distribute and make changes to this code, as long
 *   as you include this header and maintain the version list.
 *
 *  Version 1.0 11/21/91 Alex Howansky
 *    initial version for DKBTrace v2.12
 *
 *  Version 1.1 8/23/93 Alex Howansky
 *    modified for use with POV-Ray v1
 *
 *  Version 1.2 2/7/94 Alex Howansky
 *    modified for use with POV-Ray v2
 */

#include <stdio.h>
#include <math.h>

void exit (int);

void main (int argc, char *argv[])
 {
   FILE *ofp = stdout, *ifp = NULL;
   int ch;
   char *ifname = NULL, *ofname = NULL, *label = "Rose";
   double theta, r;
   double begin = 0.0 * M_PI, end = 2.0 * M_PI, step = M_PI / 60;
   double n = 2.0, radius = 0.5, scale = 10.0, Rscale = 0.0;

   if (argc == 1)
    {
      fprintf(stderr,"\nrose " VERSION " by Alex Howansky\n");
      fprintf(stderr,"\nUsage: rose [-iFILE] [-oFILE] [-b#] [-e#] [-s#] [-S#] [-n#] [-r#] [-R#] [-lNAME]");
      fprintf(stderr,"\nOptions:\n");
      fprintf(stderr,"   -i   Names the node description input file.\n");
      fprintf(stderr,"   -o   Names the output file.\n");
      fprintf(stderr,"   -b   Beginning value for theta.\n");
      fprintf(stderr,"   -e   Ending value for theta.\n");
      fprintf(stderr,"   -s   Theta step value.\n");
      fprintf(stderr,"   -S   Scaling factor.\n");
      fprintf(stderr,"   -n   Multiplication value.\n");
      fprintf(stderr,"   -r   Node sphere radius.\n");
      fprintf(stderr,"   -R   Node object scaling factor.\n");
      fprintf(stderr,"   -l   Names the label of the rose union.\n\n");
      exit(1);
   }

   while (--argc > 0)
    if (argv[argc][0] == '-')
     switch (argv[argc][1])
      {
        case 'o' :
         ofname = &argv[argc][2];
         if ((ofp = fopen(ofname,"w")) == NULL)
          {
            fprintf(stderr,"Error: unable to open output file %s\n",ofname);
            exit(1);
          }
         break;
        case 'i' :
         ifname = &argv[argc][2];
         if ((ifp = fopen(ifname,"r")) == NULL)
          {
            fprintf(stderr,"Error: unable to open input file %s\n",ifname);
            exit(1);
          }
         break;
        case 'b' : begin = atof(&argv[argc][2]) * M_PI / 180; break;
        case 'e' : end = atof(&argv[argc][2]) * M_PI / 180; break;
        case 's' : step = atof(&argv[argc][2]) * M_PI / 180; break;
        case 'S' : scale = atof(&argv[argc][2]); break;
        case 'n' : n = atof(&argv[argc][2]); break;
        case 'r' : radius = atof(&argv[argc][2]); break;
        case 'R' : Rscale = atof(&argv[argc][2]); break;
        case 'l' : label = &argv[argc][2]; break;
        default  :
         fprintf(stderr,"Error: unknown option %s\n",argv[argc]);
         fprintf(stderr,"Usage: rose [-iFILE] [-oFILE] [-b#] [-e#] [-s#] [-S#] [-n#] [-r#] [-R#] [-lNAME]\n");
         exit(1);
      }

   fprintf(ofp,"/*\n   This file was generated by\n\n");
   fprintf(ofp,"   Rose " VERSION " by Alex Howansky\n\n");
   if (ifname != NULL)
     fprintf(ofp,"   i = %s\n",ifname);
   if (ofname != NULL)
     fprintf(ofp,"   o = %s\n",ofname);
   fprintf(ofp,"   b = %9.4f degrees\n",begin * 180 / M_PI);
   fprintf(ofp,"   e = %9.4f degrees\n",end * 180 / M_PI);
   fprintf(ofp,"   s = %9.4f degrees\n",step * 180 / M_PI);
   fprintf(ofp,"   S = %9.4f\n",scale);
   fprintf(ofp,"   n = %9.4f\n",n);
   fprintf(ofp,"   r = %9.4f\n",radius);
   fprintf(ofp,"   R = %9.4f\n",Rscale);
   fprintf(ofp,"   l = %s\n*/\n\n",label);

   fprintf(ofp,"#declare %s = union {\n",label);

   for (theta = begin; theta < end; theta += step)
    {
      r = scale * sin(theta * n);
      if (Rscale && (int)r == 0)
       continue;
      if (ifp == NULL)
       fprintf(ofp,"  sphere { < %f, %f, 0 >, %f }\n",r * cos(theta),r * sin(theta),Rscale == 0.0 ? radius : radius * abs(r) * Rscale);
      else
       {
         fprintf(ofp,"  object {\n");
         rewind(ifp);
         while ((ch = getc(ifp)) != EOF)
          fputc(ch,ofp);
         if (Rscale != 0.0)
          fprintf(ofp,"   scale < %f, %f, %f >\n",abs(r) * Rscale,abs(r) * Rscale,abs(r) * Rscale);
         fprintf(ofp,"    translate < %f, %f, 0 >\n",r * cos(theta),r * sin(theta));
         fprintf(ofp,"  }\n");
       }
    }

   fprintf(ofp,"}\n");
   if (ifp != NULL)
     fclose(ifp);

   fclose(ofp);
 }
