// file RMIClient.java jlouie 4.97
// a demonstration or Remote Method Invocation, Polymorphism and Active Notification
// using the ListenerInterface and Broadcaster Interface
// RMIClient [processName] [host] --> default <TestConnection> <localhost> 
                 
import java.rmi.*;
import java.net.*;
import java.awt.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import jlouie.rmi.*;

public class RMIClient extends jlouie.rmi.WFrame
                  implements ActionListener,
                             ListenerInterface {
 
   private RMIConnectionInterface connectionI= null;
   private ClientInterface clientI= null;
   private TextField tf= new TextField();
   private TextArea  ta= new TextArea();
   private TextField error= new TextField();
   private MenuItem  menuItemLogOn= new MenuItem("Log On");
   private MenuItem  menuItemLogOff= new MenuItem("Log Off");
   private MenuItem  menuItemExit= new MenuItem("Exit");
   private MenuItem  menuItemChangeAdministrator= new MenuItem("Change Administrator PW");
   private MenuItem  menuItemChangeAnonymous= new MenuItem("Change Anonymous PW");
   private MenuBar   menuBar= new MenuBar();
   private Menu      mFile= new Menu("File");
   private Menu      mEdit= new Menu("Edit");
   private Menu      mLog= new Menu("Log");
   private Menu      mRemote= new Menu("Remote");
    
   private boolean isApplication= true;
   private jlouie.rmi.LogOn logOnBox= new jlouie.rmi.LogOn();
   private jlouie.rmi.ChangeAdministratorPW changeAdministratorPW=
      new jlouie.rmi.ChangeAdministratorPW();
   
   public static final String TITLE= "RMIClient jlouie 4.97";

   public static void main(String[] args){
      String processName= "TestConnection";
      String hostName= "localhost";
      if (args.length >2) {
         System.out.println("Invalid Argument.");
         System.out.println("Use: RMIClient [processName] [hostName]");
         return;
      }
      if (args.length > 0) {
         processName= args[0];
      }
      if (args.length > 1) {
         hostName= args[1];
      }
      try {
            new RMIClient(processName, hostName);
      }
      catch(jlouie.rmi.RMIException e) {
            System.out.println(e);
      }
   }

   public RMIClient(String processName, String hostName) 
            throws jlouie.rmi.RMIException {
      super(TITLE);
      System.setSecurityManager(new RMISecurityManager());
      connectionI= RMIConnect(processName,hostName);
      if (connectionI == null) throw new RMIException("Failed to connect.");
      addNotify();
      try {
            String signOn= connectionI.getSignOn()+"\n"+
                  "Process Name: "+connectionI.getProcessName()+".\n"+
                  "Remote Host: "+connectionI.getHost()+".";
            ta.setText(signOn);
      }
      catch(java.rmi.RemoteException e) {
            error.setText(e.toString());
      }
      setLayout(new BorderLayout());
      add("North", tf);
      add("Center", ta);
      add("South", error);
      mFile.add(menuItemExit);
      menuBar.add(mFile);
      menuBar.add(mEdit);
      mLog.add(menuItemLogOn);
      mLog.add(menuItemLogOff);
      menuBar.add(mLog);
      mRemote.add(menuItemChangeAdministrator);
      menuBar.add(mRemote);
      setMenuBar(menuBar);
      updateMenus();
      setSize(300,200);
      registerListeners();
      show(); 
      tf.requestFocus();
   }
   
   // routine methods
   
   public synchronized void show() {
      setLocation(20,20);
      super.show();
   }
   
   private void registerListeners() {
      addWindowListener(this);
      tf.addActionListener(this);
      logOnBox.addListener(this);
      changeAdministratorPW.addListener(this);
      menuItemExit.addActionListener(this);
      menuItemLogOn.addActionListener(this);
      menuItemLogOff.addActionListener(this);
      menuItemChangeAdministrator.addActionListener(this);
      menuItemChangeAnonymous.addActionListener(this);
   }
   
   private void doExit() {
      setVisible(false);
      dispose();
      if (isApplication)System.exit(0);
   }
   
   private void updateMenus() {
      if (isAdministrator()) {
            menuItemChangeAdministrator.setEnabled(true);
      }
      else menuItemChangeAdministrator.setEnabled(false);
      if (clientI == null) {
            menuItemLogOn.setEnabled(true);
            menuItemLogOff.setEnabled(false);
      }
      else {
            menuItemLogOn.setEnabled(false);
            menuItemLogOff.setEnabled(true);
      }
   }
  
   
   // event implementations
   
   public void windowClosing(WindowEvent e) {
      doExit();
   }
   
   public void actionPerformed(ActionEvent e) {
      Object source= e.getSource();
      if (source instanceof TextField) {
            if (source == tf) {
              doEcho();
            }
      }
      else if (source instanceof MenuItem) {
            if (source == menuItemLogOn) logOnBox.show();
            else if (source == menuItemLogOff) logOff();
            else if (source == menuItemExit) doExit();
            else if (source == menuItemChangeAdministrator) changeAdministratorPW();
            else if (source == menuItemChangeAnonymous) changeAnonymousPW();
      }
   }
   
   // application methods
   
   // doEcho demonstrates a remote method call
   
   private void doEcho() {
      String inString= tf.getText();
      tf.selectAll();
      if (clientI != null) {
            try {
                  ta.setText(clientI.getEcho(inString));
            }
            catch(java.rmi.RemoteException e) {
                  error.setText(e.toString());
            }
      }
      else {
            error.setText("Not logged on.");
      }
   }
   
   // RMIConnect creates the remote connection
   
   private RMIConnectionInterface RMIConnect(String processName, String hostName) {
      try {
         Object o= Naming.lookup("//"+hostName+"/"+processName);
         if (! (o instanceof RMIConnectionInterface)) {
            throw new jlouie.rmi.RMIException("Warning: Internal error.");
         }
         return (RMIConnectionInterface)o;
      }
      catch (Exception e) {
         error.setText(e.toString());
         return null;
      }  
   }
   
   // logOn uses Polymorphism to support Administrators and Anonymous users
   // the remote Administrator and Anonymous objects implement the ClientInterface
   // and these remote objects may respond in a unique way.
   
   private ClientInterface logOn(String id, String pw) {
      try {
            ClientInterface clientI= connectionI.logOn(id,pw);
            String message= "Logged on as: "+id+".\n"+
                  "Your privileges: "+Constants.getPrivileges(clientI.getPrivileges())+
                  ".";
            ta.setText(message);
            error.setText("Logged on.");
            return clientI;
         }
         catch (Exception e) {
            error.setText(e.toString());
            return null;
         }
   }
   
   private void logOff() {
      clientI= null;
      updateMenus();
   }
   
   private boolean isAdministrator() {
      if (clientI == null) return false;
      try {
           return (clientI.getPrivileges() == 1);
      }
      catch(Exception e) {
            error.setText(e.toString());
            return false;
      }
   }
   
   private void changeAdministratorPW() {
      if ((changeAdministratorPW != null)&&
            (isAdministrator())) {
            changeAdministratorPW.show();
      }
      else error.setText("Access denied.");
   }
   
   private void changeAdministratorPW(String oldPW, String newPW) {
      if (isAdministrator()) {
            if ((oldPW != null) && (newPW != null)) {
                  try {
                        if (clientI.isValidPassword(oldPW)){
                              clientI.setMyPassword(newPW);
                              error.setText("Password changed.");
                        }
                        else {
                              error.setText("Invalid password.");
                        }
                  }
                  catch(Exception e) {
                        error.setText(e.toString());
                  }
            }
            else error.setText("Invalid password.");     
      }
      else error.setText("Access denied.");
   }
   private void changeAnonymousPW() {
   
   }
   
   // listener interface implementation
   
   public void notify(BroadcasterInterface broadcaster) {
      if (broadcaster instanceof LogOn) {
            Vector vMessages= broadcaster.getVector();
            String[] aMessages={Constants.ANONYMOUS_ID,Constants.ANONYMOUS_PW};
            Enumeration e= vMessages.elements();
            if (e.hasMoreElements()) {
                  Object o= e.nextElement();
                  if (o instanceof String) {
                        aMessages[0]= (String)o;
                  }      
            }
            if (e.hasMoreElements()) {
                  Object o= e.nextElement();
                  if (o instanceof String) {
                        aMessages[1]= (String)o;
                  }      
            }
            this.clientI= logOn(aMessages[0],aMessages[1]);
            updateMenus();
      }
      else if (broadcaster instanceof ChangeAdministratorPW) {
            Vector vMessages= broadcaster.getVector();
            String[] aMessages={"",""};
            Enumeration e= vMessages.elements();
            if (e.hasMoreElements()) {
                  Object o= e.nextElement();
                  if (o instanceof String) {
                        aMessages[0]= (String)o;
                  }      
            }
            if (e.hasMoreElements()) {
                  Object o= e.nextElement();
                  if (o instanceof String) {
                        aMessages[1]= (String)o;
                  }      
            }
            changeAdministratorPW(aMessages[0],aMessages[1]);
      }
   }
   
}
