/* testchg.c - tests character delimited get functions */
/* recio version 2.13, release September 4, 1995 */
/* Copyright (C) 1994-1995, William Pierpoint */

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <float.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "recio.h"

/* errors normally to stderr; but for test purposes, stdout */
#define errout  stdout

/****************************************************************************/
void                         /* return nothing                              */
    rfixnum(                 /* fix number                                  */
        REC *rp,             /* record pointer                              */
        int  errnum,         /* error number                                */
        void (*rfix)(REC *)) /* rfix function pointer                       */
/****************************************************************************/
{
  switch (errnum) {
  case R_EMISDAT:
    fprintf(errout, "...substituting zero\n");
    rsetfldstr(rp, "0");
    break;
  default:
    fprintf(errout, "...substituting best guess\n");
    rfix(rp);
    break;
  }
}

/****************************************************************************/
void                         /* return nothing                              */
    rfixtime(                /* fix time                                   */
        REC *rp,             /* record pointer                              */
        int  errnum,         /* error number                                */
        void (*rfix)(REC *)) /* rfix function pointer                       */
/****************************************************************************/
{
  switch (errnum) {
  case R_EMISDAT:
    fprintf(errout, "...substituting 1/1/70 00:00:00\n");
    rsetfldstr(rp, "1/1/70 0:0:0");
    break;
  default:
    fprintf(errout, "...substituting best guess\n");
    rfix(rp);
    break;
  }
}

/****************************************************************************/
void                         /* return nothing                              */
    rfixchar(                /* fix character                               */
        REC *rp,             /* record pointer                              */
        int  errnum,         /* error number                                */
        void (*rfix)(REC *)) /* rfix function pointer                       */
/****************************************************************************/
{
  switch (errnum) {
  case R_EMISDAT:
    fprintf(errout, "...substituting the letter N\n");
    rsetfldstr(rp, "N");
    break;
  default:
    fprintf(errout, "...substituting best guess\n");
    rfix(rp);
    break;
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
    rwarnfn(                 /* recio callback warning function             */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
  if (risvalid(rp)) {
    switch (rwarning(rp)) {
    case R_WNOREG:   /* atexit() full */
      fprintf (errout, "WARNING %s\n", rwarnstr(rp));
      break;
    case R_WEMPSTR:  /* empty data string */
    case R_WTMFMT:   /* time data incomplete */
      fprintf(errout, "WARNING reading %s at record %lu and field %u -- %s\n", 
       rnames(rp), rrecno(rp), rfldno(rp), rwarnstr(rp));
      break;
    }
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
     errnofn(                /* errno callback error function               */
        void)                /* no parameters                               */
/****************************************************************************/
{
  switch (errno) {

  /* non-fatal errors */
  case EACCES:
  case EMFILE:
    fprintf(errout, "ERROR: %s\n", strerror(errno));
    break;

  /* fatal errors (EINVAL, ENOMEM) */
  default:
    fprintf(errout, "FATAL ERROR: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
    break;
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
    rerrfn(                  /* recio callback error function               */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
  int errnum;       /* error number */

  if (risvalid(rp)) {
  
    /* if reof indicator set */
    if (reof(rp)) { 
      fprintf(errout, "ERROR reading %s: "
       "tried to read past end of file\n", rnames(rp));
    
    /* else rerror indicator set */
    } else {
 
      /* determine cause of error */
      errnum = rerror(rp);
      switch (errnum) {
 
      /* data errors */
      case R_EDOM:     /* data out of domain */
      case R_ERANGE:   /* data out of range */
      case R_EINVDAT:  /* invalid data */
      case R_EMISDAT:  /* missing data */
        fprintf(errout, "DATA ERROR reading %s at record %lu and field %u "
         "-- %s\n", rnames(rp), rrecno(rp), rfldno(rp), rerrstr(rp));
          
        /* determine context */
        switch (rcxtno(rp)) {
        case RECIN:
          
          /* determine field */
          switch (rfldno(rp)) {
          case 1:  /* the integer field i */
            rfixnum(rp, errnum, rfixi);
            break;
          
          case 2:  /* the unsigned integer field ui */
            rfixnum(rp, errnum, rfixui);
            break;

           case 3:  /* the long field l */
            rfixnum(rp, errnum, rfixl);
            break;

           case 4:  /* the unsigned long field ul */
            rfixnum(rp, errnum, rfixul);
            break;

           case 5:  /* the float field f */
            rfixnum(rp, errnum, rfixf);
            break;

          case 6:  /* the double field d */
            rfixnum(rp, errnum, rfixd);
            break;

          case 7: /* the character field ch */
            rfixchar(rp, errnum, rfixc);
            break;

          case 9: /* the time field t */
            rfixtime(rp, errnum, rfixt);
            break;

          default: /* programming error - no case for field */
            fprintf(errout, "FATAL ERROR %s -- code missing for field %u\n",
                             rnames(rp), rfldno(rp));
            exit(EXIT_FAILURE);
            break;
          }
          break;
        
        default:  /* programming error - missing context number */
          fprintf (errout, "FATAL ERROR in %s -- missing context number\n", 
                            rnames(rp));
          exit(EXIT_FAILURE);
          break;
        }
        break;
        
      /* fatal errors (R_EINVMOD, R_EINVAL, R_ENOMEM) */
      default:
        fprintf(errout, "FATAL ERROR in %s -- %s\n", rnames(rp), rerrstr(rp));
        exit(EXIT_FAILURE);
        break;
      }
    }
  
  /* invalid record pointer */
  } else {
    errnofn();
  }
}

/****************************************************************************
main
*****************************************************************************/
#include <io.h>

int main()
{
  int i;                        /* integer field */
  unsigned int ui;              /* unsigned integer field */
  long l;                       /* long field */
  unsigned long ul;             /* unsigned long field */
  float f;                      /* float field */
  double d;                     /* double field */
  int ch;                       /* character field */
  char *str = NULL;             /* string field */
  time_t t;                     /* time field */
  
  /* install error and warning functions */
  rinit(rerrfn, rwarnfn);
  
  /* set field and text delimiters */
  rsetfldch(recin, ',');
  rsettxtch(recin, '"');
  rsettmfmt(recin, "%m/%d/%y %H:%M:%S");
  
  /* if input not redirected */
  if (isatty(fileno(stdin))) {
    /* print instructions */
    puts("TESTCHG version 2.13 Copyright (C) 1994-1995, William Pierpoint");
    puts("Tests recio character delimited get functions.");
    puts("It reads nine fields, separated by commas, from the console.");
    puts("The field types are: int, unsigned int, long, unsigned long,");
    puts("                     float, double, character, string, and time\n");
    puts("Example:");
    puts("-1,1,-99999,99999,3.14159,3.14159265,Y,\"Hello\",11/1/94 12:00:00\n");
    puts("Press Ctrl-Z followed by the Enter key to exit program.");
    puts("You may begin now.\n");
  }
  
  /* loop through input */
  while (rgetrec(recin)) {

    /* if input redirected, echo record contents */
    if (!isatty(fileno(stdin))) puts(rrecs(recin));

    /* parse record */
    i   = rgeti(recin);
    ui  = rgetui(recin);
    l   = rgetl(recin);
    ul  = rgetul(recin);
    f   = rgetf(recin);
    d   = rgetd(recin);
    ch  = rgetc(recin);
    scpys(str, rgets(recin));
    t   = rgett(recin);
    
    /* print results */
    printf("\n");
    printf("         Integer field: %d\n", i);
    printf("Unsigned Integer field: %u\n", ui);
    printf("            Long field: %ld\n", l);
    printf("   Unsigned Long field: %lu\n", ul);
    printf("           Float field: %.*e\n", FLT_DIG-1, f);
    printf("          Double field: %.*e\n", DBL_DIG-1, d);
    printf("       Character field: %c\n", ch);
    printf("          String field: %s\n", str);
    printf("            Time field: %s\n\n", ctime(&t));
  }
  
  /* free dynamic string fields */
  free (str);
  
  /* check stream for error */
  if (rerror(recin)) { 
    fprintf(errout, "ERROR reading %s: %s\n", 
     rnames(recin), rerrstr(recin));
    exit (EXIT_FAILURE);
  }
  return EXIT_SUCCESS;
}
