*--------------------------------------------------------------------------
* RCmpDemo.PRG - Program to demonstrate the use of the functions
*                in the Clipper Library RCmpLib
*
* Used functions :
*
*	R_Compress ()	- Compress a file
*	R_DeComp ()	- Decompress a file
*	R_CPName ()	- Get the original name of a compressed file
*	R_CPSize ()	- Get the original size of a compressed file
*	R_FSize ()	- Get the file size of a file
*	R_IsRCmp ()	- Determine if a file is compressed by RCmpLib
*
* This demo has been written for Clipper version 5.xx
*
* Compile    :	CLIPPER RCMPDEMO /N
*
* Link       :	RTLINK   file RCMPDEMO lib RCMPLIB    - or -
*		BLINKER  file RCMPDEMO lib RCMPLIB    - or -
*		EXOSPACE file RCMPDEMO lib RCMPEXO
*
* Syntax     :  RCMPDEMO
*--------------------------------------------------------------------------
* Date       :  20/09/93
*--------------------------------------------------------------------------
* Author     :  Rolf van Gelder
*               Binnenwiertzstraat 27
*               5615 HG  EINDHOVEN
*	        THE NETHERLANDS
*
* E-Mail     :  Internet: RCROLF@urc.tue.nl
*               BitNet  : RCROLF@heitue5
*--------------------------------------------------------------------------
* (c) 1993  Rolf van Gelder, All rights reserved
*--------------------------------------------------------------------------
MEMVAR	GetList				&& To eliminate Clipper /W warning


*--------------------------------------------------------------------------
* Standard Clipper HEADER files
*--------------------------------------------------------------------------
#include "Directry.CH"


*--------------------------------------------------------------------------
* RCMPLIB header file
*--------------------------------------------------------------------------
#include "RCmpLib.CH"

*-- Initialize the array with error messages (from RCmpLib.CH)
STATIC	aErrTxt := CP_ERRMSG


*--------------------------------------------------------------------------
* STATIC CODEBLOCKS
*--------------------------------------------------------------------------

*-- "Hit any key" message
STATIC	bHitKey := { || DevPos (MaxRow(),0),DevOut('Hit any key ...'),;
                        InKey (0) }

*-- Headerline (with clear screen)
STATIC	bHeader := { || Scroll(), DevPos (0,0), ;
                        DevOut ('RCmpDemo: Demo program for RCmpLib v2.0 - '+;
                                '20/09/93       (C) 1993  Rolf v Gelder' ), ;
                        DevPos (1,0), ;
                        DevOut ( Replicate ('',80) ) }


*--------------------------------------------------------------------------
*
*                          Main function : RCmpDemo
*
*--------------------------------------------------------------------------
FUNCTION RCmpDemo

*-- Main menu
LOCAL	aMenu := { 'Compress   .DBF, .DBT, .NTX files', ;
                   'Decompress .DBF, .DBT, .NTX files', ;
                   'Decompress ALL files', ;
                   'List of ALL compressed files', ;
                   'End of Demo' }

*-- Choice
LOCAL	nChoice := 1

IF IsColor ()
   *-- Set screen color
   SetColor ( 'W+/RB' )
ENDIF

*--------------------------------------------------------------------------
* M A I N   P R O G R A M   L O O P
*--------------------------------------------------------------------------
DO WHILE .t.

   *-- Display header lines
   Eval ( bHeader )

   DevPos ( 3, 31 )
   DevOut ( '-+- MAIN  MENU -+-' )

   *-- Draw box
   @5,18 TO 11,61 DOUBLE

   *-- Display main menu
   nChoice := AChoice ( 6, 20, 10, 59, aMenu, , , nChoice )

   IF LastKey () = 27 .or. nChoice = 5
      *-- <Esc> or 'End of Demo'
      EXIT
   ENDIF

   *-- Display header lines
   Eval ( bHeader )

   DO CASE
   CASE nChoice = 1
      *-- Compress .DBF, .DBT and .NTX files

      DevPos ( 3, 0 )
      DevOut ( '>>> COMPRESSION OF *.DBF *.DBT and *.NTX FILES' )
      DevPos ( 5, 0 )

      DevOut ( 'Compression of the file----   Files size----------  ' + ;
               'Gain----  Seconds' )
      DevPos ( 6, 0 )

      *-- The function MultiDir creates a directory array containing
      *--    all files with the specified extensions
      CompArray ( MultiDir ( { '*.DBF', '*.DBT', '*.NTX' } ) )


   CASE nChoice = 2
      *-- Decompress .DBF, .DBT and .NTX files

      DevPos ( 3, 0 )
      DevOut ( '>>> DECOMPRESSION of *.DBF *.DBT and *.NTX FILES' )
      DevPos ( 5, 0 )

      *-- Note :
      *-- In the file names of compressed files the first letter of the
      *--    extension is replaced by the (#) character
      DeCompArr ( MultiDir ( { '*.#BF', '*.#BT', '*.#TX' } ) )


   CASE nChoice = 3
      *-- Decompress ALL compressed files in the current DOS directory

      DevPos ( 3, 0 )
      DevOut ( '>>> DECOMPRESSION OF ALL FILES IN THE CURRENT DIRECTORY' )
      DevPos ( 5, 0 )

      DeCompAll ()


   CASE nChoice = 4
      *-- Create a list of ALL compressed files in the current directory

      DevPos ( 3, 0 )
      DevOut ( '>>> LIST OF ALL COMPRESSED FILES IN THE CURRENT DIRECTORY' )

      CompList ()

   ENDCASE

ENDDO

DevPos ( 23, 0 )

RETURN nil


*--------------------------------------------------------------------------
*
*                             CompArray ( aFiles )
*
*--------------------------------------------------------------------------
* Function to compress files :
*    The file names of the files to compress are passed in an array
*
* INPUT
*       aFiles
*		Array (created by the DIRECTORY()-function) with information
*		about the files to be compressed
* OUTPUT
*       nil
*--------------------------------------------------------------------------
STATIC FUNCTION CompArray ( aFiles )

LOCAL	i				&& Counter
LOCAL	nFiles  := Len ( aFiles )	&& Number of files in the array
LOCAL	cOutFile			&& Name of output file
LOCAL	cInFile				&& Name of input  file
LOCAL	nRetCode			&& Return code from R_Compress()
LOCAL	nFSizeIn			&& Size of input  file
LOCAL	nFSizeOut			&& Size of output file
LOCAL	nCmpFact   := 0			&& Gain = Compression factor
LOCAL	nTBegin				&& Starting time (in secs)
LOCAL	nTEnd				&& Ending   time (in secs)

IF nFiles < 1

   ALERT ( 'No files found to compress ...' )

   RETURN nil

ENDIF

*-- Process the files in the array
FOR i := 1 TO nFiles

   cInFile  := aFiles [i,F_NAME]	&& Name of input file
   nFSizeIn := aFiles [i,F_SIZE]	&& Size of input file

   *--------------------------------------------------------------------
   * The extension of the default output file name starts with the '#'
   * sign.
   *--------------------------------------------------------------------
   cOutFile := Left ( cInFile, AT ( '.', cInfile ) ) + '#' + ;
      Right ( cInfile, 2 )

   *-- Display the file name
   QOut ( PadR ( cInFile, 12 ) + ' => ' + PadR ( cOutFile, 12 ) + '  ' )

   *-- Start timer
   nTBegin  := Seconds ()

   *-- COMPRESS THE INPUT FILE
   nRetCode := R_Compress ( cInFile )

   *-- Stop timer
   nTEnd    := Seconds ()

   IF nRetCode = CP_OKAY
      *-- Compression okay !

      *-- Determine the size of the output file
      nFSizeOut := R_FSize ( cOutFile )

      *-- Calculate the compression factor
      nCmpFact  := 100 * ( nFSizeIn - nFSizeOut ) / nFSizeIn

      *-- Show the statistics
      QQOut ( Str ( nFSizeIn, 8 ) + ' => ' + Str ( nFSizeOut, 8 ) + ;
              '  ' + Str ( nCmpFact, 6, 2 ) + ' %   ' + ;
              Str ( nTEnd - nTBegin, 7, 2 ) )

      *-- Compression was okay : original file can be deleted
      FErase ( cInFile )

   ELSE
      *-- Error during compression : display error message

      QQOut ( ' => Error: ' + aErrTxt [ nRetCode ] )

   ENDIF

* v1.0a *
   IF Row () > ( MaxRow () - 3 )
      *-- Screen full !

      *-- Hit any key
      Eval ( bHitKey )

      @6,0 Clear

      DevPos ( 6, 0 )

   ENDIF

NEXT

*-- Hit any key
Eval ( bHitKey )

RETURN nil


*--------------------------------------------------------------------------
*
*                            DeCompArr ( aFiles )
*
*--------------------------------------------------------------------------
* Function to decompress files :
*    The file names of the files to decompress are passed in an array
*
* INPUT
*       aFiles
*		Array (created by the DIRECTORY()-function) with information
*		about the files to be decompressed
* OUTPUT
*       nil
*--------------------------------------------------------------------------
FUNCTION DeCompArr ( aFiles )

LOCAL	nFiles  := Len ( aFiles )	&& Number of files in the array
LOCAL	i				&& Counter
LOCAL	cInFile				&& Name of the input file
LOCAL	nRetCode			&& Return code from R_DeComp()
LOCAL	nTBegin				&& Starting time (in secs)
LOCAL	nTEnd				&& Ending   time (in secs)

IF nFiles < 1

   ALERT ( 'No file found to decompress ...' )

   RETURN nil

ENDIF

*-- Note :
*-- The FOR ... NEXT LOOP can be nicely replaced by the AEval() function !

FOR i := 1 TO nFiles

    cInFile := aFiles [i,F_NAME]	&& Name of the input file

    *-- Display the file name
    QOut ( 'DECompressing: ' + PadR ( cInFile,15 ) )

    *-- DECOMPRESS THE INPUT FILE
    nTBegin  := Seconds ()
    nRetCode := R_DeComp ( cInFile )
    nTEnd    := Seconds ()

    IF nRetCode = CP_OKAY
       *-- Decompression okay : original file can be deleted !

       FErase ( cInFile )

       QQOut ( ' => Okay !  Time: '+Str (nTEnd-nTBegin,7,2) + ' secs.' )

   ELSE
       *-- Error decompressing file : display error message

       QQOut ( ' => Error: ' + aErrTxt [ nRetCode ] )

   ENDIF

   IF Row () > ( MaxRow () - 3 )
      *-- Screen full !

      *-- Hit any key
      Eval ( bHitKey )

      @5,0 Clear

      DevPos ( 5, 0 )

   ENDIF

NEXT

*-- Hit any key
Eval ( bHitKey )

RETURN nil


*--------------------------------------------------------------------------
*
*                            DeCompAll ( )
*
*--------------------------------------------------------------------------
* Function to decompress ALL compressed files in the current DOS directory
*
* INPUT
*       (Geen)
* OUTPUT
*       nil
*--------------------------------------------------------------------------
STATIC FUNCTION DeCompAll

LOCAL	aCmpFil  := {}			&& Array with compressed files

*-- Place all the by RCmpLib compressed files in the array <aCmpFil>
AEval ( Directory ( '*.*' ), ;
        { |dir| ;
          IF (R_IsRCmp ( dir [F_NAME] ), AAdd ( aCmpFil, dir ), nil ) } )

IF Len ( aCmpFil ) < 1
   *-- No files found ...

   Alert ( 'There are no compressed files in the current directory ...' )

ELSE

   DeCompArr ( aCmpFil )

ENDIF

RETURN nil


*--------------------------------------------------------------------------
*
*                               CompList ()
*
*--------------------------------------------------------------------------
* Displays a list of all compressed files in the current directory.
* Some additional information about the files is given.
*
* INPUT
*       (Geen)
* OUTPUT
*       nil
*--------------------------------------------------------------------------
STATIC FUNCTION CompList

LOCAL	aFiles   := Directory ( '*.*' )	&& All files in current directory
LOCAL	nFiles   := Len ( aFiles )	&& Number of files in the array
LOCAL	i				&& Counter
LOCAL	aComp    := {}			&& Output array
LOCAL	cOrgName			&& Original file name
LOCAL	nOrgSize			&& Original file size
LOCAL	nCmpFact			&& Compression factor
LOCAL	nTotFact := 0			&& Total compression factor (v1.0a)


*-- Note :
*-- The FOR ... NEXT LOOP can be nicely replaced by the AEval() function !

FOR i := 1 TO nFiles

   IF R_IsRCmp ( aFiles [i,F_NAME] )
      *-- File is compressed by RCmpLib !

      *-- Determine the original file name
      cOrgName := R_CPName ( aFiles [i,F_NAME] )

      *-- Determine the original file size
      nOrgSize := R_CPSize ( aFiles [i,F_NAME] )

      *-- Calculate the compression factor
      *--    aFiles [i,F_SIZE] = size of the compressed file
      nCmpFact := 100 * ( ( nOrgSize - aFiles [i,F_SIZE] ) / nOrgSize )

      nTotFact += nCmpFact

      *-- Format the information and add a line to the output array
      AAdd ( aComp, ;
         PadR ( cOrgName, 12 ) + '  ' + ;
         Str ( nOrgSize, 8 ) + '  ' + ;
         PadR ( aFiles [i,F_NAME], 12 ) + '  ' + ;
         Str ( aFiles [i,F_SIZE], 8 ) + '  ' + ;
         Str ( nCmpFact, 8, 2 ) + ' % ' )

   ENDIF

NEXT

IF Len ( aComp ) < 1
   *-- No files found

   Alert ( 'There are no compressed file in this directory ...' )

ELSE
   *-- Sort the array on file name
   aComp := ASort ( aComp )

   *-- Display header lines for report
   DevPos (5,7)
   DevOut ('ͻ')
   DevPos (6,7)
   DevOut (' ORIGINAL FILE            COMPRESSED FILE          COMPRESSION')
   DevPos (7,7)
   DevOut ('͹     FACTOR ')
   DevPos (8,7)
   DevOut (' FILE NAME         SIZE  FILE NAME         SIZE             ')
   DevPos (9,7)
   DevOut ('ͼ')

   DevPos ( MaxRow ()-1, 0 )
   DevOut ( 'Average compression factor : ' + ;
      Str ( nTotFact / Len ( aComp ), 7, 2 ) + ' % ' )

   DevPos ( MaxRow (), 0 )
   DevOut ( 'Press <Esc> to return to the main menu ...' )    && v1.0a

   *-- Display the array with file info
   AChoice ( 10, 9, 17, 70, aComp )

ENDIF

RETURN nil


*--------------------------------------------------------------------------
*
*                               MultiDir ()
*
*--------------------------------------------------------------------------
*
* MultiDir creates a directory array with all files that match one of the
* specified directory specifications.
*
* For example :
* aFiles := MultiDir ( { 'R*.DBF', '*.NXT' } )
*
* Results :
* The array <aFiles> containing all files that match the specification
* R*.DBF and/or *.NTX.
*
* INPUT
*       aDirSpec
*		Array with directory specifications
* OUTPUT
*       aDirectory
*		Multi dimensional array with file info.
*		The array has the same structure as the array that is
*		returned by the Clipper Directory() function.
*--------------------------------------------------------------------------
STATIC FUNCTION MultiDir ( aDirSpec )

LOCAL	aDirectory := {}		&& Return array
LOCAL	aTemp      := {}		&& Temporary array

*-- This is a nice example of the use of the AEval() function !

AEval ( aDirSpec, ;
   { |spec| aTemp := Directory ( spec ), ;
     AEval ( aTemp, { |temp| AAdd ( aDirectory, temp ) } ) } )

RETURN aDirectory
*
* EOF RCmpDemo.PRG ========================================================