/*
  Radio buttons for Clipper 5.2

  For a demonstration, run the batch file MAKEDEMO.BAT to make an EXE or use
  the command:

            rmake radiobtn /dTEST

  To use the radio buttons GET Reader in an application, recompile
  RADIOBTN.PRG without the /dTEST preprocessor directive.

  Ŀ
   The functions contained herein are the original work of Dan Comeau 
   and others and are placed in the public domain.                    
  

  Modifications:

  Version     Changes
       
    1.0        Original release.
    1.0a       Disabled GET variables now return 0; ie., when the WHEN
                expression is false.  Thanks to John Forsberg [75170,641]
                for this modification.  Note that if the user ends the READ
                with Esc, PgUp, or PgDn, the disabled GET value may not
                be 0.
    1.0b       New parameter for drawing shadow around box.  See
                RADIOBTN.CH for syntax.  Thanks (again) to John Forsberg
                [75170,641] for this modification.  Check out the
                DrawBoxShadow() function he wrote to draw a shadow around a
                box using pure Clipper code.
    1.1        VALID clause now supported.
    1.2        Horizontal spacing option, HSPACING, added.  Thanks to
                Rich Miller [70632,734] for this modification.
               New syntax: VIA RADIOBUTTONS. The previous syntax of
                WITH RADIOBUTTONS is still supported. You don't need to
                change your code because of this change.
               Made several minor internal changes to optimize display.


  I'd like to hear about any enhancements you make to these functions.  In
  fact, with your permission, I'd like to add your enhancements and make a
  new version.  Send revised source code or questions to my CompuServe
  account [70451,2312] or my mailing address:

            Dan Comeau
            603-1320 Richmond Rd
            Ottawa, Ontario, K2B 8L3, Canada.

  If you end up using all or parts of this code in a commercial or
  shareware library, I'd appreciate it if you let me know.

*/

#include "radiobtn.ch"

#include "box.ch"          // as shipped with Clipper
#include "getexit.ch"      // ''   ''     ''     ''
#include "inkey.ch"        // ''   ''     ''     ''

#define RB_LEFT   "("      // left bracket
#define RB_RIGHT  ")"      // right bracket
#define RB_YES   chr(7)   // the dot in the middle of the brackets
#define RB_NO     " "      // unselected option

#define HBRACKETSPACING  4    // space (ie.,"( ) ") between 1st bracket & text of horizontal choices
#ifndef K_SPACE               // this was finally defined in Clipper 5.2
  #define K_SPACE      32
#endif

static aAllButtons := {}   // for all the get radio buttons


/*
 
  Test function for the radio buttons.

  To make the test version, use:  rmake radiobtn /dTEST
 
*/
#ifdef TEST

  #define SCREEN_COLOR    "GR+/B"    // regular screen color
  #define SHADOWBOX_COLOR "GR+/R"    // for box with a shadow around it
  #define GREYED_COLOR    "W/B"      // color of disabled GET

  function test()

    local aGet1    := { "1 Choice1","2 Disable Buttons #2","3 Choice1" }
    local cGet1    := "Get 1"
    local cGet2    := "Get 2"
    local GetList  := {}
    local nChoice1 := 2          // initial choices for radio buttons
    local nChoice2 := 1
    local nChoice3 := 1
    local nChoice4 := 2
    local nChoice5 := 2

    set scoreboard off

    // demonstrate set key ability within a radio button GET
    setkey( K_F1, { |s,r,c| s:=savescreen(), ;
                            r:=row(),;
                            c:=col(),;
                            scroll(), ;
                            setpos(5,5), ;
                            dispout("You pressed F1.  Press any key to continue . . ."), ;
                            inkey(0), ;
                            restscreen(,,,,s),;
                            setpos(r,c) } )

    setcolor( SCREEN_COLOR )
    clear screen

    @ 2,5  say "Use the Tab, Shift-Tab, arrow, space bar, and Enter keys to move around."
    @ 3,5  say "Press F1 to show use of setkey within GETs."

    @ 5,5  say "Normal Get #1:" ;
           get cGet1

    @ 6,5  say "Buttons #1" ;
           get nChoice1 ;
           color SCREEN_COLOR ;
           via radiobuttons aGet1

    @ 6,40 say "Buttons #2" ;
           get nChoice2 ;
           when nChoice1 != 2 ;
           color SCREEN_COLOR+","+GREYED_COLOR ;
           via radiobuttons { "1 Choice2","2 Choice2","3 Choice2","4 Choice2" } ;
           nobox

    @ 7,60 get nChoice3 ;
           color SCREEN_COLOR ;
           via radiobuttons { "1 Choice3","2 Choice3","3 Choice3","4 Choice3","5 Choice3" } ;
           nobox

    @ 11,5 say "Normal Get #2:" ;
           get cGet2

    @ 13,5 say "Buttons #4:"
    @ 13,col()+1 get nChoice4 ;
                 color SCREEN_COLOR ;
                 via radiobuttons { "1 Choice4","2 Choice4","3 Choice4" } ;
                 nobox ;
                 horizontal

    @ 18,2 say "This is a radiobutton with a shadowed box drawn around it."
    @ 15,5 say"Choice 5 With Horizontal Spacing = 5 and a Long Title" ;
           get nChoice5 ;
           color SHADOWBOX_COLOR ;
           with radiobuttons { "1 Choice5","2 Choice5" } ;
           double ;
           horizontal ;
           hspacing 5 ;
           shadow

    read
    RadioBtnKill()  // reduce memory requirement by setting array to NIL

    // display values
    ? "Normal Get Values  :",cGet1, cGet2
    ? "Radio Button Values:", nChoice1, nChoice2, nChoice3, nChoice4, nChoice5

  return nil

#endif


/*
 
  Initialization for Radio Buttons.  Display title and choices.  Optionally
  draw box around choices.  Horizontal choices must fit on one line.
 
*/
function RadioBtnNew( oGet, bWhen,                      ;
                      nRow, nCol, cTitle, nChoice,      ;
                      aChoices, lNoBox, lDblBox,        ;
                      lHoriz, lShadow, nHSpacing )

  local cColorSpec    // color string
  local n             // temp variable
  local nWidth        // width of button box

  dispbegin()         // buffer the display output

    if cTitle == NIL
      cTitle := ""      // init to enable testing in len()
    endif

    if valtype( nHSpacing ) != "N"  // set default horizontal spacing
       nHSpacing := 2
    endif


    if nChoice < 1 .or. nChoice > len( aChoices )  // make sure nChoice is in valid range
      nChoice := 1
    endif

    // Add choices array to the aAllButtons array.
    aadd( aAllButtons, { oGet:Name, aChoices, nHSpacing } )

    // Ŀ
    //  Draw box around buttons 
    // 
    if ! lNoBox   // draw box around buttons
      if lHoriz   // draw horizontal box

        // find total width of aChoices choices
        nWidth := 0
        aeval( aChoices, { |c, n| nWidth += if( n == 1, 1, nHSpacing ) ;
                                          + HBRACKETSPACING ;
                                          + len( c )  } )
        nWidth := max( nWidth + 1, len( cTitle ) + 2 ) //  make sure title fits

        // draw single or double line box
        dispbox( nRow, nCol, nRow+2, nCol+nWidth+1, ;
                 if( lDblBox, B_DOUBLE, B_SINGLE )+space(1), oGet:ColorSpec )

        if lShadow // draw shadow around box
          DrawBoxShadow( nRow, nCol, nRow+2, nCol+nWidth+1 )
        endif

      else        // draw vertical box

        // find max width of aChoices choices
        nWidth := len( aChoices[1] )
        aeval( aChoices, { |c| nWidth := max(nWidth, len(c)) } )
        nWidth := max( nWidth+5, len(cTitle)+1 )    // add 5 spaces for " ( ) "

        // draw single or double line box
        dispbox( nRow, nCol, nRow+len(aChoices)+1, nCol+nWidth+2, ;
                 if( lDblBox, B_DOUBLE, B_SINGLE )+space(1), oGet:ColorSpec )

        if lShadow // draw shadow around box
          DrawBoxShadow( nRow, nCol, nRow+len(aChoices)+1, nCol+nWidth+2 )
        endif

      endif
    endif

    // Ŀ
    //  Put title at top left corner 
    // 
    if !empty( cTitle )
      if lNoBox   // no box around buttons
        @ nRow, nCol say cTitle color oGet:ColorSpec
      else        // box drawn around buttons
        @ nRow, nCol+1 say " "+cTitle+" " color oGet:ColorSpec
      endif
    endif

    // Ŀ
    //  Display radio button choices 
    // 
    // check when condition for this get; use this to set colors
    if ( bWhen == NIL ) .or. eval( bWhen, oGet )
      // normal color
      cColorSpec := oGet:ColorSpec
    else
      // failed pre-validation (ie., WHEN)
      // grey out the radio button box choices
      cColorSpec := if( (n:=at(",",oGet:ColorSpec)) > 0, ;  // find comma delimiter
                    substr(oGet:ColorSpec,n+1), ;           // remainder of color string
                    oGet:ColorSpec )                        // same color as regular
      nChoice := 0   // don't show any choices for greyed out radio buttons

      // return zero for disabled button
      oGet:VarPut( nChoice )   // update get var

    endif

    // draw the buttons
    /*
      Note: 1 is subtracted from nCol when horizontal and no box or
            title. This was needed to line up the buttons with the oGet
            supplied coordinates when DrawRadioButtons() is called from
            RadioBtnReader().
    */
    DrawRadioButtons( nRow, ;
                      nCol - if( lHoriz .and. lNoBox .and. empty( cTitle ), ;
                                 1, 0 ;
                               ), ;
                      aChoices, nChoice, nChoice, cColorSpec, ;
                      lNoBox, lHoriz, empty( cTitle ), nHSpacing ;
                    )
  dispend()

return nil


/*
* 
*  Draw Radio Buttons choices
* 
*/

// code blocks to display the buttons
#xtranslate bHORIZONTALCONTROL ;
         => { | c, n | dispout( replicate( " ", if( n == 1, if( lNoBox, 0, 1 ), nHSpacing ) ) + ;  // space before bracket
                                RB_LEFT + ;                                                        // left bracket
                                ( if( n == nCursor, nCursorPos := col() + ;                        // set position of selected button
                                                                  if( n == 1, ;
                                                                      0, ;
                                                                      nHSpacing-if( lNoBox, 0, 1 ) ), ), ;   
                                  if( n == nChoice, RB_YES, RB_NO ) ;             // bw the brackets
                                ) + ;
                                RB_RIGHT + " " + c, ;                             // right bracket + text
                                cColorSpec ;
                              ) ;
            }

#xtranslate bVERTICALCONTROL ;
         => { | c, n | setpos( row() + 1, if( lNoBox, nCol, nCol + 2 ) ), ;
                       dispout( RB_LEFT + ( if( n == nCursor, nCursorPos := row(), ), ;
                                            if( n == nChoice, RB_YES, RB_NO );
                                          ) + ;
                                RB_RIGHT + " " + c, ;
                                cColorSpec ;
                              ) ;
            }


static function DrawRadioButtons( nRow, nCol, aChoices, nChoice, nCursor, ;
                                  cColorSpec, lNoBox, lHoriz, lNoTitle, ;
                                  nHSpacing )

  local nCursorPos := 0   // cursor position (could be either row or col)

  dispbegin()

    set cursor off

    if lHoriz   // horizontal radio buttons
      if lNoBox
        setpos( nRow + if(lNoTitle,0,1), nCol )
        aeval( aChoices, bHORIZONTALCONTROL )            // show buttons
        setpos( nRow + if(lNoTitle,0,1), nCursorPos+1 )  // display cursor at this coordinate

      else   // with a box around buttons
        setpos( nRow+1, nCol+1 )
        aeval( aChoices, bHORIZONTALCONTROL )          // show buttons
        setpos( nRow+1, nCursorPos+2 )                 // display cursor at this coordinate
      endif

    else        // vertical radio buttons
      if lNoBox
        setpos( nRow-if(lNoTitle,1,0), nCol )
        aeval( aChoices, bVERTICALCONTROL )            // show buttons
        setpos( nCursorPos, nCol+1 )                   // display cursor at this coordinate
      else
        setpos( nRow, nCol )
        aeval( aChoices, bVERTICALCONTROL )            // show buttons
        setpos( nCursorPos, nCol+3 )                   // display cursor at this coordinate
      endif
    endif

    set cursor on

  dispend()

return nil


/*
 
  Radio Buttons GET Reader.
  Supports WHEN and VALID.
 
*/
function RadioBtnReader( oGet, lNoBox, lHoriz, cTitle )

  local aChoices      // radio button choices
  local cGetVar       // current get variable
  local cGreyColor    // greyed out color if WHEN condition failed
  local cSavedScreen  // to save portion of screen normally showing GET value
  local n             // temp variable
  local nChoice       // button choices (1st one is name of get variable)
  local nCursor       // button cursor (may be different than nChoice)
  local nFoundChoice  // array position of this gadget in all gadgets
  local nHSpacing     // how many spaces to leave between horizontal choices
  local nKey          // key pressed
  local nMaxChoices   // max number of choices
  local nOldChoice    // to save current choice
  local nOldCursor    // to save current cursor position
  local bHotKey       // code block for a set key that is pressed

  // initialize variables
  nFoundChoice := ascan( aAllButtons, { |a| a[1] == oGet:Name } )
  aChoices     := aAllButtons[ nFoundChoice, 2 ]
  nHSpacing    := aAllButtons[ nFoundChoice, 3 ]

  // read the GET if the WHEN condition is satisfied
  if ( GetPreValidate( oGet ) )  // note: see our own version of this udf below

    // initialize variables
    n           := 0
    // make a copy of the get var value; return zero for disabled button
    nChoice     := if( oGet:VarGet() != 0, oGet:VarGet(), 1 )
    nCursor     := nChoice        // cursor position
    nKey        := 0
    nMaxChoices := len( aChoices )


    // activate the GET for reading
    dispbegin()
    // save the 1 character spot where the GET value is about to be displayed
    cSavedScreen := savescreen( oGet:row, oGet:col, oGet:row, oGet:col )
    oGet:SetFocus()
    // restore the 1 character spot where the GET displayed its value
    restscreen( oGet:row, oGet:col, oGet:row, oGet:col, cSavedScreen )
    // redraw buttons: sets cursor under choice
    DrawRadioButtons( oGet:Row, oGet:Col, aChoices, nChoice, nCursor, ;
                      oGet:ColorSpec, lNoBox, lHoriz, empty(cTitle), ;
                      nHSpacing )
    dispend()

    do while ( oGet:ExitState == GE_NOEXIT )

      nOldChoice := nChoice      // save "old" choice before movement
      nOldCursor := nCursor      // save "old" cursor choice before movement
      nKey       := inkey(0)     // wait for a key to be pressed

      // see if a hot key was pressed

      if ( bHotKey := setkey( nKey ) ) != nil
        eval( bHotKey, procname(1), procline(1), readvar() )
        loop  // get next key
      endif

      // determine what key was pressed

      do case
      case nKey == K_ESC        // cancel selection
        oGet:ExitState := GE_ESCAPE

      case nKey == K_SPACE      // move to cursor or the next radio button choice
        if ! nCursor == nChoice
          // move choice to cursor position
          nChoice := nCursor
        else
          // move choice to next button
          nCursor := nChoice := if( nChoice == nMaxChoices, 1, nChoice+1 )
        endif

      case nKey == K_ENTER      // get to the next get
        oGet:ExitState := GE_ENTER

      case nKey == K_UP         // up arrow
        if lHoriz               // horizontal box: exit to previous get
          oGet:exitstate := GE_UP
        else                    // vertical box: move cursor up
          if nCursor == 1
            oGET:exitstate := GE_UP  // move to previous get
          else
            nCursor--
          endif
        endif

      case nKey == K_DOWN       // down arrow
        if lHoriz               // horizontal box: exit to next get
          oGET:exitstate := GE_DOWN
        else                    // vertical box: move cursor down
          if nCursor == nMaxChoices
            oGET:exitstate := GE_DOWN  // move to next get
          else
            nCursor++
          endif
        endif

      case nKey == K_LEFT       // left arrow
        if lHoriz               // horizontal box: move cursor to previous choice
          if nCursor == 1
            nCursor := nMaxChoices
            // to move to the previous get,
            // comment the line above and uncomment the next line
            // oGET:exitstate := GE_UP  // move to previous get
          else
            nCursor--
          endif
        else                    // vertical box
          // uncomment this line if you want the cursor to move to previous get
          // oGet:exitstate := GE_UP
        endif

      case nKey == K_RIGHT      // right arrow
        if lHoriz               // horizontal box: move cursor to next choice
          if nCursor == nMaxChoices
            nCursor := 1
            // to move to the next get,
            // comment the line above and uncomment the next line
            // oGET:exitstate := GE_DOWN  // move to next get
          else
            nCursor++
          endif
        else                    // vertical box
          // uncomment this line if you want the cursor to move to next get
          // oGET:exitstate := GE_DOWN
        endif

      case nKey == K_TAB        // tab: exit to next get
        oGET:exitstate := GE_DOWN

      case nKey == K_SH_TAB     // shift-tab: exit to previous get
        oGet:exitstate := GE_UP

      case nKey == K_PGUP       // page up
        oGET:ExitState := GE_WRITE

      case nKey == K_PGDN       // page down
        oGet:ExitState := GE_WRITE

      otherwise
        // handle if user pressed a key to select the first letter
        // 1st, continue search from current location
        n := ascan( aChoices, ;
                    { |c| upper( left(c,1) ) == upper ( chr(nKey) ) },;
                    nChoice+1, nMaxChoices )
        if n == 0
          // 2nd, if another not found, restart search from the top
          n := ascan( aChoices, ;
                      { |c| upper( left(c,1) ) == upper ( chr(nKey) ) },;
                      1, nChoice - 1 )
        endif
        nCursor := nChoice := if( n > 0, n, nChoice )  // move cursor if a match

      endcase

      // check if moved to new radio button selection
      if ! nOldChoice == nChoice .or. ! nOldCursor == nCursor
        DrawRadioButtons( oGet:Row, oGet:Col, aChoices, nChoice, nCursor, ;
                          oGet:ColorSpec, lNoBox, lHoriz, empty(cTitle), ;
                          nHSpacing )
      endif

      // disallow exit if the VALID condition is not satisfied
      if ! GetPostValidate( oGet )
        oGet:ExitState := GE_NOEXIT
      end

    enddo ( oGet:ExitState == GE_NOEXIT )

    oGet:VarPut( nChoice )   // update get var

    // de-activate the GET
    dispbegin()
    // save the 1 character spot where the GET value is about to be displayed
    cSavedScreen := savescreen( oGet:row, oGet:col, oGet:row, oGet:col )
    oGet:KillFocus()
    // restore the 1 character spot where the GET displayed its value
    restscreen( oGet:row, oGet:col, oGet:row, oGet:col, cSavedScreen )
    dispend()

  else
    // failed pre-validation (ie., WHEN)
    // grey out the radio button box choices
    cGreyColor := if( (n:=at(",",oGet:ColorSpec)) > 0, ;  // find comma dilimiter
                  substr(oGet:ColorSpec,n+1), ;           // remainder of color string
                  oGet:ColorSpec )                        // same color as regular

    // return zero for disabled button
    oGet:VarPut( 0 )   // update get var

    DrawRadioButtons( oGet:Row, oGet:Col, aChoices, nChoice, nCursor, ;
                      cGreyColor, lNoBox, lHoriz, empty(cTitle), ;
                      nHSpacing )

  endif

return nil


//
// A copy from Nantucket's version with some modifications.
//
/*
 GetPreValidate()
 Test entry condition (WHEN clause) for a GET.
*/
static function GetPreValidate( get )

  local when := .t.

	if ( get:preBlock <> NIL )
    when := Eval(get:preBlock, get)
  end

  if ( !when )
    get:exitState := GE_WHEN    // indicates failure

	else
		get:exitState := GE_NOEXIT		// prepares for editing

	end

return when


//
// A copy from Nantucket's version with some modifications.
//
/*
 GetPostValidate()
 Test exit condition (VALID clause) for a GET.
*/
static function GetPostValidate( get )

  local valid := .t.

	if ( get:exitState == GE_ESCAPE )
		return (.t.)					// NOTE
	end

	// check VALID condition if specified
  if ( get:postBlock <> NIL )
    valid := Eval(get:postBlock, get)
  end

return (valid)


/*
 
  Draw Shadow to the right and under Box
 
*/
static function DrawBoxShadow( nTop, nLeft, nBottom, nRight )

  // save old color
  local cOldColor := set( _SET_COLOR )

  // build bottom shadow buffer array (account for screen height)
  local BottomBuf := if( nBottom < maxrow(), ;
                         { nBottom + 1, ;
                           nLeft + 1, ;
                           nBottom + 1, ;
                           if( nRight < maxcol(), ;
                               nRight + 1, ;
                               nRight ;
                             ), ;
                           savescreen( nBottom + 1, ;
                                       nLeft + 1, ;
                                       nBottom+1, ;
                                       if( nRight < maxcol(), ;
                                           nRight + 1, ;
                                           nRight ;
                                         ) ;
                                     ) ;
                         }, ;
                         nil ;
                       )

  // build right shadow buffer array (account for screen width)
  local RightBuf := if( nRight < maxcol(), ;
                        { nTop + 1, ;
                          nRight + 1, ;
                          if( nBottom < maxrow(), ;
                              nBottom + 1, ;
                              nBottom ;
                            ), ;
                          nRight + 1, ;
                          savescreen( nTop + 1, ;
                                      nRight + 1, ;
                                      if( nBottom < maxrow(), ;
                                          nBottom + 1, nBottom ;
                                        ), ;
                                      nRight + 1 ;
                                    ) ;
                        }, ;
                        nil ;
                      )

  // code block to evaluate shadow buffer arrays
  local ShdwStrip := { | buf | ( restscreen( buf[1], buf[2], buf[3], buf[4], ;
                                   transform( buf[5], ;
                                              replicate( "X" + chr(8), ;
                                                         len( buf[5] ) * 0.5 ;
                                                       ) ;
                                            ) ;
                                           ) ;
                               ) ;
                     }

  // draw bottom shadow
  if ! BottomBuf == NIL
    eval( ShdwStrip, BottomBuf )
  endif

  // draw right shadow
  if ! RightBuf == NIL
    eval( ShdwStrip, RightBuf )
  endif

  // restore original color
  set( _SET_COLOR, cOldColor )

return nil


/*
 
  Clear the radio button array.  Do this after the READ to free up memory.
 
*/
function RadioBtnKill()
  aAllButtons := {}
return nil

*: EOF: RADIOBTN.PRG
