unit TablePrn;

{
Unit:         TABLEPRN.PAS
Description:  Simple component that on calling it's Execute method
                will print the fields of the connected DataSet
                according to their Visible, Index and DisplayWidth
                values, under their DisplayCaption headers;
Programmer:   Justin Turberville
              P.O.Box 122
              Kenton-on-Sea  6191
              South Africa
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DBTables, DBGrids, Printers, DB;

type
  TTablePrinter = class(TComponent)
  private
    FDevice:      System.Text;
    FDest:        string;
    FAppend:      Boolean;
    FDataSource:  TDataSource;
    FMargin:      Integer;
    FMarginStr:   string[15];
    FSpacing:     Integer;
    FSpaceStr:    string[7];
    FFirstRecord: Integer;
    FLastRecord:  Integer;
    FFirstField:  Integer;
    FLastField:   Integer;
    FMainHeader:  string;
    FSubHeader:   string;
    FFooter:      string;
    FBoldTitles:  Boolean;    {Titles = column headers}
    FTitlesColor: TColor;
    FMainHeaderFont: TFont;
    FSubHeaderFont:  TFont;
    FBodyFont:       TFont;   {NB: Must be fixed pitch font - ie Courrier New}
    FFooterFont:     TFont;
    procedure SetDataSource(ADataSource: TDataSource);
    procedure SetMargin(AValue: Integer);
    procedure SetSpacing(AValue: Integer);
    procedure SetMainHeaderFont(AFont: TFont);
    procedure SetSubHeaderFont(AFont: TFont);
    procedure SetBodyFont(AFont: TFont);
    procedure SetFooterFont(AFont: TFont);
  protected
    procedure WriteLine(AString: string; AFont: TFont); virtual;
    procedure WriteTitles; virtual;
    procedure WriteRow; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean; virtual;
  published
    property Destination: string read FDest write FDest;
    property Append: Boolean read FAppend write FAppend;
    property DataSource: TDataSource read FDataSource write SetDataSource;
    property Margin: Integer read FMargin write SetMargin
      default 5;
    property Spacing: Integer read FSpacing write SetSpacing
      default 1;
    property FirstRecord: Integer read FFirstRecord write FFirstRecord
      default 0;     {start at first tuple}
    property LastRecord: Integer read FLastRecord write FLastRecord
      default -1;    {adjust to actual last tuple}
    property FirstField: Integer read FFirstField write FFirstField
      default 0;     {start at first domain}
    property LastField: Integer read FLastField write FLastField
      default -1;    {adjust to actual last domain}
    property MainHeader: string read FMainHeader write FMainHeader;
    property SubHeader: string read FSubHeader write FSubHeader;
    property Footer: string read FFooter write FFooter;
    property BoldTitles: Boolean read FBoldTitles write FBoldTitles
      default False;
    property TitlesColor: TColor read FTitlesColor write FTitlesColor
      default clBlack;
    property MainHeaderFont: TFont read FMainHeaderFont write SetMainHeaderFont;
    property SubHeaderFont: TFont read FSubHeaderFont write SetSubHeaderFont;
    property BodyFont: TFont read FBodyFont write SetBodyFont;
    property FooterFont: TFont read FFooterFont write SetFooterFont;
  end;
{
procedure Register;
}
implementation

{*** Public ***}

constructor TTablePrinter.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDest := 'PRN';
  SetMargin(5);
  SetSpacing(1);
  FLastRecord  := -1;        {use FDataSet.RecordCount-1}
  FLastField   := -1;        {use FDataSet.FieldCount-1}
  FTitlesColor := clBlack;
  FMainHeaderFont := TFont.Create;
  FSubHeaderFont  := TFont.Create;
  FBodyFont       := TFont.Create;
  FFooterFont     := TFont.Create;
  with FMainHeaderFont do begin
    Height := -13;
    Name   := 'System';
    Size   := 10;
    Color  := clBlack;
    Style  := [];
    Pitch  := fpDefault;
  end;
  FSubHeaderFont.Assign(FMainHeaderFont);
  FBodyFont.Assign(FMainHeaderFont);
  FBodyFont.Name  := 'Courier New';       {must be fixed pitch!}
  FFooterFont.Assign(FMainHeaderFont);
end;

destructor TTablePrinter.Destroy;
begin
  FMainHeaderFont.Free;
  FSubHeaderFont.Free;
  FBodyFont.Free;
  FFooterFont.Free;
  inherited Destroy;
end;

function TTablePrinter.Execute: Boolean;
var
  place: TBookmark;
  i: Integer;
begin
  Result := False;    {will remain so if an exception occurs}
  if not (Assigned(FDataSource) and FDataSource.DataSet.Active) then
    Abort;                     {cannot continue}
  with FDataSource.DataSet do begin
    if (FLastField > FieldCount-1) or (FLastField < 0) then
      FLastField := FieldCount-1;
    if FFirstField > FLastField then Abort;
    if FFirstField < 0 then FFirstField := 0;
    if (FLastRecord > RecordCount-1) or (FLastRecord < 0) then
      FLastRecord := RecordCount-1;
    if FFirstRecord > FLastRecord then Abort;
    if FFirstRecord < 0 then FFirstRecord := 0;
    place := GetBookmark;
    try
      if FDest = '' then FDest := 'PRN';
      for i := 1 to Length(FDest) do
        FDest[i] := UpCase(FDest[i]);
      if FDest = 'PRN' then
        AssignPrn(FDevice) else
        AssignFile(FDevice, FDest);
      if FAppend and (FDest <> 'PRN') then
        System.Append(FDevice) else
        Rewrite(FDevice);
      DisableControls;
      First;
      WriteLine(FMainHeader, FMainHeaderFont);
      WriteLine(FSubHeader, FSubHeaderFont);
      Writeln(FDevice);
      WriteTitles;
      Printer.Canvas.Font.Assign(FBodyFont);
      i := FFirstRecord;
      while (i <= FLastRecord) and not Eof do begin
        WriteRow;
        Next;
        Inc(i);
      end;
      Writeln(FDevice);
      WriteLine(FFooter, FFooterFont);
      Result := True;     {no exceptions raised during printing}
    finally
      GotoBookmark(place);
      FreeBookmark(place);
      EnableControls;
      CloseFile(FDevice);
    end;
  end;
end;

{*** Protected ***}

procedure TTablePrinter.WriteLine(AString: string; AFont: TFont);
begin
  Printer.Canvas.Font.Assign(FBodyFont);   {same margin width as body}
  Write(FDevice, FMarginStr);
  Printer.Canvas.Font.Assign(AFont);
  Writeln(FDevice, AString);
end;

procedure TTablePrinter.WriteTitles;
var
  i: Integer;
  a: string[1];
begin
  Printer.Canvas.Font.Assign(FBodyFont);
  Write(FDevice, FMarginStr);
  with Printer.Canvas.Font do begin
    Color := FTitlesColor;
    Style := [fsUnderline];
    if FBoldTitles then
      Style := Style + [fsBold];
  end;
  for i := FFirstField to FLastField do begin
    if FDataSource.DataSet.Fields[i] is TNumericField
      then a := '' else a := '-';
    with FDataSource.DataSet.Fields[i] do
      if Visible then
        Write(FDevice, Format('%' + a + '*.*s',
          [DisplayWidth, DisplayWidth, DisplayLabel]) + FSpaceStr);
  end;
  Writeln(FDevice);
end;

procedure TTablePrinter.WriteRow;
var
  i: Integer;
  s: string;
begin
  Write(FDevice, FMarginStr);
  with FDataSource.DataSet do begin
    for i := FFirstField to FLastField do
      with Fields[i] do
        if Visible then begin
          if Fields[i] is TCurrencyField then
            s := Format('%*m', [DisplayWidth, AsFloat])
          else if Fields[i] is TFloatField then
            s := Format('%*.*f', [DisplayWidth, TFloatField(Fields[i]).Precision, AsFloat])
          else if Fields[i] is TNumericField then
            s := Format('%*.*s', [DisplayWidth, DisplayWidth, AsString])
          else  {display as string}
            s := Format('%-*.*s', [DisplayWidth, DisplayWidth, AsString]);
          Write(FDevice, s + FSpaceStr);
        end;
    Writeln(FDevice);
  end;
end;

{*** Private ***}

procedure TTablePrinter.SetDataSource(ADataSource: TDataSource);
begin
  FDataSource := ADataSource;
  FFirstRecord := 0;
  FLastRecord  := -1;
  FFirstField  := 0;
  FLastField   := -1;
end;

procedure TTablePrinter.SetMargin(AValue: Integer);
var
  i: Integer;
begin
  if FMargin <> AValue then begin
    FMargin := AValue;
    if FMargin < 0 then FMargin := 0;
    if FMargin > 15 then FMargin := 15;
    FMarginStr := '';
    for i := 1 to FMargin do
      FMarginStr := FMarginStr + ' ';
  end;
end;

procedure TTablePrinter.SetSpacing(AValue: Integer);
var
  i: Integer;
begin
  if FSpacing <> AValue then begin
    FSpacing := AValue;
    if FSpacing < 0 then FSpacing := 0;
    if FSpacing > 7 then FSpacing := 7;
    FSpaceStr := '';
    for i := 1 to FSpacing do
      FSpaceStr := FSpaceStr + ' ';
  end;
end;

procedure TTablePrinter.SetMainHeaderFont(AFont: TFont);
begin
  FMainHeaderFont.Assign(AFont);
end;

procedure TTablePrinter.SetSubHeaderFont(AFont: TFont);
begin
  FSubHeaderFont.Assign(AFont);
end;

procedure TTablePrinter.SetBodyFont(AFont: TFont);
begin
  FBodyFont.Assign(AFont);
end;

procedure TTablePrinter.SetFooterFont(AFont: TFont);
begin
  FFooterFont.Assign(AFont);
end;

{*** Registration ***}
{
procedure Register;
begin
  RegisterComponents('Samples', [TTablePrinter]);
end;
}
end.
