unit qiGroup;
{
Copyright 1996 by Justin Turberville
EMail: JustinT @ cyberjoe.co.za
}
interface

uses
  WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, ExtCtrls, DB, DBTables, SysUtils, qiField, qiQuery,
  Clipbrd;

type
  TQIGroupDialog = class(TComponent)
  private
    FAbout: TQIAboutProperty;
    FCaption:     string;
    FMaxItems:    Integer;
    FModalResult: TModalResult;
    FQueryInfo:   TQueryInfo;
    FShowGlyphs:  Boolean;
    FSortFields:  Boolean;
    procedure SetQueryInfo(Value: TQueryInfo);
    procedure SetMaxItems(Value: Integer);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    function Execute: Boolean;
    property ModalResult: TModalResult read FModalResult write FModalResult;
  published
    property About: TQIAboutProperty read FAbout write FAbout stored False;
    property Caption: string read FCaption write FCaption;
    property QueryInfo: TQueryInfo read FQueryInfo write SetQueryInfo;
    property ShowGlyphs: Boolean read FShowGlyphs write FShowGlyphs default False;
    property MaxItems: Integer read FMaxItems write SetMaxItems default 5;
    property SortFields: Boolean read FSortFields write FSortFields;
  end;

  TQIGroupForm = class(TForm)
    FieldLabel: TLabel;
    FieldList: TListBox;
    InclBtn: TSpeedButton;
    ExclBtn: TSpeedButton;
    SortLabel: TLabel;
    EditList: TListBox;
    ClearBtn: TSpeedButton;
    UpBtn: TSpeedButton;
    DownBtn: TSpeedButton;
    OkBtn: TBitBtn;
    CancelBtn: TBitBtn;
    Bevel1: TBevel;
    procedure FormShow(Sender: TObject);
    procedure ListDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure ListDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure InclBtnClick(Sender: TObject);
    procedure ExclBtnClick(Sender: TObject);
    procedure FieldListDblClick(Sender: TObject);
    procedure EditListDblClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure UpBtnClick(Sender: TObject);
    procedure DownBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure EditListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure ClearBtnClick(Sender: TObject);
    procedure FormDblClick(Sender: TObject);
    procedure ListClick(Sender: TObject);
  private
    procedure UpdateBtns;
  public
    QueryInfo: TQueryInfo;
    MaxItems: Integer;
  end;

implementation

{$R *.DFM}

type
  TEditObj = class(TObject)
    GroupItem, EditItem: TGroupItem;
    constructor Create(AItem: TObject);
    destructor Destroy; override;
    procedure AssignEdit;
    function ListName: string;
  end;

const
  sAsc = '';
  sDesc = '';

{TOrderDialog}

constructor TQIGroupDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCaption := 'Define Grouping';
  FMaxItems := 5;
end;

procedure TQIGroupDialog.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = QueryInfo) then
    QueryInfo := nil;
end;

function TQIGroupDialog.Execute: Boolean;
begin
  Result := False;
{}if FindWindow(sDelphiWdw, nil) = 0 then
    raise EComponentError.Create(sCrippled);
  if FQueryInfo = nil then
    raise EComponentError.Create(sNoQueryInfo);
  with TQIGroupForm.Create(Self) do
  try
    Caption   := FCaption;
    QueryInfo := FQueryInfo;
    MaxItems  := FMaxItems;
    FieldList.Sorted := FSortFields;
    if FShowGlyphs then begin
      OKBtn.Kind := bkOK;

      CancelBtn.Kind := bkCancel;

    end;

    FModalResult := ShowModal;
  finally

    Free;

    Result := FModalResult = mrOK;
  end;
end;

procedure TQIGroupDialog.SetQueryInfo(Value: TQueryInfo);
begin
  FQueryInfo := Value;
  if Value <> nil then
    Value.FreeNotification(Value);
end;

procedure TQIGroupDialog.SetMaxItems(Value: Integer);
begin
  if Value in [1..15] then FMaxItems := Value;
end;

{TEditObj}

constructor TEditObj.Create(AItem: TObject);
begin
  inherited Create;
  EditItem := TGroupItem.Create;
  if AItem is TGroupItem then begin         {an existing item -}
    GroupItem := TGroupItem(AItem);
    GroupItem.EditFlag := efOld;
    EditItem.Assign(GroupItem);
  end
  else begin                                 {a new item -}
    GroupItem := TGroupItem.Create;
    GroupItem.EditFlag := efNew;
    if AItem is TFieldItem then
      EditItem.Field := TFieldItem(AItem);
  end;
end;

destructor TEditObj.Destroy;
begin
  EditItem.Free;
  if GroupItem.EditFlag = efNew then   {a new item}
    GroupItem.Free else                {so destroy (should'nt be in Grouping)}
    GroupItem.EditFlag := efDelete;    {else mark as to delete if edit OK'd}
  inherited Destroy;                             {- is in Grouping list}
end;

procedure TEditObj.AssignEdit;
begin
  GroupItem.Assign(EditItem);
  GroupItem.EditFlag := efOld;
end;

function TEditObj.ListName: string;
begin
  with EditItem do
    if (Field = nil) or (Field.UserName = '') then
      Result := sUnknown else
      Result := Field.UserName;
end;

{TGroupForm}

procedure TQIGroupForm.FormShow(Sender: TObject);
var
  i: Integer;
  e: TEditObj;
begin
  with QueryInfo do begin
    for i := 0 to Grouping.Count-1 do begin
      e := TEditObj.Create(Grouping[i]);
      EditList.Items.AddObject(e.ListName, e);
    end;
    for i := 0 to Fields.Count-1 do  {add if not in grouping & can be grouped -}
      if (Grouping.ByField(Fields[i]) = nil) and (fuGroup in Fields[i].FieldUsage)
        then FieldList.Items.AddObject(Fields[i].UserName, Fields[i]);
  end;
  if EditList.Items.Count > 0 then
    EditList.ItemIndex := 0;
  UpdateBtns;
end;

procedure TQIGroupForm.ListDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  if Source = EditList then
    if Sender = EditList then
      Accept := EditList.Items.Count > 0 else           {moving grouping item}
      Accept := ExclBtn.Enabled                         {excluding field}
  else if Source = FieldList  then
    Accept := (Source <> Sender) and InclBtn.Enabled;   {including field}
end;

procedure TQIGroupForm.ListDragDrop(Sender, Source: TObject; X, Y: Integer);

  procedure Include(I: Integer);
  var
    e: TEditObj;
  begin
    e := TEditObj.Create(FieldList.Items.Objects[I]);
    with EditList do
      ItemIndex := Items.AddObject(e.ListName, e);
    FieldList.Items.Delete(I);
  end;

  procedure Exclude(I: Integer);
  var
    e: TEditObj;
  begin
    e := TEditObj(EditList.Items.Objects[I]);
    with FieldList, e.EditItem do    {add field to available list -}
      Selected[Items.AddObject(Field.UserName, Field)] := True;
    EditList.Items.Delete(I);
    e.Free;
  end;

var
  i: Integer;
begin  {ListDragDrop}
  if Source = Sender then               {a EditList drag move -}
    with EditList do begin
      i := ItemAtPos(Point(X, Y), False);
      if i >= Items.Count then Dec(i);
      Items.Move(ItemIndex, i);
      ItemIndex := i;
    end
  else if Source = FieldList then begin      {including a field -}
    i := 0;
    while i < FieldList.Items.Count do
      if FieldList.Selected[i] then
        if EditList.Items.Count >= MaxItems then Break
        else Include(i)
      else Inc(i)
  end
  else if Source = EditList then            {excluding a field -}
    with EditList do begin
      i := ItemIndex;
      Exclude(ItemIndex);
      if i > Items.Count-1 then Dec(i);
      if i > -1 then ItemIndex := i;
    end;
  UpdateBtns;
end;

procedure TQIGroupForm.InclBtnClick(Sender: TObject);
begin
  ListDragDrop(EditList, FieldList, 0, 0);
end;

procedure TQIGroupForm.ExclBtnClick(Sender: TObject);
begin
  ListDragDrop(FieldList, EditList, 0, 0);
end;

procedure TQIGroupForm.FieldListDblClick(Sender: TObject);
begin
 if InclBtn.Enabled then ListDragDrop(EditList, FieldList, 0, 0);
end;

procedure TQIGroupForm.EditListDblClick(Sender: TObject);
begin
 if ExclBtn.Enabled then ListDragDrop(FieldList, EditList, 0, 0);
end;

procedure TQIGroupForm.UpdateBtns;
begin
  with EditList do begin
    InclBtn.Enabled := (FieldList.SelCount > 0) and (Items.Count < MaxItems);
    ExclBtn.Enabled := ItemIndex > -1;
    UpBtn.Enabled   := ItemIndex > 0;
    DownBtn.Enabled := ExclBtn.Enabled and (ItemIndex < Items.Count-1);
    ClearBtn.Enabled := Items.Count > 0;
  end;
end;

procedure TQIGroupForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case Key of
    VK_INSERT: if InclBtn.Enabled then InclBtnClick(nil);
    VK_DELETE: if ExclBtn.Enabled then ExclBtnClick(nil);
  end;
end;

procedure TQIGroupForm.UpBtnClick(Sender: TObject);
begin
  with EditList do begin
    Items.Move(ItemIndex -1, ItemIndex);
    if ItemIndex < TopIndex then
      TopIndex := TopIndex -1;
  end;
  UpdateBtns;
end;

procedure TQIGroupForm.DownBtnClick(Sender: TObject);
begin
  with EditList do begin
    Items.Move(ItemIndex +1, ItemIndex);
    if ItemIndex >= TopIndex + (Height div ItemHeight) then
      TopIndex := TopIndex +1;
  end;
  UpdateBtns;
end;

procedure TQIGroupForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  if ModalResult = mrOK then
    with QueryInfo, Grouping do begin
      while Count > 0 do          {clear to restore in edited order (free if}
        Remove(Items[0], Items[0].EditFlag = efDelete);   { deleted in edit)}
      for i := 0 to EditList.Items.Count-1 do     {add edited grouping -}
        with TEditObj(EditList.Items.Objects[i]) do begin
          AssignEdit;                {assign edited properties to group item}
          Add(GroupItem);            {add grouper to Grouping list}
        end;
      ChangeNotify(Grouping);
    end;
  with EditList.Items do     {destroy edit objects -}
    for i := 0 to Count-1 do
      Objects[i].Free;
end;

procedure TQIGroupForm.EditListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if ssCtrl in Shift then
    case Key of
      VK_UP, VK_LEFT: begin
        if UpBtn.Enabled then UpBtnClick(nil);
        Key := 0;
      end;
      VK_DOWN, VK_RIGHT: begin
        if DownBtn.Enabled then DownBtnClick(nil);
        Key := 0;
      end;
      VK_HOME:
        if UpBtn.Enabled then begin
          with EditList do Items.Move(ItemIndex, 0);
          UpdateBtns;
        end;
      VK_END:
        if DownBtn.Enabled then begin
          with EditList do Items.Move(ItemIndex, Items.Count-1);
          UpdateBtns;
        end;
    end;
end;

procedure TQIGroupForm.ClearBtnClick(Sender: TObject);
begin
  while EditList.Items.Count > 0 do begin
    EditList.ItemIndex := 0;
    ExclBtnClick(nil);
  end;
end;

procedure TQIGroupForm.FormDblClick(Sender: TObject);
var
  bmp: TBitmap;
begin
  bmp := GetFormImage;
  try
    Clipboard.Assign(bmp);
  finally
    bmp.Free;
  end;
end;

procedure TQIGroupForm.ListClick(Sender: TObject);
begin
  UpdateBtns;
end;

end.
