PROGRAM QHead; {v1.25 - Free DOS utility: Get message headers from QWK files.}
{$M 5120,0,0}  { 5k stack, no heap needed }
{$N-,E- no math support needed}
{$X- function calls may not be discarded}
{$I- disable I/O checking (trap errors by checking IOResult)}

{===========================================================================}
                       (** Global declarations ... **)
{===========================================================================}

USES
  DOS, ARCID;

CONST
  cursorState : BYTE = 1;  {0..3}
  cursorData : ARRAY [0..3] OF CHAR = (#179, #47, #196, #92);

VAR
  unQWK, unARC, unARJ, unHAP, unLZH,
  unPAK, unRAR, unUC2, unZIP, unZOO : PATHSTR;

  qheader, qline : string[128];
  confnumb : WORD;
  ExtractAll : Boolean;
  QWKname : string[13];

{===========================================================================}
                   (** Custom help & exit procedure ... **)
{===========================================================================}

VAR SavedExitProc: POINTER;
PROCEDURE cursorOn; FORWARD;
FUNCTION WordToHex (i: WORD): STRING; FORWARD;

PROCEDURE CustomExit; FAR;
{---- Always exit through here ----}
CONST
  NL = #13#10;
VAR
  message: STRING [79];
BEGIN
  ExitProc := SavedExitProc;
  cursorOn;
  IF (ExitCode > 0) THEN BEGIN
    Writeln('QHead v1.25 - Free DOS utility: Extract message headers from QWK packets.');
    WriteLn ('July 8, 1996. Copyright (c) 1996 by David Daniel Anderson - Reign Ware.'+NL);
    Writeln('Usage:    QHead <QWKpacket(s)> [conference]'+NL);
    Writeln('Where:    "[conference]" is any valid DOS filename, with an embedded conference');
    Writeln('          number.  If no number is found embedded within the filename, then');
    Writeln('          *all* the conference headers in the QWK packet will be extracted.'+NL);
    Writeln('Examples: QHead c:\qwks\*.qwk cnf100.hdr');
    Writeln('          QHead c:\qwk\channel1.qwk ch-all.hdr');
    Writeln('          QHead *.qwk  [writes all headers to "QHEAD.OUT"]'+NL);
    Writeln('Note:     DOS wildcards may be used when specifying the QWKpackets.');
  END;
  IF ErrorAddr <> NIL THEN {If an unanticipated run-time error occured...}
  BEGIN
    WriteLn ('An unanticipated error occurred, please contact DDA with the following data:');
    WriteLn ('Address = ', WordToHex (Seg (ErrorAddr^)), ':', WordToHex (Ofs (ErrorAddr^)));
    WriteLn ('Code    = ', ExitCode);
    ErrorAddr := NIL; {IMPORTANT!!!}
  END
  ELSE
    IF (ExitCode IN [1..254]) THEN BEGIN
      CASE ExitCode OF
        1 : message := 'Invalid parameter on command line or parameter missing.';
        2 : message := 'No files found.  First parameter must be a valid file specification.';
        3 : message := 'The second parameter must contain a conference number.';
        5 : message := 'Not enough memory to extract MESSAGES.DAT - aborting!';
        6 : message := 'You cannot just specify a path, add "*.*" or "\*.*" for all files.';
        7 : message := 'File handling error.  File may have been corrupted or deleted!';
        ELSE  message := 'Unknown error.';
      END;
      WriteLn (#7, 'Error encountered (#', ExitCode, '):'); WriteLn (message);
    END;
END;

{===========================================================================}
                      (** Supporting subroutines ... **)
{===========================================================================}

FUNCTION WordToHex (i: WORD): STRING; {Convert a WORD variable to STRING}
CONST
  HexLetters : ARRAY [0..15] OF CHAR = '0123456789ABCDEF';
BEGIN
  WordToHex := Concat (HexLetters [Hi (i) SHR 4], HexLetters [Hi (i) AND 15],
                       HexLetters [Lo (i) SHR 4], HexLetters [Lo (i) AND 15]);
END;

PROCEDURE CheckIO;
BEGIN
  IF IOResult <> 0 THEN Halt (7);
END;

PROCEDURE cursorOn; ASSEMBLER; ASM
  mov AH, 3; mov BH, 0; Int $10; AND CH, NOT $20; mov AH, 1; Int $10;
END;

PROCEDURE cursorOff; ASSEMBLER; ASM
  mov AH, 3; mov BH, 0; Int $10; OR CH, $20; mov AH, 1; Int $10;
END;

PROCEDURE updateCursor;
BEGIN
  cursorState := Succ (cursorState) AND 3;
  Write (cursorData [cursorState], ^H);
END;

FUNCTION WhereX: BYTE; ASSEMBLER;
(* Routine from SWAG *)
ASM
  MOV AH, 3     {Ask For current cursor position}
  MOV BH, 0     { On page 0 }
  Int 10h       { Return inFormation in DX }
  Inc DL        { Bios Assumes Zero-based. Crt.WhereX Uses 1 based }
  MOV AL, DL    { Return X position in AL For use in Byte Result }
END;

FUNCTION WhereY: BYTE; ASSEMBLER;
(* Routine from SWAG *)
ASM
  MOV AH, 3    {Ask For current cursor position}
  MOV BH, 0    { On page 0 }
  Int 10h      { Return inFormation in DX }
  Inc DH       { Bios Assumes Zero-based. Crt.WhereY Uses 1 based }
  MOV AL, DH   { Return Y position in AL For use in Byte Result }
END;

PROCEDURE GotoXY (X, Y: BYTE); ASSEMBLER;
(* Routine from SWAG *)
ASM
  MOV DH, Y    { DH = Row (Y) }
  MOV DL, X    { DL = Column (X) }
  Dec DH       { Adjust For Zero-based Bios routines }
  Dec DL       { Turbo Crt.GotoXY is 1-based }
  MOV BH, 0    { Display page 0 }
  MOV AH, 2    { Call For SET CURSOR POSITION }
  Int 10h
END;

PROCEDURE WriteCharAtCursor (X: CHAR);
(* Routine from SWAG *)
VAR
  reg: REGISTERS;
BEGIN
  reg. AH := $0A;
  reg. AL := Ord (X);
  reg. BH := $00;    {* Display Page Number. * for Graphics Modes! *}
  reg. CX := 1;      {* Word for number of characters to write *}
  Intr ($10, reg);
END;

PROCEDURE ClrEol;
(* Routine by DDA *)
VAR
  NumCol: WORD ABSOLUTE $0040: $004A; { Number of CRT columns (1-based) }
  X, Y, DistanceToRight: BYTE;
BEGIN
  X := WhereX;
  Y := WhereY;
  DistanceToRight := NumCol - X;
  Write ('': DistanceToRight);
  WriteCharAtCursor (#32);
  GotoXY (X, Y);
END;

FUNCTION LeadingZero (w : WORD) : STRING;
VAR
  s : STRING;
BEGIN
  Str (w: 0, s);
  IF Length (s) = 1 THEN
    s := '0' + s;
  LeadingZero := s;
END;

PROCEDURE UpFast (VAR Str: STRING);  {** from SWAG **}
INLINE ($8C/$DA/$5E/$1F/$FC/$AC/$30/$E4/$89/$C1/$E3/$12/$BB/Ord('a')/Ord('z')/
        $AC/$38/$D8/$72/$08/$38/$F8/$77/$04/$80/$6C/$FF/$20/$E2/$F1/$8E/$DA);

FUNCTION Upper (lstr : STRING): STRING;
BEGIN
  upfast (lstr);
  Upper := lstr;
END;

FUNCTION RPad (bstr: STRING; len: BYTE): STRING;
BEGIN
  WHILE (Length (bstr) < len) DO
    bstr := bstr + #32;
  RPad := bstr;
END;

FUNCTION RTrim (InStr: STRING): STRING;
BEGIN
  WHILE (Length (InStr) > 0) AND (InStr [Length (InStr)] IN [#0, #9, #32]) DO
    Dec (InStr [0]);
  RTrim := InStr;
END;

FUNCTION LTrim (InStr: STRING): STRING;
BEGIN
  WHILE (Length (InStr) > 0) AND (InStr [1] IN [#0, #9, #32]) DO
    Delete (InStr, 1, 1);
  LTrim := InStr;
END;

FUNCTION Squeeze (ss: STRING): STRING;
VAR
  controlCHAR: CHAR;
BEGIN
  FOR controlCHAR := #0 TO #31 DO
    WHILE (Ord (ss [0]) > 0) AND (Pos (controlCHAR, ss) > 0) DO
      ss [Pos (controlCHAR, ss)] := #32;
  Squeeze := RTrim (LTrim (ss));
END;

FUNCTION IsFile (CONST FileName: PATHSTR): BOOLEAN;
VAR
  Attr  : WORD;
  cFile : FILE;
BEGIN
  Assign (cFile, FileName);
  GetFAttr (cFile, Attr);
  IF (DosError = 0) AND ((Attr AND Directory) <> Directory)
    THEN IsFile := TRUE
    ELSE IsFile := FALSE;
END;

PROCEDURE EraseFile (CONST FileName : STRING);
VAR
  cFile : FILE;
BEGIN
  IF IsFile (FileName) THEN BEGIN
    Assign (cFile, FileName);
    SetFAttr (cFile, 0);
    Erase (cFile); CheckIO;
  END;
END;

FUNCTION IsDir (CONST FileName: PATHSTR): BOOLEAN;
VAR
  Attr  : WORD;
  cFile : FILE;
BEGIN
  Assign (cFile, FileName);
  GetFAttr (cFile, Attr);
  IF (DosError = 0) AND ((Attr AND Directory) = Directory)
    THEN IsDir := TRUE
    ELSE IsDir := FALSE;
END;

FUNCTION GetFilePath (CONST PSTR: STRING; VAR sDir: DIRSTR): PATHSTR;
VAR
  jPath     : PATHSTR;  { file path,       }
  jDir      : DIRSTR;   {      directory,  }
  jName     : NAMESTR;  {      name,       }
  jExt      : EXTSTR;   {      extension.  }
BEGIN
  jPath := PSTR;
  IF jPath = '' THEN jPath := '*.*';
  IF (NOT (jPath [Length (jPath)] IN [':', '\'])) AND IsDir (jPath) THEN
    jPath := jPath + '\';
  IF (jPath [Length (jPath)] IN [':', '\']) THEN
    jPath := jPath + '*.*';

  FSplit (FExpand (jPath), jDir, jName, jExt);
  jPath := jDir + jName+ jExt;

  sDir := jDir;
  GetFilePath := jPath;
END;

{===========================================================================}
                       (** Primary subroutines ... **)
{===========================================================================}

PROCEDURE InitConfig;
VAR
  cpath : PATHSTR; {cpath, etc fully qualified pathnames of *.cfg files}
  cdir  : DIRSTR;
  cname : NAMESTR;
  cext  : EXTSTR;
  CfgFile: TEXT;
  CfgLine,
  CfgVar, CfgVal: STRING;
  equalPos: BYTE;

BEGIN
  FSplit (FExpand (ParamStr(0)), cdir, cname, cext); { break up path into components }
  cpath := cdir + cname + '.cfg';

  qheader := '';
  qline := '';

  unQWK := 'gus';

  unARC := 'pkxarc';
  unARJ := 'arj e -y';
  unHAP := 'pah e';
  unLZH := 'lha e';
  unPAK := 'pak e /wa';
  unRAR := 'rar e';
  unUC2 := 'uc e -f';
  unZIP := 'pkunzip -# -o';
  unZOO := 'zoo -extract';

  IF IsFile (cpath) THEN
  BEGIN
    Assign (CfgFile, cpath);
    Reset (CfgFile); CheckIO;
    WHILE NOT SeekEoF (CfgFile) DO
    BEGIN { find vars }
      ReadLn (CfgFile, CfgLine);
      equalPos := Pos ('=', CfgLine);
      IF (equalPos > 1) AND (Length (CfgLine) > 8) THEN BEGIN

        CfgVar := Squeeze (Upper (Copy (CfgLine, 1, equalPos - 1)));
        CfgVal := Squeeze (Upper (Copy (CfgLine, equalPos + 1, Length (CfgLine) - equalPos)));

        IF (CfgVar = 'UNQWK') THEN
          unQWK := CfgVal
        ELSE IF (CfgVar = 'UNARC') THEN
          unARC := CfgVal
        ELSE IF (CfgVar = 'UNARJ') THEN
          unARJ := CfgVal
        ELSE IF (CfgVar = 'UNHAP') THEN
          unHAP := CfgVal
        ELSE IF (CfgVar = 'UNLZH') THEN
          unLZH := CfgVal
        ELSE IF (CfgVar = 'UNPAK') THEN
          unPAK := CfgVal
        ELSE IF (CfgVar = 'UNRAR') THEN
          unRAR := CfgVal
        ELSE IF (CfgVar = 'UNUC2') THEN
          unUC2 := CfgVal
        ELSE IF (CfgVar = 'UNZIP') THEN
          unZIP := CfgVal
        ELSE IF (CfgVar = 'UNZOO') THEN
          unZOO := CfgVal

        ELSE IF (CfgVar = 'QHEADER') THEN
          qheader := CfgVal
        ELSE IF (CfgVar = 'QLINE') THEN
          qline := CfgVal

      END;
    END; { loop back to read another line }
    Close (CfgFile);
  END;
END;
{===========================================================================}

FUNCTION IsArchive (CONST SomeFile: PATHSTR): PATHSTR;
VAR
  ExCMD : PATHSTR;
  AID : STRING[15];
  FileID : ARCTYPE;
BEGIN
  ExCMD := '';
  FileID := IsArc (SomeFile);
  CASE FileID OF
    NONE : BEGIN AID := '[??? archive]'; ExCMD := ''    END;
    ARC  : BEGIN AID := '[ARC archive]'; ExCMD := unARC END;
    ARJ  : BEGIN AID := '[ARJ archive]'; ExCMD := unARJ END;
    HAP  : BEGIN AID := '[HAP archive]'; ExCMD := unHAP END;
    LZH  : BEGIN AID := '[LHA archive]'; ExCMD := unLZH END;
    PAK  : BEGIN AID := '[PAK archive]'; ExCMD := unPAK END;
    RAR  : BEGIN AID := '[RAR archive]'; ExCMD := unRAR END;
    UC2  : BEGIN AID := '[UC2 archive]'; ExCMD := unUC2 END;
    ZIP  : BEGIN AID := '[ZIP archive]'; ExCMD := unZIP END;
    ZOO  : BEGIN AID := '[ZOO archive]'; ExCMD := unZOO END
    ELSE   BEGIN AID := '[??? archive]'; ExCMD := ''    END
  END;

  Write (AID);
  IsArchive := ExCMD;
END;

FUNCTION ExtractFile (CONST ArchiveFile, FileToEx: PATHSTR; ExCMD: PATHSTR): BOOLEAN;
VAR
  X, Y, newX: BYTE;
BEGIN
  X := WhereX;
  Y := WhereY;
  IF ExCMD <> '' THEN
  BEGIN
    ExCMD := ExCMD + #32 + ArchiveFile + #32 + FileToEx;

    SwapVectors;
      Exec (GetEnv ('COMSPEC'), ' /c '+ExCMD+' >nul');
    SwapVectors;

    newX := WhereX;
    IF (Y = WhereY) and (WhereX >= newX) THEN
    BEGIN  {If we haven't moved to a new line... }
      GotoXY (X, Y);  {return to where we were at start of procedure}
      ClrEol;
    END;
    cursorOff;
    ExtractFile := IsFile (FileToEx)
  END
  ELSE ExtractFile := FALSE;
END;
{===========================================================================}

PROCEDURE ProcessHeader (VAR MSGFile: FILE; VAR TXTfile: TEXT; VAR NumChunks: INTEGER);

(* Note: the meaning of the status flag (byte 1) in the header of the QWK
   format specification is interpreted differently by different products.

   According to Robomail v1.30, an asterisk ('*') apparently means private
   and received, and the plus sign ('+') means private but NOT received.

   Most every other program and document I have encountered seem to agree
   that the meaning of those two symbols is reversed.  Therefore, in this
   program, the private and received flags will be translated into the
   following symbols:

       public, unread   =  ' '  (#32)
       public, read     =  '-'  (#45)
       private, unread  =  '*'  (#42)
       private, read    =  '+'  (#43)
*)
TYPE
  MSGDATHdr = RECORD
                Status   : CHAR;
                MSGNum   : ARRAY [1..7] OF CHAR;
                Date     : ARRAY [1..8] OF CHAR;
                Time     : ARRAY [1..5] OF CHAR;
                WhoTo    : ARRAY [1..25] OF CHAR;
                WhoFrom  : ARRAY [1..25] OF CHAR;
                Subject  : ARRAY [1..25] OF CHAR;
                PassWord : ARRAY [1..12] OF CHAR;
                ReferNum : ARRAY [1..8] OF CHAR;
                NumChunk : ARRAY [1..6] OF CHAR;
                Alive    : BYTE;
                ConfNum  : WORD;
                Reserved : ARRAY [1..3] OF CHAR;
              END;
VAR
  VErr          : INTEGER;
  MessageHeader : MSGDATHdr;
  PRIVATE       : CHAR;
BEGIN
  updateCursor;
  BlockRead (MSGFile, MessageHeader, 1);
  Val (Squeeze (MessageHeader. NumChunk), NumChunks, VErr);
  IF (VErr <> 0) THEN NumChunks := 0;
  IF NumChunks <> 0 THEN
    WITH MessageHeader DO BEGIN
      IF ExtractAll OR (confnum = ConfNumb) THEN BEGIN
        IF (Pos (status, '+*~`!#') > 0)
           THEN PRIVATE := #158
           ELSE PRIVATE := #32;
        WriteLn (TXTfile,
        '{', QWKname, ConfNum:4, ',', (RTrim (MSGNum)): 7, '}',
        PRIVATE:2, #32,
        Copy (Date, 1, 5), #32,
        Copy (WhoFrom, 1, 21), #32,
        Copy (WhoTo, 1, 21), #32,
        Copy (Subject, 1, 25), #32);
      END;
    END;
END;
{===========================================================================}

PROCEDURE ProcessFiles (VAR MSGFile: FILE; VAR TXTfile: TEXT);
VAR
  QWKrecs,
  Chunks    : INTEGER;
BEGIN
  QWKrecs := 2;                         { start at RECORD #2 }
  WHILE QWKrecs < FileSize (MSGFile) DO BEGIN
    Seek (MSGFile, QWKrecs - 1);
    ProcessHeader (MSGFile, TXTfile, Chunks);
    IF Chunks = 0
      THEN Chunks := 1;
    Inc (QWKrecs, Chunks);
  END;
END;

FUNCTION GetConfNumb (CONST PSTR: STRING): PATHSTR;
VAR
  ConfNumbpath   : PATHSTR;    { ConfNumb file path,     }
  ConfNumbdir    : DIRSTR;     {             directory,  }
  ConfNumbname   : NAMESTR;    {             name,       }
  ConfNumbext    : EXTSTR;     {             extension.  }

  sTemp : NAMESTR;
  index : BYTE;
  VErr  : INTEGER;
BEGIN
  ExtractAll := FALSE;

  ConfNumbpath := PSTR;
  IF ConfNumbpath = '' THEN
     ConfNumbpath := 'QHEAD.OUT';

  IF ConfNumbpath [1] IN ['/', '-'] THEN Halt (255);
  FSplit (FExpand (ConfNumbpath), ConfNumbdir, ConfNumbname, ConfNumbext);
  IF (ConfNumbname = '') THEN Halt (6);

  sTemp := '';

  FOR index := 1 TO Length (ConfNumbname) DO
    IF ConfNumbname [index] IN ['0'..'9'] THEN
      sTemp := sTemp + ConfNumbname [index];

  IF sTemp = '' THEN BEGIN
    ExtractAll := TRUE;
    ConfNumb := 0;
  END
  ELSE BEGIN
    Val (sTemp, confnumb, VErr);  { confnumb is a GLOBAL var }
    IF VErr <> 0 THEN Halt (3);
  END;

  GetConfNumb := ConfNumbdir + ConfNumbname + ConfNumbext;
END;

{===========================================================================}
                           (** Main program ... **)
{===========================================================================}

CONST
  MSGFileName = 'MESSAGES.DAT';

VAR
  MSGFile : FILE;
  TXTfile : TEXT;

  QWKpath  : PATHSTR;    { QWK file path. }
  QWKdir   : DIRSTR;     { QWK file dir.  }
  TXTpath  : PATHSTR;    { TXT file path. }
  fileinfo : SEARCHREC;
  ExCMD    : PATHSTR;

BEGIN
  SavedExitProc := ExitProc;
  ExitProc := @CustomExit;
  cursorOff;
  IF NOT (ParamCount in [1,2]) THEN Halt (255);

  TXTpath := GetConfNumb (ParamStr (2));
  InitConfig;
  QWKpath := GetFilePath (ParamStr (1), QWKdir);

  FindFirst (QWKpath, Archive, fileinfo); IF DosError <> 0 THEN Halt (2);
  WriteLn ('QHead v1.25 - Free QWK header extractor is now working.');
  WHILE DosError = 0 DO
  BEGIN
    QWKpath := QWKdir + fileinfo. Name;
    QWKname := RPad (fileinfo. Name, 13);
    Write ('Extracting MESSAGES.DAT from ', QWKpath, ' ... ');
    EraseFile (MSGFileName);
    ExCMD := IsArchive (QWKpath);
    IF ExtractFile (QWKpath, MSGfileName, ExCMD) THEN
    BEGIN
      WriteLn (' done!');
      Assign (MSGFile, MSGFileName);
      Reset (MSGFile, 128); CheckIO;
      Assign (TXTfile, TXTpath);
      IF IsFile (TXTpath) THEN BEGIN
        Append (TXTfile); CheckIO;
        Write ('Appending ', TXTpath);
      END
      ELSE BEGIN
        Rewrite (TXTfile); CheckIO;
        IF qheader <> '' THEN
          WriteLn (TXTfile, qheader);
        IF qline <> '' THEN
          WriteLn (TXTfile, qline);
        Write ('Creating ', TXTpath);
      END;
      Write (', and now extracting headers', #32);
      ProcessFiles (MSGFile, TXTfile);
      WriteLn (#8, ', done!');
      Close (MSGFile); CheckIO;
      Close (TXTfile); CheckIO;
      EraseFile (MSGFileName);
    END
    ELSE
      WriteLn (' - bad QWK - skipping.');
    FindNext (fileinfo);
  END;
  WriteLn ('Mission accomplished!');
END.
