
{*******************************************************}
{                                                       }
{       Turbo Pascal for Windows Run-time Library       }
{       ObjectWindows Unit                              }
{                                                       }
{       Copyright (c) 1992 Borland International        }
{                                                       }
{*******************************************************}

unit Printer;

{$R PRINTER.RES}

{$S-}

interface

uses WinTypes, WinProcs, WObjects;

{ TPrinter states }
const
  ps_Ok = 0;
  ps_InvalidDevice = -1;     { Device parameters (to set device)
                               invalid }
  ps_Unassociated = -2;      { Object not associated with a printer }

{ TPrintout banding flags }
const
  pf_Graphics  = $01;        { Current band only accepts text }
  pf_Text      = $02;        { Current band only accepts graphics }
  pf_Both      = $03;        { Current band accepts both text and
                               graphics }

{ TPrintout represents the physical printed document which is to
  sent to a printer to be printed. TPrintout does the rendering of
  the document onto the printer.  For every document, or document
  type, a cooresponding TPrintout class should be created. }

type
  PPrintout = ^TPrintout;
  TPrintout = object(TObject)
    Title: PChar;
    Banding: Boolean;
    ForceAllBands: Boolean;
    constructor Init(ATitle: PChar);
    destructor Done; virtual;
    procedure PrintPage(DC: HDC; Page: Word; Size: TPoint; var Rect: TRect;
      Flags: Word); virtual;
    function IsNextPage: Boolean; virtual;
   end;

{ TPrinter represent the physical printer device.  To print a
  TPrintout, send the TPrintout to the TPrinter's Print method. }

  PPrinter = ^TPrinter;
  TPrinter = object(TObject)
    Device, Driver, Port: PChar;        { Printer device description }
    Status: Integer;                    { Device status, error is <> ps_Ok }
    Error: Integer;                     { < 0 if error occured during print }
    DeviceModule: THandle;              { Handle to printer driver module }
    DeviceMode: TDeviceMode;            { Function pointer to DevMode }
    ExtDeviceMode: TExtDeviceMode;      { Function pointer to ExtDevMode }
    DevSettings: PDevMode;              { Local copy of printer settings }
    DevSettingSize: Integer;            { Size of the printer settings }

    constructor Init;
    destructor Done; virtual;
    procedure ClearDevice;
    procedure Configure(Window: PWindowsObject);
    function GetDC: HDC; virtual;
    function InitAbortDialog(Parent: PWindowsObject;
      Title: PChar): PDialog; virtual;
    function InitSetupDialog(Parent: PWindowsObject): PDialog; virtual;
    procedure ReportError(Printout: PPrintout); virtual;
    procedure SetDevice(ADevice, ADriver, APort: PChar);
    procedure Setup(Parent: PWindowsObject);
    function Print(ParentWin: PWindowsObject; Printout: PPrintout): Boolean;
  end;

{ TPrinterSetupDlg is a dialog to modify which printer a TPrinter
  object is attached to.  It displays the all the active printers
  in the system allowing the user to select the desired printer.
  The dialog also allow the user to call up the printer's
  "setup" dialog for further configuration of the printer. }

const
  id_Combo = 100;
  id_Setup = 101;

type
  PPrinterSetupDlg = ^TPrinterSetupDlg;
  TPrinterSetupDlg = object(TDialog)
    Printer: PPrinter;
    constructor Init(AParent: PWindowsObject; TemplateName: PChar;
      APrinter: PPrinter);
    destructor Done; virtual;
    procedure TransferData(TransferFlag: Word); virtual;
    procedure IDSetup(var Msg: TMessage);
      virtual id_First + id_Setup;
    procedure Cancel(var Msg: TMessage);
      virtual id_First + id_Cancel;
  private
    OldDevice, OldDriver, OldPort: PChar;
    DeviceCollection: PCollection;
  end;

const
  id_Title  = 101;
  id_Device = 102;
  id_Port   = 103;

type
  PPrinterAbortDlg = ^TPrinterAbortDlg;
  TPrinterAbortDlg = object(TDialog)
    constructor Init(AParent: PWindowsObject; Template, Title,
      Device, Port: PChar);
    procedure SetupWindow; virtual;
    procedure WMCommand(var Msg: TMessage);
      virtual wm_First + wm_Command;
  end;

implementation

uses Strings;

const
  sr_On             = 32512;
  sr_ErrorTemplate  = 32513;
  sr_OutOfMemory    = 32514;
  sr_OutOfDisk      = 32515;
  sr_PrnCancel      = 32516;
  sr_PrnMgrAbort    = 32517;
  sr_GenError       = 32518;
  sr_ErrorCaption   = 32519;

const
  UserAbort: Boolean = False;

{ TPrintout -------------------------------------------------------- }

constructor TPrintout.Init(ATitle: PChar);
begin
  TObject.Init;
  Title := StrNew(ATitle);
  Banding := False;
  ForceAllBands := True;
end;

destructor TPrintout.Done;
begin
  StrDispose(Title);
  TObject.Done;
end;

procedure TPrintout.PrintPage(DC: HDC; Page: Word; Size: TPoint;
  var Rect: TRect; Flags: Word);
begin
  Abstract;
end;

function TPrintout.IsNextPage: Boolean;
begin
  IsNextPage := False;
end;

{ FetchStr --------------------------------------------------------- }
{   Returns a pointer to the first comma delimited field pointed to  }
{   by Str. It replaces the comma with a #0 and moves the Str to the }
{   beginning of the next string (skipping white space).  Str will   }
{   will point to a #0 character if no more strings are left.  This  }
{   routine is used to fetch strings out of text retrieved from      }
{   WIN.INI.                                                         }

function FetchStr(var Str: PChar): PChar;
begin
  FetchStr := Str;
  if Str = nil then Exit;
  while (Str^ <> #0) and (Str^ <> ',') do
    Str := AnsiNext(Str);
  if Str^ = #0 then Exit;
  Str^ := #0;
  Inc(Str);
  while Str^ = ' ' do
    Str := AnsiNext(Str);
end;

{ TReplaceStatic --------------------------------------------------- }

type
  PReplaceStatic = ^TReplaceStatic;
  TReplaceStatic = object(TStatic)
    Text: PChar;
    constructor InitResource(AParent: PWindowsObject; ResourceID: Word;
      AText: PChar);
    destructor Done; virtual;
    procedure SetupWindow; virtual;
  end;

constructor TReplaceStatic.InitResource(AParent: PWindowsObject; ResourceID: Word;
  AText: PChar);
begin
  TStatic.InitResource(AParent, ResourceID, 0);
  Text := StrNew(AText);
end;

destructor TReplaceStatic.Done;
begin
  StrDispose(Text);
  TStatic.Done;
end;

procedure TReplaceStatic.SetupWindow;
var
  A: array[0..80] of Char;
  B: array[0..80] of Char;
begin
  TStatic.SetupWindow;
  GetText(A, SizeOf(A) - 1);
  WVSPrintF(B, A, Text);
  SetText(B);
end;

{ TPrinterAbortDlg ----------------------------------------------------- }

constructor TPrinterAbortDlg.Init(AParent: PWindowsObject; Template,
  Title, Device, Port: PChar);
var
  Tmp: PWindowsObject;
begin
  TDialog.Init(AParent, Template);
  Tmp := New(PReplaceStatic, InitResource(@Self, id_Title, Title));
  Tmp := New(PReplaceStatic, InitResource(@Self, id_Device, Device));
  Tmp := New(PReplaceStatic, InitResource(@Self, id_Port, Port));
end;

procedure TPrinterAbortDlg.SetupWindow;
begin
  TDialog.SetupWindow;
  EnableMenuItem(GetSystemMenu(HWindow, False), sc_Close, mf_Grayed);
end;

procedure TPrinterAbortDlg.WMCommand(var Msg: TMessage);
begin
  UserAbort := True;
end;

{ TPrinter --------------------------------------------------------- }

{ This object type is an ecapsulation around the Windows printer
  device interface.  After the object is initialized the Status
  field must be check to see of the object was created correctly.
  Examples:
    Creating a default device printing object:

      DefaultPrinter := New(PPrinter, Init);

    Creating a device for a specific printer:

      PostScriptPrinter := New(PPrinter, Init);
      PostScriptPrinter^.SetDevice('PostScript Printer',
        'PSCRIPT','LPT2:');

    Allowing the user to configure the printer:

      DefaultPrinter^.Configure(MyWindow);
}

{ Initialize the TPrinter object assigned to the default printer }

constructor TPrinter.Init;
begin
  TObject.Init;
  Device := nil;
  Driver := nil;
  Port := nil;
  DeviceModule := 0;
  DevSettings := nil;
  Error := 0;
  SetDevice(nil, nil, nil);  { Associate with default printer }
end;

{ Deallocate allocated resources }

destructor TPrinter.Done;
begin
  ClearDevice;
  TObject.Done;
end;

{ Clears the association of this object with the current device }

procedure TPrinter.ClearDevice;
begin
  StrDispose(Device); Device := nil;
  StrDispose(Driver); Driver := nil;
  StrDispose(Port); Port := nil;
  if DeviceModule >= 32 then
  begin
    FreeLibrary(DeviceModule);
    DeviceModule := 0;
  end;
  if DevSettings <> nil then
    FreeMem(DevSettings, DevSettingSize);
  Status := ps_Unassociated;
end;

{ Associates the printer object with a new device. If the ADevice
  parameter is nil the Windows default printer is used, otherwise,
  the parameters must be ones contained in the [devices] section
  of the WIN.INI file. }

procedure TPrinter.SetDevice(ADevice, ADriver, APort: PChar);
var
  DriverName: array[0..80] of Char;
  DevModeSize: Integer;
  StubDevMode: TDevMode;

  procedure GetDefaultPrinter;
  var
    Printer: array[0..80] of Char;
    Cur: PChar;

  begin
    GetProfileString('windows', 'device', '', Printer,
      SizeOf(Printer) - 1);
    Cur := Printer;
    Device := StrNew(FetchStr(Cur));
    Driver := StrNew(FetchStr(Cur));
    Port := StrNew(FetchStr(Cur));
  end;

  function Equal(S1, S2: PChar): Boolean;
  begin
    Equal := (S1 <> nil) and (S2 <> nil) and
      (StrComp(S1, S2) = 0);
  end;

begin
  if Equal(Device, ADevice) and Equal(Driver, ADriver) and
    Equal(Port, APort) then Exit;
  ClearDevice;
  if ADevice = nil then
    GetDefaultPrinter
  else
  begin
    Device := StrNew(ADevice);
    Driver := StrNew(ADriver);
    Port := StrNew(APort);
  end;
  Status := ps_Ok;
  StrLCopy(DriverName, Driver, SizeOf(DriverName) - 1);
  StrLCat(DriverName, '.DRV', SizeOf(DriverName) - 1);
  DeviceModule := LoadLibrary(DriverName);
  if DeviceModule < 32 then Status := ps_InvalidDevice
  else
  begin
    { Grab the DevMode procedures }
    @ExtDeviceMode := GetProcAddress(DeviceModule, 'ExtDeviceMode');
    @DeviceMode := GetProcAddress(DeviceModule, 'DeviceMode');
    if (@DeviceMode = nil) and (@ExtDeviceMode = nil) then
      Status := ps_InvalidDevice;
    if @ExtDeviceMode <> nil then
    begin
      { Get default printer settings }
      DevSettingSize := ExtDeviceMode(0, DeviceModule, StubDevMode,
        Device, Port, StubDevMode, nil, 0);
      GetMem(DevSettings, DevSettingSize);
      ExtDeviceMode(0, DeviceModule, DevSettings^, Device, Port,
        DevSettings^, nil, dm_Out_Buffer);
    end
    else
      DevSettings := nil; { Cannot use local settings }
  end;
end;

{ Configure brings up a dialog as a child of the given window
  to configure the associated printer driver. }

procedure TPrinter.Configure(Window: PWindowsObject);
begin
  if Status = ps_Ok then
    if @ExtDeviceMode = nil then { driver is only supports DevMode }
      { If DeviceMode = nil, Status will <> ps_Ok }
      DeviceMode(Window^.HWindow, DeviceModule, Device, Port)
    else
      { Request driver to modify local copy of printer settings }
      ExtDeviceMode(Window^.HWindow, DeviceModule, DevSettings^, Device,
        Port, DevSettings^, nil, dm_In_Buffer or dm_Prompt or
          dm_Out_Buffer);
end;

{ Returns a device context for the associated printer, 0 if an
  error occurs or Status is <> ps_Ok }

function TPrinter.GetDC: HDC;
begin
  if Status = ps_Ok then
    GetDC := CreateDC(Driver, Device, Port, DevSettings)
  else GetDC := 0;
end;

{ Abort procedure used for printing }
function AbortProc(Prn: HDC; Code: Integer): Boolean; export;
var
  Msg: TMsg;
begin
  while not UserAbort and PeekMessage(Msg, 0, 0, 0, pm_Remove) do
    if not Application^.ProcessAppMsg(Msg) then
    begin
      TranslateMessage(Msg);
      DispatchMessage(Msg);
    end;
  AbortProc := not UserAbort;
end;

function TPrinter.Print(ParentWin: PWindowsObject;
  Printout: PPrintout): Boolean;
type
  TAbortProc = function (Prn: HDC; Code: Integer): Boolean;
var
  PageSize: TPoint;
  PrnDC: HDC;
  BandRect: TRect;
  Banding: Boolean;
  FirstBand: Boolean;
  Flags: Word;
  AbortProcInst: TFarProc;
  Dlg: PWindowsObject;
  UseBandInfo: Boolean;
  PageNumber: Word;

procedure CalcBandingFlags;
type
  TBandInfoStruct = record
    fGraphicsFlag: Bool;
    fTextFlag: Bool;
    GraphcisRect: TRect;
  end;
var
  BandInfoRec: TBandInfoStruct;
  pFlags: Word;
begin
  { Calculate text verses graphics banding }
  if UseBandInfo then
  begin
    Escape(PrnDC, BandInfo, SizeOf(TBandInfoStruct), nil, @BandInfoRec);
    if BandInfoRec.fGraphicsFlag then pFlags := pf_Graphics;
(*    if BandInfoRec.fTextFlag then pFlags := pf_Text; *)
    if BandInfoRec.fTextFlag then pFlags := pFlags or pf_Text;
    Flags := (Flags and not pf_Both) or pFlags;
  end
  else
  begin
    { If a driver does not support BandInfo the Microsoft
      Recommended way of determining text only bands is if
      the first band is the full page, all others are
      graphcis only.  Otherwise it handles both. }
    if FirstBand and (LongInt((@BandRect.left)^) = 0)
       and (BandRect.right = PageSize.X) and
       (BandRect.bottom = PageSize.Y) then
      Flags := pf_Text
    else
      if Flags and pf_Both = pf_Text then
        { All other bands are graphics only }
        Flags := (Flags and not pf_Both) or pf_Graphics
      else
        Flags := Flags or pf_Both;
  end;

  FirstBand := False;
end;

begin
  Print := False; { Assume error occured }

  Error := 0;

  if Printout = nil then Exit;
  if ParentWin = nil then Exit;

  PrnDC := GetDC;
  if PrnDC = 0 then Exit;

  Dlg := Application^.MakeWindow(InitAbortDialog(ParentWin,
    Printout^.Title));

  if Dlg = nil then
  begin
    DeleteDC(PrnDC);
    Exit;
  end;

  EnableWindow(ParentWin^.HWindow, False);

  AbortProcInst := MakeProcInstance(@AbortProc, hInstance);
  Escape(PrnDC, SetAbortProc, 0, PChar(AbortProcInst), nil);

  { Get the page size }
  PageSize.X := GetDeviceCaps(PrnDC, HorzRes);
  PageSize.Y := GetDeviceCaps(PrnDC, VertRes);

  { Only band if the user requests banding and the printer
    supports banding }
  Banding := Printout^.Banding and
    (GetDeviceCaps(PrnDC, RasterCaps) or rc_Banding <> 0);

  if not Banding then
  begin
    { Set the banding rectangle to full page }
    LongInt((@BandRect.left)^) := 0;
    TPoint(Pointer(@BandRect.right)^) := PageSize;

  end
  else
  begin
    { Only use BandInfo if supported (note: using Flags as a temporary) }
    Flags := BandInfo;
    UseBandInfo :=
      Escape(PrnDC, QueryEscSupport, SizeOf(Flags), @Flags, nil) <> 0;
  end;

  Flags := pf_Both;

  Error := Escape(PrnDC, StartDoc, StrLen(Printout^.Title),
    Printout^.Title, nil);
  PageNumber := 1;
  if Error > 0 then
  begin
    repeat
      if Banding then
      begin
        FirstBand := True;
        Error := Escape(PrnDC, NextBand, 0, nil, @BandRect);
      end;
      repeat
        { Call the abort proc between bands or pages }
        TAbortProc(AbortProcInst)(PrnDC, 0);

        if Banding then
        begin
          CalcBandingFlags;
          if (Printout^.ForceAllBands) and
             (Flags and pf_Both = pf_Text) then
            SetPixel(PrnDC, 0, 0, 0);
        end;

        if Error > 0 then
        begin
          Printout^.PrintPage(PrnDC, PageNumber, PageSize, BandRect, Flags);
          if Banding then
            Error := Escape(PrnDC, NextBand, 0, nil, @BandRect);
        end;
      until (Error <= 0) or not Banding or IsRectEmpty(BandRect);

      { NewFrame should only be called if not banding }
      if (Error > 0) and not Banding then
        Error := Escape(PrnDC, NewFrame, 0, nil, nil);

      Inc(PageNumber);
    until (Error <= 0) or not Printout^.IsNextPage;

    { Tell GDI the document is finished }
    if Error > 0 then
      if Banding and UserAbort then
        Escape(PrnDC, AbortDoc, 0, nil, nil)
      else
        Escape(PrnDC, EndDoc, 0, nil, nil);
  end;

  { Free allocated resources }
  FreeProcInstance(AbortProcInst);
  EnableWindow(ParentWin^.HWindow, True);
  Dispose(Dlg, Done);
  DeleteDC(PrnDC);

  if Error and sp_NotReported <> 0 then
    ReportError(Printout);

  Print := (Error > 0) and not UserAbort;

  UserAbort := False;
end;

function TPrinter.InitAbortDialog(Parent: PWindowsObject;
  Title: PChar): PDialog;
var
  Dlg: PDialog;
begin
  InitAbortDialog := New(PPrinterAbortDlg, Init(Parent, 'AbortDialog',
    Title, Device, Port));
end;

function TPrinter.InitSetupDialog(Parent: PWindowsObject): PDialog;
begin
  InitSetupDialog := New(PPrinterSetupDlg, Init(Parent, 'PrinterSetup',
    @Self));
end;

procedure TPrinter.Setup(Parent: PWindowsObject);
begin
  Application^.ExecDialog(InitSetupDialog(Parent));
end;

procedure TPrinter.ReportError(Printout: PPrintout);
var
  ErrorMsg: array[0..80] of Char;
  ErrorCaption: array[0..80] of Char;
  ErrorTemplate: array[0..40] of Char;
  ErrorStr: array[0..40] of Char;
  ErrorId: Word;
  Msg, Title: PChar;
begin
  case Error of
    sp_AppAbort:    ErrorId := sr_PrnCancel;
    sp_Error:       ErrorId := sr_GenError;
    sp_OutOfDisk:   ErrorId := sr_OutOfDisk;
    sp_OutOfMemory: ErrorId := sr_OutOfMemory;
    sp_UserAbort:   ErrorId := sr_PrnMgrAbort;
  else
    Exit;
  end;

  LoadString(hInstance, sr_ErrorTemplate, ErrorTemplate,
    SizeOf(ErrorTemplate));
  LoadString(hInstance, ErrorId, ErrorStr, SizeOf(ErrorStr));
  Title := Printout^.Title;
  Msg := ErrorStr;
  WVSPrintF(ErrorMsg, ErrorTemplate, Title);
  LoadString(hInstance, sr_ErrorCaption, ErrorCaption,
    SizeOf(ErrorCaption));
  MessageBox(0, ErrorMsg, ErrorCaption, mb_Ok or mb_IconStop);
end;

{ TPrinterSetupDlg ------------------------------------------------- }

{ TPrinterSetupDlg assumes the template passed has a ComboBox with
  the control ID of 100 and a "Setup" button with id 101 }

const
  pdStrWidth = 80;

type
  PTransferRec = ^TTransferRec;
  TTransferRec = record
    Strings: PCollection;
    Selected: array[0..0] of Char;
  end;

  PDeviceRec = ^TDeviceRec;
  TDeviceRec = record
    Driver, Device, Port: PChar;
  end;

  PDeviceCollection = ^TDeviceCollection;
  TDeviceCollection = object(TCollection)
    procedure FreeItem(P: Pointer); virtual;
  end;

procedure TDeviceCollection.FreeItem(P: Pointer);
begin
  with PDeviceRec(P)^ do
  begin
    StrDispose(Driver);
    StrDispose(Device);
    StrDispose(Port);
  end;
  Dispose(PDeviceRec(P));
end;

constructor TPrinterSetupDlg.Init(AParent: PWindowsObject;
  TemplateName: PChar; APrinter: PPrinter);
var
  tmp: PComboBox;
  Devices,                                  { List of devices from the
                                              WIN.INI }
  Device: PChar;                            { Current device }
  DevicesSize: Integer;                     { Amount of bytes allocated
                                              to store 'devices' }
  Driver,                                   { Name of the driver for the
                                              device }
  Port: PChar;                              { Name of the port for the
                                              device }
  DriverLine: array[0..pdStrWidth] of Char; { Device line from WIN.INI }
  LineCur: PChar;                           { FetchStr pointer into
                                              DriverLine }
  DriverStr: array[0..pdStrWidth] of Char;  { Text being built for display }
  StrCur: PChar;                            { Temp pointer used for copying
                                              Port into the line }
  StrCurSize: Integer;                      { Room left in DriverStr to
                                              copy Port }
  DevRec: PDeviceRec;                       { Record pointer built to
                                              store in DeviceCollection }

  procedure FormDriverStr(DriverStr: PChar; MaxLen: Integer;
    Device, Port: PChar);
  begin
    StrLCopy(DriverStr, Device, MaxLen);
    LoadString(hInstance, sr_On, @DriverStr[StrLen(DriverStr)],
      MaxLen - StrLen(DriverStr) - 1);
    StrLCat(DriverStr, Port, MaxLen);
  end;

begin
  TDialog.Init(AParent, TemplateName);
  tmp := New(PComboBox, InitResource(@Self, id_Combo, pdStrWidth));
  GetMem(TransferBuffer, SizeOf(PCollection) * 2 + pdStrWidth);
  PTransferRec(TransferBuffer)^.Strings := New(PStrCollection,
    Init(5, 5));
  Printer := APrinter;
  DeviceCollection := New(PDeviceCollection, Init(5, 5));

  if MaxAvail div 2 > 4096 then DevicesSize := 4096
  else DevicesSize := MaxAvail div 2;
  GetMem(Devices, DevicesSize);

  { Save initial values of printer for Cancel }
  OldDevice := StrNew(Printer^.Device);
  OldDriver := StrNew(Printer^.Driver);
  OldPort := StrNew(Printer^.Port);

  with PTransferRec(TransferBuffer)^ do
  begin
    { Get a list of devices from WIN.INI.  Stored in the form of
      <device 1>#0<device 2>#0...<driver n>#0#0
    }
    GetProfileString('devices', nil, '', Devices, DevicesSize);

    Device := Devices;
    while Device^ <> #0 do
    begin
      GetProfileString('devices', Device, '', DriverLine,
        SizeOf(DriverLine) - 1);

      FormDriverStr(DriverStr, SizeOf(DriverStr) - 1,Device, '');

      { Get driver portion of DeviceLine }
      LineCur := DriverLine;
      Driver := FetchStr(LineCur);

      { Copy the port information from the line }
      (*   This code is complicated because the device line is of
          the form:
           <device name> = <driver name> , <port> { , <port> }
          where port (in {}) can be repeated. *)

      StrCur := @DriverStr[StrLen(DriverStr)];
      StrCurSize := SizeOf(DriverStr) - StrLen(DriverStr) - 1;
      Port := FetchStr(LineCur);
      while Port^ <> #0 do
      begin
        StrLCopy(StrCur, Port, StrCurSize);
        Strings^.Insert(StrNew(DriverStr));
        New(DevRec);
        DevRec^.Device := StrNew(Device);
        DevRec^.Driver := StrNew(Driver);
        DevRec^.Port := StrNew(Port);
        DeviceCollection^.AtInsert(Strings^.IndexOf(@DriverStr), DevRec);
        Port := FetchStr(LineCur);
      end;
      Inc(Device, StrLen(Device) + 1);
    end;
    FreeMem(Devices, DevicesSize);

    { Set the current selection to Printer's current device }
    FormDriverStr(Selected, pdStrWidth, Printer^.Device, Printer^.Port);
  end;
end;

destructor TPrinterSetupDlg.Done;
begin
  StrDispose(OldDevice);
  StrDispose(OldDriver);
  StrDispose(OldPort);
  Dispose(DeviceCollection, Done);
  Dispose(PTransferRec(TransferBuffer)^.Strings, Done);
  FreeMem(TransferBuffer, SizeOf(PCollection) + pdStrWidth);
  TDialog.Done;
end;

procedure TPrinterSetupDlg.TransferData(TransferFlag: Word);
var
  DevRec: PDeviceRec;
begin
  TDialog.TransferData(TransferFlag);
  if TransferFlag = tf_GetData then
    with PTransferRec(TransferBuffer)^ do
      { Use the current selection to set Printer }
      with PDeviceRec(DeviceCollection^.At(Strings^.IndexOf(@Selected)))^ do
        { Set the printer to the new device }
        Printer^.SetDevice(Device, Driver, Port);
end;

procedure TPrinterSetupDlg.IDSetup(var Msg: TMessage);
begin
  TransferData(tf_GetData);
  Printer^.Configure(@Self);
end;

procedure TPrinterSetupDlg.Cancel(var Msg: TMessage);
begin
  TDialog.Cancel(Msg);
  { Restore old settings, just in case the user pressed the Setup button }
  if OldDriver = nil then Printer^.ClearDevice
  else Printer^.SetDevice(OldDevice, OldDriver, OldPort);
end;

end.
