(****************************************************************************

  SAFETY.PAS -- A unit to detect .EXE file tampering by inexperienced
                hackers or viren.

  ------>  Version 1.2 -- 10/09/1992 -- Last Update: 10/09/1992

  Author: John Joseph Schultz  (2:2403/32.2 @ FidoNet)

  Copyright (c) 1991-1992, John Joseph Schultz. Released to the Public Domain.

                          ͻ
     Ķ  !!! W A R N I N G !!!  Ŀ
                         ͼ                     
        Any programs using this unit MUST be compiled to diskette--not   
        memory.  Compiling into memory will lock up TURBO.EXE and it     
        will not run any longer; it will have to be restored from its    
        distribution diskette to run again.  SAFETY should only be used  
        with programs which are in their final form, and are being       
        compiled to an EXE file for use and/or distribution.             
     

        This unit saves the size, time, execution point, the first eight bytes
        of executed code--Addr (execution point)--and, optionally, checksum
        of an .EXE file in constants within the .EXE file itself.  These
        values are written to the .EXE file during the first execution.
        Successive executions of the program simply check whether the file's
        size, time, first eight bytes of executed code and, optionally,
        checksum still match their original values.  If any one is different,
        Safety displays one of the following messages and terminates the
        program:

            ERROR in .EXE file execution point -- check for virus infection.
            ERROR in .EXE file execution code -- check for virus infection.
            ERROR in .EXE file time stamp -- cannot continue
            ERROR in .EXE file size -- cannot continue

... and, if checksum checking is enabled:
            ERROR in .EXE file checksum -- cannot continue

        these messages are not immediately apparent in the .EXE file;
        they reside in the "stamped" .EXE file in an encrypted form.

        If the values have remained unchanged, Safety allows the program
        to execute.  The immediate termination of the program may indicate
        that the program has been infected by a virus, but further
        examination of the .EXE file will be necessary to confirm this.

        Because the program's execution point is the primary point which a
        virus modifies, this is the only error message which mentions
        possible virus infection.  All other messages simply indicate that
        the program cannot continue.

*********
   If checksum verification of the file's contents is not desired ...
*********

        ..it may be turned off by defining a compiler directive (NOCHKSUM).
        This will check only the file's size, time, execution point, and
        first eight bytes of execution code.

        Checksum verification is usually disabled for one or more of the
        follwing reasons:

          o  The .EXE file's contents are modified by the program itself,
             which would invalidate any checksum value stored by this unit.

          o  The .EXE file is large, and may take excessive time to generate
             a checksum for the file.

        HOWEVER, there is one reason NOT to disable checksum verification
        (especially if the program using SAFETY is commercial):

          o  The .EXE file contains an embedded copyright notice.


###### Notes on using SAFETY with self-modifying .EXE files ################

        Because SAFETY includes the file's time stamp as part of its check,
        the original time stamp should be saved in a longint and restored
        before the .EXE file is closed.  This will ensure that the time
        stamp does not change and, therefore, prevent the program from
        running.

        Example:

        var
          TStamp : longint;
          F : file;

        begin
          assign (F, Paramstr (0));
          GetFTime (F, TStamp);
          rewrite (F, 1);
          .
          .
          .
          SetFTime (F, TStamp);
          close (F);
        end;


###### Compiler Notes ######################################################

        Safety requires only that the program's stack space be at least 3K.

        Technical notes on this unit's operation are contained at the end of
        this file.

        NOTE:  Use of an EXE file compression program (like PKLITE) will 
               nulify SAFETY.  Because a file compression program creates a
               new EXE file with different parameters, the values in SAFETY
               can never match the file's parameters.  Bottom line: it's
               either SAFETY or an EXE compressor--not both.

###### EXE file size increase ##############################################

        Using Safety increases the EXE file's size _up to_ the following
        byte counts:

                               Debug Code       No Debug Code
                               ----------       -------------
             without checksum     5088              4544
             with checksum        5658              4976

        These figures were derived by compiling the following blank program
        file with Version 5.5:

        begin
        end.

        The difference of the EXE file's size with and without the clause:

        uses safety;

        ..was then measured to derive the figures.  Programs which use the
        dos unit will--more than likely--increase by a smaller amount than
        that listed above.


#########  CREDITS   #######################################################

     Trevor Carlsen (3:690/644 @ Fido) for the .EXE file modification
           technique  (from Fido's Echo: PASCAL.GER)

     Konrad Knauer (2:2403/36.2 @ Fido) for his critical eye and suggestions
           during this unit's development.


##### HAPPY LAWYER DEPT. ###################################################

     As usual, the author (me) provides no warranties, written or implied,
     for the software, nor assumes any liability for any implied
     damages--inconsequential or consequential--resulting from this use of
     this software.  The author, furthermore, provides no guarantee that the
     software, in its released form, is in any way suitable for a particular
     task or purpose, nor that it will provide ABSOLUTE protection against
     unwanted tampering.


######### DEDICATION #######################################################

     This code is dedicated to the public domain in the hope that it will
     provide an effective method of: 1) deterring tampering and/or plagerism
     of programs that you write, and 2) providing an effective method of
     detecting and alerting users of possible virus infection in the .EXE
     file.

     I encourage you to make modifications to the checksum routines, code
     placement, method of verification, etc., to customize the program to
     your needs--and to prevent standardization of this unit.

     This software may be freely used in commercial programs, with no
     royalties to the author.

Version History:
  1.2 -- 10/09/92 -- corrects a problem which did not allow the EXE file to
                     have a read-only atribute.
  1.1 -- 06/10/91 -- lists all errors found, as well as full path of EXE file
                     that is corrupted.
  1.0 -- 03/17/91 -- first version

****************************************************************************)

unit Safety;

{$IFNDEF DEBUG}
   {$A+,B-,D-,E-,F-,I+,L-,N-,O-,R-,S+,V-}
{$ELSE}
   {$A+,B-,D+,E-,F-,I+,L+,N-,O-,R+,S+,V+}
{$ENDIF}

interface

{$IFDEF BETA}
function Decode (S : string) : string;
function Encode (S : string) : string;
{$ELSE}
   (* no public functions, procedures, or variables are in this unit *)
{$ENDIF}

implementation

uses dos;

type
  PtrRec = record
             ofs, seg : word;
           end;
  messagetype = (initial, notexec, exec_p);
  {$IFNDEF NOCHKSUM}
    tmessagetype = (time, size, checksum, exec_code, exec_point);
  {$ELSE}
    tmessagetype = (time, size, exec_code, exec_point);
  {$ENDIF}

const
  CryptKeyStart = $FF;

  Message : array [initial..exec_p] of string [29] = (
            'ERROR in .EXE file ',
            ' -- cannot continue',
            ' -- check for virus infection');

  {$IFNDEF NOCHKSUM}
  TMessage : array [time..exec_point] of string [15] = (
            'time stamp', 'size', 'checksum',
            'execution code', 'execution point');
  {$ELSE}
  TMessage : array [time..exec_point] of string [15] = (
            'time stamp', 'size',
            'execution code', 'execution point');
  {$ENDIF}

  P_ExecPoint : longint = 0;
  P_ExecCode : array [1..8] of byte = (0,0,0,0,0,0,0,0);
  P_FileTime : longint = 0;
  P_FileSize : longint = 0;
{$IFNDEF NOCHKSUM}
  P_FileCHKSUM : longint = 0;
{$ENDIF}

var
  CryptKey : byte;
  ProgFile : file;
  HdrSize : word;
  F_ExecCode, T_ExecCode : array [1..8] of byte;
  F_ExecPoint, T_ExecPoint,
  F_FileTime, T_FileTime,
  F_FileSize, T_FileSize : longint;

{$IFNDEF NOCHKSUM}
  F_FileCHKSUM, T_FileCHKSUM : longint;
  checksum_reg : longint;
{$ENDIF}

  Attributes : word;

function EncryptChar (ch : char) : char;
begin
  EncryptChar := chr (ord (ch) xor CryptKey);
  CryptKey := ((CryptKey shl 4) and $FF) xor ord (ch);
  if CryptKey = 0 then CryptKey := CryptKeyStart;
end;

function DecryptChar (ch : char) : char;

var
  chx : char;

begin
  chx := chr (ord (ch) xor CryptKey);
  DecryptChar := chx;
  CryptKey := ((CryptKey shl 4) and $FF) xor ord (chx);
  if CryptKey = 0 then CryptKey := CryptKeyStart;
end;

function Decode (S : string) : string;

var
  index : byte;

begin
  Decode := S;
  CryptKey := CryptKeyStart;
  for index := 1 to length (S) do
    Decode [index] := DecryptChar (S [index]);
end;

function Encode (S : string) : string;

var
  index : byte;

begin
  Encode := S;
  CryptKey := CryptKeyStart;
  for index := 1 to length (S) do
    Encode [index] := EncryptChar (S [index]);
end;

{$IFNDEF NOCHKSUM}

procedure CalculateFilechecksum;

const
  buffer_size = $400;    { Can be changed to accomodate larger buffers }
                         { Be sure to adjust stack size accordingly    }
var
  skip_point, current_loc : longint;
  buffer : array [0..buffer_size - 1] of byte;
  read_size, pos_point : word;
  skipping : byte;

begin
  seek (ProgFile, 0);
  skip_point := longint (HdrSize + Seg (P_FileCHKSUM) - PrefixSeg) shl 4 +
                      Ofs (P_FileCHKSUM) - 256;
  skipping := 0;
  checksum_reg := 0;
  current_loc := 0;
  repeat
    BlockRead (ProgFile, buffer, buffer_size, read_size);
    pos_point := 0;
    while read_size <> pos_point do
      begin
        if current_loc + pos_point = skip_point then
          skipping := 4;
        if skipping > 0 then
          dec (skipping)
        else
          begin
            inc (checksum_reg, buffer [pos_point]);
            checksum_reg := checksum_reg xor $EDB88320;
          end;
        inc (pos_point);
      end;
    inc (current_loc, read_size);
  until eof (ProgFile);
  F_FileCHKSUM := checksum_reg;

end;
{$ENDIF}

procedure DisplayErrorType (errortype : tmessagetype);

begin
  write (Decode (message [initial]));
  write (Decode (TMessage [errortype]));

  case errortype of
      exec_point,
      exec_code : writeln (Decode (message [exec_p]));
    else
      writeln (Decode (message [notexec]));
    end;

end;

procedure SetProgFilePointer (var data);

begin
  Seek (ProgFile,
    longint (HdrSize + Seg (data) - PrefixSeg) shl 4 + Ofs (data) - 256);
end;

procedure StampFile;

var
  mcount : messagetype;
  tcount : tmessagetype;
  EncodedString : string;

begin
  { Get Time First so that it can be reset before closing the file }
  GetFTime (ProgFile, F_FileTime);

  F_FileSize := FileSize (ProgFile);

  Seek (ProgFile, $14);  { Execution Point stored in .EXE Header }
  BlockRead (ProgFile, F_ExecPoint, SizeOf (F_ExecPoint));

  Seek (ProgFile, longint (HdrSize) shl 4 +
                  longint (PtrRec (F_ExecPoint).seg) shl 4 +
                  PtrRec (F_ExecPoint).ofs);
  BlockRead (ProgFile, F_ExecCode, SizeOf (F_ExecCode));

  { first encode error messages }
  for mcount := initial to exec_p do
    begin
      SetProgFilePointer (Message [mcount]);
      EncodedString := Encode (Message [mcount]);
      BlockWrite (ProgFile, EncodedString, SizeOf (Message [mcount]));
    end;

  for tcount := time to exec_point do
    begin
      SetProgFilePointer (TMessage [tcount]);
      EncodedString := Encode (TMessage [tcount]);
      BlockWrite (ProgFile, EncodedString, SizeOf (TMessage [tcount]));
    end;

  { next write all values to P_... except CHKSUM (that goes last) }

  SetProgFilePointer (P_FileTime);
  BlockWrite (ProgFile, F_FileTime, SizeOf (F_FileTime));

  SetProgFilePointer (P_FileSize);
  BlockWrite (ProgFile, F_FileSize, SizeOf (F_FileSize));

  SetProgFilePointer (P_ExecPoint);
  BlockWrite (ProgFile, F_ExecPoint, SizeOf (F_ExecPoint));

  SetProgFilePointer (P_ExecCode);
  BlockWrite (ProgFile, F_ExecCode, SizeOf (F_ExecCode));

  { ...lastly, if selected, write the file's checksum }

{$IFNDEF NOCHKSUM}
   CalculateFilechecksum;
   SetProgFilePointer (P_FileCHKSUM);
   BlockWrite (ProgFile, F_FileCHKSUM, SizeOf (F_FileCHKSUM));
{$ENDIF}

  SetFTime (ProgFile, F_FileTime);
end;

function OriginalStampedFormat : boolean;

var
  errorfound, stop_checking : boolean;
  count : byte;

begin
  errorfound := false;

  GetFTime (ProgFile, F_FileTime);
  F_FileSize := FileSize (ProgFile);

  SetProgFilePointer (P_ExecPoint);
  BlockRead (ProgFile, T_ExecPoint, SizeOf (T_ExecPoint));
  Seek (ProgFile, $14);  { Execution Point stored in .EXE Header }
  BlockRead (ProgFile, F_ExecPoint, SizeOf (F_ExecPoint));
  if F_ExecPoint <> T_ExecPoint then
    begin
      errorfound := true;
      DisplayErrorType (exec_point);
    end;

  SetProgFilePointer (P_ExecCode);
  BlockRead (ProgFile, T_ExecCode, SizeOf (T_ExecCode));
  Seek (ProgFile, longint (HdrSize) shl 4 +
                  longint (PtrRec (F_ExecPoint).seg) shl 4 +
                  PtrRec (F_ExecPoint).ofs);
  BlockRead (ProgFile, F_ExecCode, SizeOf (F_ExecCode));
  count := 1;
  stop_checking := false;
  while (count < 9) and not stop_checking do
    begin
      if F_ExecCode [count] <> T_ExecCode [count] then
        begin
          errorfound := true;
          stop_checking := true;
          DisplayErrorType (exec_code);
        end;
      inc (count);
    end;

  SetProgFilePointer (P_FileSize);
  BlockRead (ProgFile, T_FileSize, SizeOf (T_FileSize));
  if F_FileSize <> T_FileSize then
    begin
      errorfound := true;
      DisplayErrorType (size);
    end;

  SetProgFilePointer (P_FileTime);
  BlockRead (ProgFile, T_FileTime, SizeOf (T_FileTime));
  if F_FileTime <> T_FileTime then
    begin
      errorfound := true;
      DisplayErrorType (time);
    end;

{$IFNDEF NOCHKSUM}
  CalculateFilechecksum;
  if F_FileCHKSUM <> P_FileCHKSUM then
    begin
      errorfound := true;
      DisplayErrorType (checksum);
    end;
{$ENDIF}

  OriginalStampedFormat := not errorfound;

end;

(* MAIN CODE ***********************************************************)

begin
  assign (ProgFile, Paramstr (0));
  GetFAttr (ProgFile, Attributes);
  SetFAttr (ProgFile, Attributes and not ReadOnly);
  reset (ProgFile, 1);
  seek (ProgFile, 8);
  BlockRead (ProgFile, HdrSize, SizeOf (HdrSize));

  if P_ExecPoint = 0 then
    StampFile
  else
    if not OriginalStampedFormat then
      begin
        writeln;
        writeln (Paramstr (0), ' is corrupted.');
        {$I-}
        close (ProgFile);
        {$I+}
        halt (255);
      end;

  close (ProgFile);
  SetFAttr (ProgFile, Attributes);
end.
 ---------------------------------------------------------------------------

              *************************************************
             ***                                             ***
          *****     Technical Notes on SAFETY's operation     *****
             ***                                             ***
              *************************************************


      There are three sets of variables used to store, retrieve, and compare
      values stored in the .EXE file.  They are:

          P_File...      Constants which mark the location in the .EXE file of
                         the stamped values.  They also contain the values in
                         memory, but the values in memory are ignored.

          T_File...      Memory variables which contain the stamped values read
                         from the P_File... locations in the .EXE file--not
                         from memory.

          F_File...      Memory variables which contain the same data type as
                         the T_File..., but the values are actually calculated
                         from the current .EXE file contents.

            Comparisons are then made between T_File... and F_File...

       When determining if a file is in its original stamped format, the
       following order of evaluation is used.

          Priority 1:  Execution Point
                           First place to notice virus tampering.
                   2:  Execution Code
                           I have never seen it, but it is conceivable that a
                           virus could locate the execution point, then
                           replace the first several bytes at this location
                           with a jump to its location.  Checking the first
                           eight bytes of execution code stops this.
                   3:   File Size
                           Virus code would increase the file size, but so
                           could tampering.
                   4:   File TimeStamp
                           Mostly detects tampering.
                   5:   File Checksum
                           Takes time to execute (proportional to EXE file's
                           size), but ensures that no tampering has occured).
                           TimeStamp catches most tampering efforts, but
                           the checksum ensures full, original file integrity.

      Checksum calculation excludes the 4-bytes of the actual checksum
      itself.  I would like to have included the checksum in the actual
      checksum calculation, but I don't spend my life buried in math books.

      When stamping a file, the P_File variables and messages are stamped as
      follows:

                   1:   Encode all error message string constants in the
                        EXE file.

                   2:   Write Time, Size, Execution Point, and Execution Code
                        to the P_File... constant locations in the file.

                   3:   Finally, calculate the file's CRC and write its value
                        to its P_File... constant in the file.

      As a last action, the EXE file's time stamp is restored to the value in
      F_FileTime.  If this were not done, DOS would change the file's time
      stamp to the current system time since the file's contents were
      modified; thus invalidating the stamped time.

      *****************************************************************
