{*******************************************************
              FileFind Unit

              Paul Warren
     HomeGrown Software Development
   (c) 1992 Langley British Columbia.

This unit creates a file finder dialog for inclusion in
Turbo Vision programs. This dialog will search your disk
starting from either the current directory on down the
tree or from a directory indicated in the file mask
input line. The dialog will locate all file matching
the mask it then lists them, displays the number of files
found and will print them. The file mask can include
standard DOS wildcards such as * and ?.
  
Remember, this is shareware. If you use this dialog in
your programs or you use the code in your own programs
you must register. To register send $5 to

              Paul Warren
  #232 5400 206th St. Langley BC. Canada.

The FileObj object, DCommon procedures and the Diag code
used in this dialog and supplied on this disk are from
Tom Swan's Duplicat program. The code mentioned is un-
changed because it does the job I wanted perfectly. The
DInfo object is supplied but is not used in the FileFind
dialog. You can easily insert it if you want the additional
file information available.
********************************************************}

unit FileFind;

{$F+,O+,V-,X+,D-}
interface

uses Dos, Printer, Objects, Drivers, Dialogs, Views, App, DCommon,
             DFileObj, DInfo, Diag;

type
  { The display of record numbers object }
  PRecDisplay = ^TRecDisplay;
  TRecDisplay = object(TView)
    FileNames: PFileCollection;
    Number: string[15];
    constructor Init(var Bounds: TRect);
    procedure Draw; virtual;
    function GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

  { The list of files matching the mask object }
  PRecView = ^TRecView;
  TRecView = object(TListBox)
    FileNames: PFileCollection;
    CurrentDir: PathStr;
    constructor Init(var Bounds: TRect; ANumCols: word;
        AScrollBar: PScrollBar);
    destructor Done; virtual;
    function FileObjStr(P: PFileObj; MaxLen: integer): string;
    function GetText(Item: integer; MaxLen: integer): string; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure PrintFileNames;
    procedure ReadFileNames(Mask: PathStr);
    procedure SelectItem(Item: integer); virtual;
    procedure ShowFileInfo(Item: integer);
  end;

  { The dialog object }
  PFileFindDialog = ^TFileFindDialog;
  TFileFindDialog = object(TDialog)
    FileName: PInputLine;
    constructor Init(ATitle: PathStr; AWildCard: PathStr);
    procedure GetMask(var S: PathStr);
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

const
  { Command constants used in this unit }
  cmFileGetInfo      = 103;
  cmFilePrint        = 104;
  cmChangeTitle      = 107;
  cmFileList         = 1106;
  cmUpdateRec        = 1108;

  { The object registration numbers. Note this
  object can be stored on streams!! }
  RRecDisplay: TStreamRec = (
     ObjType: 5564;
     VmtLink: Ofs(TypeOf(TRecDisplay)^);
     Load:    @TRecDisplay.Load;
     Store:   @TRecDisplay.Store
  );
  RRecView: TStreamRec = (
     ObjType: 5565;
     VmtLink: Ofs(TypeOf(TRecView)^);
     Load:    @TRecView.Load;
     Store:   @TRecView.Store
  );
  RFileFindDialog: TStreamRec = (
     ObjType: 5566;
     VmtLink: Ofs(TypeOf(TFileFindDialog)^);
     Load:    @TFileFindDialog.Load;
     Store:   @TFileFindDialog.Store
  );

  { The registration procedure declaration }
procedure RegisterFileFind;

implementation

uses Memory;

const
  { constants for the collection initialization }
  scReserve = 500;
  scDelta = 100;

var
  { global variable for transfering the number of
  records found from the RecView object to the
  RecDisplay object for display. }
  RecordNum: String;

{ TRecDisplay }
constructor TRecDisplay.Init(var Bounds: TRect);
begin
  TView.Init(Bounds);
  EventMask := EventMask or evBroadcast;
end;

procedure TRecDisplay.Draw;
var
  Color: Byte;
  I: Integer;
  B: TDrawBuffer;
begin
  Color := GetColor(1);
  I := Size.X - Length(Number) - 2;
  MoveChar(B, ' ', Color, Size.X);
  MoveStr(B[I + 1], Number, Color);
  WriteBuf(0, 0, Size.X, 1, B);
end;

function TRecDisplay.GetPalette: PPalette;
const
  P: string[1] = #19;
begin
  GetPalette := @P;
end;

procedure TRecDisplay.HandleEvent(var Event: TEvent);
var
  S: String;
begin
  TView.HandleEvent(Event);
  if Event.What = evBroadCast then
  begin
    case Event.Command of
    cmUpdateRec:
      begin
        Number := RecordNum;
        DrawView;
      end;
    end;
  end;
end;

{ TRecView }
constructor TRecView.Init(var Bounds: TRect; ANumCols: word;
                 AScrollBar: PScrollBar);
begin
  TListBox.Init(Bounds, ANumCols, AScrollBar);
  GrowMode := gfGrowHiX + gfGrowHiY;
  EventMask := EventMask or evBroadcast;
end;

destructor TRecView.Done;
begin
  if FileNames <> nil then Dispose(FileNames, Done);
  TListBox.Done;
end;

function TRecView.FileObjStr(P: PFileObj; MaxLen: integer): string;
var
  S: string;
begin
  with P^ do
  begin
    S := FName^;
    while Length(S) < 13 do
    S := S + ' ';
    S := S + DateString(FDate) + ' ' + P^.FPath^;
  end;
  FileObjStr := S;
end;

function TRecView.GetText(Item: integer; MaxLen: integer): string;
var
  P: PFileObj;
begin
  P := FileNames^.At(Item);
  GetText := FileObjStr(P, MaxLen);
end;

procedure TRecView.HandleEvent(var Event: TEvent);
begin
  TListBox.HandleEvent(Event);
  if Event.What = evBroadcast then
  begin
    case Event.Command of
      cmFilePrint: PrintFileNames;
      cmFileList: ReadFileNames(PFileFindDialog(Owner)^.FileName^.Data^);
      cmFileGetInfo: ShowFileInfo(Focused);
    else
      Exit;
    end;
    ClearEvent(Event);
  end;
end;

procedure TRecView.PrintFileNames;

  procedure PrintOneFile(P: PFileObj); far;
  begin
    Writeln(Lst, FileObjStr(P, 80));
  end;

begin
  if Yes('Print file names?') then
  begin
    FileNames^.ForEach(@PrintOneFile);
    Write(Lst, Chr(12));
  end;
end;

procedure TRecView.ReadFileNames(Mask: PathStr);
var
  MD: DirStr;
  MN: NameStr;
  ME: ExtStr;
  WildCard: String;
  S: String;

function LTrim(S: String): String;
var
  I: Integer;
begin
  I := 1;
  while (I < Length(S)) and (S[I] = ' ') do Inc(I);
  LTrim := Copy(S, I, 255);
end;

function RTrim(S: String): String;
var
  I: Integer;
begin
  while S[Length(S)] = ' ' do Dec(S[0]);
  RTrim := S;
end;

function RelativePath(var S: PathStr): Boolean;
var
  I,J: Integer;
  P: PathStr;
begin
  S := LTrim(RTrim(S));
  if (S <> '') and ((S[1] = '\') or (S[2] = ':')) then RelativePath := False
  else RelativePath := True;
end;

procedure GetMasks;
var
  Sr: SearchRec;
  LocalDir: PathStr;
begin
  LocalDir := FExpand('.');
  if LocalDir[Length(LocalDir)] <> '\' then
    LocalDir := LocalDir + '\';
  FindFirst(WildCard, AnyFile, Sr);
  while DosError = 0 do with Sr do
  begin
    FileNames^.Insert(New(PFileObj, Init(Sr, @LocalDir)));
    FindNext(Sr);
  end;
end;

procedure ReadDirectory;
var
  Sr: SearchRec;
  IsFileEntry: boolean;
begin
  GetMasks;
  FindFirst('*.*', Directory, Sr);
  while DosError = 0 do with Sr do
  begin
    IsFileEntry := Name[1] <> '.';
    if IsFileEntry and (Attr and Directory <> 0) then
    begin
      ChDir(Name);
      ReadDirectory;
      ChDir('..');
    end;
    FindNext(Sr);
  end;
end;

begin
  NewList(nil);
  FileNames := New(PFileCollection, Init(scReserve, scDelta));
  CurrentDir := FExpand('.');
  FSplit(Mask, MD, MN, ME);
  WildCard := MN + ME;
  if not RelativePath(Mask) then
  begin
    if (MD[Length(MD)] = '\') and (MD[Length(MD)-1] = ':') then
    {I+} ChDir(MD) {I-}
    else begin
      dec(MD[0]);
      {I+} ChDir(MD); {I-}
    end;
  end;
  Message(DeskTop, evBroadCast, cmChangeTitle, @Self);
  ReadDirectory;
  Str(FileNames^.Count, S);
  RecordNum := S;
  ChDir(CurrentDir);
  NewList(FileNames);
  Message(DeskTop, evBroadCast, cmUpdateRec, @Self);
end;

{ This code will implement the DInfo object if desired.
Just declare a new command const and insert a button to
create the appropriate Event. }

procedure TRecView.SelectItem(Item: integer);
begin
  ShowFileInfo(Item);
end;

procedure TRecView.ShowFileInfo(Item: integer);
var
  R: TRect;
begin
  R.Assign(30, 2, 78, 18);
  DeskTop^.Insert(New(PInfoWin, Init(FileNames^.At(Item))));
end;

 { TFileFindDialog }

constructor TFileFindDialog.Init(ATitle: PathStr; AWildCard: PathStr);
VAR
  R: TRect;
  SB: PScrollBar;
BEGIN
  R.Assign(3, 2, 65, 17);
  TDialog.Init(R, ATitle);
  BEGIN
    Options := Options  OR  ofCentered;    {  Center  dialog window  }

    { Input boxes }
    R.Assign( 15,  2,  59,  3);
    FileName := New(PInputLine, Init(R, 44));
    FileName^.Data^ := AWildCard;
    Insert(FileName);
    R.Assign(2,  2,  14,  3);
    Insert(New(PLabel, Init(R, '~F~ile Mask:', FileName)));

    {RecView}
    R.Assign(59, 4, 60, 11);
    SB := New(PScrollBar, Init(R));
    Insert(SB);

    R.Assign(2, 4, 59, 11);
    Insert(New(PRecView, Init(R, 1, SB)));

    { Buttons }
    R.Assign(3, 12, 13, 14);
    Insert(New(PButton, Init(R, '~L~ist', cmFileList, bfDefault + bfBroadcast)));
    R.Assign(16, 12, 26, 14);
    Insert(New(PButton, Init(R, '~P~rint', cmFilePrint, bfNormal + bfBroadcast)));
    R.Assign(29, 12, 39, 14);
    Insert(New(PButton, Init(R, '~C~ancel', cmCancel, bfNormal)));

    R.Assign(54, 12, 59, 13);
    Insert(New(PRecDisplay, Init(R)));
    R.Assign(43,  12,  53,  13);
    Insert(New(PStaticText, Init(R, '# Records:')));

   SelectNext(False);
  end;
end;               

procedure TFileFindDialog.GetMask(var S: PathStr);
begin
  S := FileName^.Data^;
end;

procedure TFileFindDialog.HandleEvent(var Event: TEvent);

  procedure ChangeTitle;
  begin
    if Title <> nil then DisposeStr(Title);
    Title := NewStr(FExpand('.'));
    Frame^.Draw;
  end;

begin
  TDialog.HandleEvent(Event);
  if (Event.What = evBroadcast) and (Event.Command = cmChangeTitle) then
  begin
    ChangeTitle;
    ClearEvent(Event);
  end;
end;

{ Registration procedure code }
procedure RegisterFileFind;
begin
  RegisterType(RRecDisplay);
  RegisterType(RRecView);
  RegisterType(RFileFindDialog);
end;

end.




