UNIT HandleINIFile;
{$X+}

(* ----------------------------------------------------------------------
   Part of 4DESC - A Simple 4DOS File Description Editor
       and 4FF   - 4DOS File Finder

   (c) 1992, 1993 Copyright by

       David Frey,         & Tom Bowden
       Urdorferstrasse 30    1575 Canberra Drive
       8952 Schlieren ZH     Stone Mountain, GA 30088-3629
       Switzerland           USA

       Code created using Turbo Pascal 7.0 (c) Borland International 1990

   DISCLAIMER: This unit is freeware: you are allowed to use, copy
               and change it free of charge, but you may not sell or hire
               this part of 4DESC. The copyright remains in our hands.

               If you make any (considerable) changes to the source code,
               please let us know. (send a copy or a listing).
               We would like to see what you have done.

               We, David Frey and Tom Bowden, the authors, provide absolutely
               no warranty of any kind. The user of this software takes the
               entire risk of damages, failures, data losses or other
               incidents.

   This unit handles the reading of settings stored in 4UTILS.INI. It uses
   a TStringCollection to hold the various .INI-file settings.
   ----------------------------------------------------------------------- *)

INTERFACE USES Objects;

VAR INIFileExists : BOOLEAN;

TYPE  PINIStrings   = ^TINIStrings;
      TINIStrings   = OBJECT(TCollection)
                       CONSTRUCTOR Init;
                       PROCEDURE FreeItem(Item: POINTER); VIRTUAL;
                      END;

VAR   INIStrings : PINIStrings;

FUNCTION  ReadSettingsChar(Section,Name: STRING; Default: CHAR): CHAR;
FUNCTION  ReadSettingsString(Section,Name: STRING; Default: STRING): STRING;
FUNCTION  ReadSettingsInt(Section,Name: STRING; Default: INTEGER): INTEGER;
FUNCTION  ReadSettingsColor(Section,Name: STRING; Default: INTEGER): BYTE;

IMPLEMENTATION USES Memory,
                    Dos, StringDateHandling;

CONST INIFileName = '4UTILS.INI';

CONSTRUCTOR TINIStrings.Init;
(* Reads the contents of the 4DOS.INI and the 4UTILS.INI files.
   A heuristical search method is used to locate the files.      *)

 VAR INIFile : TEXT;
     INIPath : DirStr;

     Name    : NameStr;
     Ext     : ExtStr;
     IORes   : INTEGER;

 PROCEDURE ReadIniFile(FilePath: PathStr);
 (* At the entry in this procedure the file is already open. *)

 VAR semicol : BYTE;
     ReadLine: STRING; (* a single line from a .INI file *)

 BEGIN
  WHILE NOT Eof(INIFile) DO
   BEGIN
    ReadLn(INIFile,ReadLine);
    StripLeadingSpaces(ReadLine);

    IF (ReadLine[1] <> ';') THEN
     BEGIN
      semicol := Pos(';',ReadLine);
      IF (semicol > 0) AND
         (DownStr(Copy(ReadLine,1,10)) <> 'delimiters') THEN
       ReadLine := Copy(ReadLine,1,semicol-1);

      IF Length(ReadLine) > 0 THEN
       BEGIN
        StripTrailingSpaces(ReadLine);

        IF MemAvail < SizeOf(ReadLine) THEN
         BEGIN
          WriteLn;
          WriteLn('OUT OF MEMORY while reading ''',FilePath,'''!');
          WriteLn('Line "',ReadLine,'"');
          WriteLn('will not be stored and its contents is lost.');
          WriteLn;
         END
        ELSE
         BEGIN
          ReadLine := DownStr(ReadLine);

          TCollection.Insert(NewStr(ReadLine));
         END;
       END;
     END; (* IF Line[1] <> ';' .. *)
   END; (* WHILE *)

  {$I-}
  Close(INIFile); IORes := IOResult;
  {$I+}
 END; (* ReadIniFile *)

 PROCEDURE SearchPath;
 (* Search for INIFile in the PATH *)

  BEGIN
   INIPath := FSearch(INIFileName,GetEnv('PATH'));
   IF INIPath > '' THEN
     BEGIN
       {$I-}
       Assign(INIFile,INIPath);
       Reset(INIFile);
       INIFileExists := (IOResult = 0);
       {$I+}
     END;
  END; (* SearchPath *)

BEGIN
 TCollection.Init(100,10);

 (* 4DOS.INI:
    i)  at the normal place: the same location where the
        4DOS.COM command interpreter lives.
    ii) in the root directory.

    P.S: the SHELL variable is not examined.                              *)

  FSplit(GetEnv('COMSPEC'),INIPath,Name,Ext);
  {$I-}
  Assign(INIFile,INIPath+'4DOS.INI'); Reset(INIFile);
  {$I+}
  INIFileExists := (IOResult = 0);

  IF INIFileExists THEN ReadIniFile(INIPath+'4DOS.INI')
  ELSE
   BEGIN
     {$I-}
     Assign(INIFile,'C:\4DOS.INI'); Reset(INIFile);
     {$I+}
     INIFileExists := (IOResult = 0);

     IF INIFileExists THEN ReadIniFile('\4DOS.INI')
   END;

 (* Search strategy for 4UTILS.INI :
    i)  Directory where this application was started from.
    ii) Environment variable called 4UTILS
    ii) in the Path                                                       *)

  FSplit(ParamStr(0),INIPath,Name,Ext);
  {$I-}
  Assign(INIFile,INIPath+INIFileName); Reset(INIFile);
  {$I+}
  INIFileExists := (IOResult = 0);

  IF NOT INIFileExists THEN
    BEGIN
      INIPath := GetEnv('4UTILS');
      IF INIPath > '' THEN
        BEGIN
          IF INIPath[Length(INIPath)] <> '\' THEN INIPath := INIPath + '\';
          {$I-}
          Assign(INIFile,INIPath+INIFileName); Reset(INIFile);
          {$I+}
          INIFileExists := (IOResult = 0);

          IF NOT INIFileExists THEN SearchPath;
        END
      ELSE
       SearchPath;
    END;

  IF INIFileExists THEN ReadIniFile(INIPath+INIFileName)
END; (* TINIStrings.Init *)

PROCEDURE TINIStrings.FreeItem(Item: POINTER);
(* Free a string by using DisposeStr. This is necessary, since we are not
   using TStringCollection (which is unfortunately sorted), so we have to
   free the strings manually                                              *)

BEGIN
 DisposeStr(Item);
END;

(* A collection of functions to read Strings/Values/Color names out of
   a initialisation file.                                              *)

FUNCTION  ReadSettingsString(Section, Name: STRING; Default: STRING): STRING;
(* An empty section means: scan the whole .INI file                    *)

VAR LineNr: BYTE;
    eq    : BYTE;
    s,res : STRING;
    sp    : PString;

BEGIN
 LineNr := 0;
 IF Section <> '' THEN Section := '['+DownStr(Section)+']';
 s := '';
 WHILE (s <> Section) AND (LineNr < INIStrings^.Count) DO
  BEGIN
    sp := INIStrings^.At(LineNr);
    IF sp <> NIL THEN s := STRING(sp^)
                 ELSE s := '';
    INC(LineNr);
  END;

 IF (s = Section) AND (LineNr < INIStrings^.Count) THEN
  BEGIN
   DownString(Name); res := ''; s := '';
   REPEAT
    sp := INIStrings^.At(LineNr);
    IF sp <> NIL THEN s := STRING(sp^)
                 ELSE s := '';
    eq := Pos('=',s);
    IF eq > 0 THEN s := Copy(s,1,eq-1);
    StripTrailingSpaces(s);
    INC(LineNr);
   UNTIL (s = Name) OR (LineNr = INIStrings^.Count);

   IF s = Name THEN
    BEGIN
     res := Copy(STRING(sp^),eq+1,255);
     StripLeadingSpaces(res);
    END
   ELSE res := Default;
  END
 ELSE res:= Default;
 ReadSettingsString := res;
END;

FUNCTION  ReadSettingsChar(Section,Name: STRING; Default: CHAR): CHAR;

VAR s : STRING;

BEGIN
 s := ReadSettingsString(Section,Name,Default);
 ReadSettingsChar := s[1];
END;

FUNCTION  ReadSettingsInt(Section,Name: STRING; Default: INTEGER): INTEGER;

VAR s  : STRING;
    res: INTEGER;
    v  : INTEGER;

BEGIN
 Str(Default,s);
 s := ReadSettingsString(Section,Name,s);
 Val(s,v,res);
 IF res > 0 THEN v := Default;
 ReadSettingsInt := v;
END;

FUNCTION  ReadSettingsColor(Section,Name: STRING; Default: INTEGER): BYTE;

CONST color : ARRAY[0..15] OF STRING[12] =
              ('black'   ,'blue'        ,'green'     ,'cyan'     ,
               'red'     ,'magenta'     ,'brown'     ,'lightgray',
               'darkgray','lightblue'   ,'lightgreen','lightcyan',
               'lightred','lightmagenta','yellow'    ,'white');

VAR s  : STRING;
    c  : BYTE;

BEGIN
 Str(Default,s);
 s := ReadSettingsString(Section,Name,'');

 IF s > '' THEN
  BEGIN
   c := 0;
   WHILE (color[c] <> s) AND (c<16) DO INC(c);
   IF color[c] <> s THEN c := Default;
  END
 ELSE c := Default;
 ReadSettingsColor := c;
END;

BEGIN
 INIFileExists := FALSE;
 INIStrings := NIL; (* never leave a Pointer uninitialized ! *)
END.
