/*
** ==================================================================
**
** NOTE TO TURBO C/C++ and BORLAND C++ USERS:
**
**  I have used the _dos_allocmem, _dos_freemem and _dos_setblock
**  functions in this routine.  These all work for MSC and TC/C++
**  EXCEPT the _dos_setblock, which does not exist in Borland
**  libraries.  You will have to replace the _dos_setblock with
**  the setblock() call, which returns a -1 on success, vs the 0
**  returned by _dos_setblock() on success.
**  You may also want to replace the _dos_allocmem and _dos_freemem
**  calls with allocmem and freemem at the same time.  The return
**  values are different for them as well.
** ==================================================================
**
** Generic memory allocation functions for C programs running under
** DOS or Falken.
**
** These functions return only FAR pointers, so they should be used
** with Large or Medium model programs.
**
** These functions replace MALLOC, CALLOC, REALLOC, and FREE
**
** How it works...
** Memory is allocated by paragraphs, meaning each request is rounded
** up to the next 16-byte boundary.  One paragraph (16 bytes) is used
** as a control record for every memory block allocated.
**
** This means that the minimum number of bytes used by ANY request is
** 32 bytes.  16 byte header, plus at least 16 bytes per allocation.
**
** There is a table of segment addresses.  At startup, the table is
** empty.  During operation, this table will contain the addresses of
** the memory segments currently in use.  Each memory segment will be
** 8K (it can go down to 2K if contiguous memory is scarce), and can
** be expanded dynamically as needed.  h_malloc calls are satisfied
** by doling out memory from these segments.
**
** Any memory request that is larger than about 2K will be handled by
** DOS - we don't use the dynamic tables for such large requests.
**
*/

#include <stdio.h>
#include <dos.h>

#define MAXSEGS 32      /* # of DOS memory segments for base addresses */

#define UNUSED      0
#define USED        1
#define TERMINATOR  2
#define DOS_MEM     3


/*
** A little trick here..
** We initialize only the first entry in the table to 0.
** When we enter the h_malloc function for the first time, we
** see that it is 0, so we set the whole rest of the table to 0.
**
** This makes the code more portable, since not all compilers
** guarantee a cleared work space, and MAXSEGS can be changed
** too easily..
*/

static unsigned int base_mem_table[MAXSEGS] = { 0 };
static unsigned int base_mem_size[MAXSEGS];

struct mem_ctrl_rec
{
    unsigned int pgf;
    unsigned int len;
    int used;
    char mflag[8];  /* a unique flag */
};

void far *makeptr(unsigned x)
{
    unsigned long j;
    j=(unsigned long)x;
    j<<=16;
    return (void far *)j;
}

unsigned int get_dos_block(unsigned int paras)
{
    unsigned a;

    if(_dos_allocmem(paras, &a)==0) return a;
    else return 0;
}

unsigned int expand_dos_block(unsigned int paras, unsigned int adr)
{
    unsigned a;

    if(_dos_setblock(paras, adr, &a)==0) return 1;  /* success */
    else return 0;
}

char far *h_malloc(unsigned int size)
{
    unsigned int j,k,n;
    struct mem_ctrl_rec *mcr, *mcr2;

    if(base_mem_table[0] == 0)
    {
        for(j=0; j<MAXSEGS; j++)
        {
            base_mem_table[j]=0;
        }
    }

    size = (size+15)/16;    /* number of paragraphs requested */

    if(size > (2048/16))  /* if > 2K bytes requested, get from DOS */
    {
        n=get_dos_block(size+1);
        if(n==0)
            return NULL;
        mcr=(struct mem_ctrl_rec *)makeptr(n);
        mcr->used = DOS_MEM;
        strcpy(mcr->mflag,"h_alloc");
        mcr->len=size;
        mcr->pgf=n;
        return makeptr(n+1);    /* return the address of the memory */
    }
    for(j=0; (j<MAXSEGS) && (base_mem_table[j]!=0); j++)
    {
        mcr=(struct mem_ctrl_rec *)makeptr(base_mem_table[j]);
/*
** Stop when we find a the block marked TERMINATOR
*/
        while(mcr->used != TERMINATOR)
        {
            if(mcr->used == UNUSED)
            {
/*
** Coalesce the contiguous free memory
*/
                for(;;)
                {
                    mcr2=makeptr(mcr->pgf + mcr->len + 1);
                    if(mcr2->used == UNUSED)
                    {
                        mcr->len += mcr2->len+1;
                        memset(mcr2,0,16);  /* clear the ctrl record */
                    }
                    else
                    {
                        break;
                    }
                }

                if(mcr->len>=size)   /* got one*/
                {
                    if(mcr->len > size)
                    {
                        mcr2 = makeptr(mcr->pgf + size + 1);
                        mcr2->used=UNUSED;
                        mcr2->pgf=(mcr->pgf+size+1);
                        mcr2->len=mcr->len-size-1;
                        strcpy(mcr2->mflag,"h_alloc");
                    }
                    mcr->len = size;
                    mcr->used = USED;
                    return makeptr(mcr->pgf + 1);
                }
            }
            mcr=makeptr(mcr->pgf+mcr->len+1);
        }
    }

    if(j)   /* if j > 0, try to expand previous block */
    {
        n=4096/16;  /* number of paras to get */
        if(expand_dos_block(n, base_mem_table[j-1]) != 0)
        {
            j--;    /* adjust for this block.. */
            base_mem_size[j]+=n;
            mcr=(struct mem_ctrl_rec *)makeptr(base_mem_table[j]);
            while(mcr->used != TERMINATOR)
            {
                mcr=makeptr(mcr->pgf+mcr->len+1);
            }
/*
** MCR now points to the last block in the segment - the marker.
*/
            mcr->used=UNUSED;
            mcr->len=n-1;
            mcr2=(struct mem_ctrl_rec *)
                makeptr(base_mem_table[j]+base_mem_size[j]-1);
            mcr2->pgf=base_mem_table[j]+base_mem_size[j]-1;
            mcr2->used=TERMINATOR;  /* end of the chain for this segment */
            mcr2->len=0;
            strcpy(mcr2->mflag,"h_alloc");
            return h_malloc(size*16);    /* now there is memory! */
        }
    }
/*
** Didn't find a free block.  couldn't expand a block.
** damn.  gotta get a new block
*/
    if(j==MAXSEGS)
        return NULL;    /* can't */

    if((base_mem_table[j]=get_dos_block(n=8192/16))==0)
        if((base_mem_table[j]=get_dos_block(n=4096/16))==0)
            if((base_mem_table[j]=get_dos_block(n=2048/16))==0)
                return NULL;

    base_mem_size[j]=n;

    mcr=(struct mem_ctrl_rec *)makeptr(base_mem_table[j]);
    mcr->pgf=base_mem_table[j];
    mcr->used=UNUSED;
    mcr->len=n-2;
    strcpy(mcr->mflag,"h_alloc");

    mcr2=(struct mem_ctrl_rec *)makeptr(base_mem_table[j]+n-1);
    mcr2->pgf=base_mem_table[j]+n-1;
    mcr2->used=TERMINATOR;
    mcr2->len=0;
    strcpy(mcr2->mflag,"h_alloc");
    return h_malloc(size*16);   /* should work, now that we have memory*/
}

/*
** Calloc is easy - allocate a segment and clear it!
*/

char far *h_calloc(unsigned int num, unsigned int size)
{
    char far *p;

    p=NULL;

    size=size*num;
    if(size)
    {
        p=h_malloc(size);
        if(p != NULL)
        {
            memset(p,0,size);
        }
    }
    return p;
}

/*
** The h_free function will not return any status, but it will make
** sevarl checks to make sure the memory being returned is on a
** h_malloc boundary.
*/

void h_free(char far *cp)
{
    struct mem_ctrl_rec far *p;
    unsigned int j;
    j=FP_OFF(cp);
    if(j)
    {
        return; /* not a valid block, since we give on para boundary */
    }
    j=FP_SEG(cp);
    p=(struct mem_ctrl_rec far *)makeptr(j-1); /* get ctrl rec address */
    if((strcmp(p->mflag,"h_alloc")==0) && (p->used == USED))
    {
        p->used=UNUSED;
    }
    else if((strcmp(p->mflag,"h_alloc")==0) && (p->used == DOS_MEM))
    {
        memset(p, 0, sizeof(struct mem_ctrl_rec));
        _dos_freemem(j-1);
    }
}

/*
** Might as well put realloc in as well.
** The simplest realloc we can get by with..
**  allocate the new region
**  copy from old region to new
**  free old region
*/

void far *h_realloc(void far *src, unsigned newsize)
{
    void far *dest;

    dest=h_malloc(newsize);
    if(dest != NULL)
    {
        memcpy(dest, src, newsize);
        h_free(src);
    }
    return dest;
}

/*
** This code put in for debugging.
** Comment it out if you like.
*/

dump_mem_ctrl()
{
    int j,k,n;
    struct mem_ctrl_rec *mcr, *mcr2;

    for(j=0; (j<MAXSEGS) && (base_mem_table[j]!=0); j++)
    {
        printf("BASE_MEM_TABLE[%d] = %04x   SIZE=%04x\n", j,
            base_mem_table[j], base_mem_size[j]);

        mcr=(struct mem_ctrl_rec *)makeptr(base_mem_table[j]);
        do
        {
            printf("ADDR=%04x   Len=%04x   Used=%d   MFLAG=%s\n",
                mcr->pgf, mcr->len, mcr->used, mcr->mflag);
            mcr=makeptr(mcr->pgf+mcr->len+1);
        } while(mcr->used != TERMINATOR);
    }
}

