/*
** This is a collection of functions for manipulating dates and times.
** The time() functions of the standard C library are good for all
** dates after 01/01/1980.  We need to reference dates prior to that
** for birthdays and such, so I have implemented a date/time scheme
** to track dates from 01/01/1900.
**
** To use these routines, you must use the structs.h template for
** the timestamp structure.
**
** The basic function is dattim().  It fills in a timestamp structure
** with the current system date and time.  The value ts_date is loaded
** with an unsigned in value identifying the current date.  The ts_time
** field is loaded with values indicating the current time.  The ts_year,
** ts_month, etc. fields are all filled in, and the ts_str char array
** contains a printable representation of the current date and time in
** the format 'mm/dd/yy hh:mm'.
**
** Individual functions are described within the body of this file.
*/

#include <stdio.h>
#include <dos.h>
#include <string.h>
#include <ctype.h>
#include "structs.h"
#include "dattim2.h"

int dc_monthsiz[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

int cur_dayofweek;

unsigned int dc_yearstart[] = {
	0, 366, 731, 1096, 1461, 1827, 2192, 2557,	/* 00-07 */
	2922, 3288, 3653, 4018, 4383, 4749, 5114, 5479,	/* 08-15 */
	5844, 6210, 6575, 6940, 7305, 7671, 8036, 8401,	/* 16-23 */
	8766, 9132, 9497, 9862, 10227, 10593, 10958, 11323,	/* 24-31 */
	11688, 12054, 12419, 12784, 13149, 13515, 13880, 14245,	/* 32-39 */
	14610, 14976, 15341, 15706, 16071, 16437, 16802, 17167,	/* 40-47 */
	17532, 17898, 18263, 18628, 18993, 19359, 19724, 20089,	/* 48-55 */
	20454, 20820, 21185, 21550, 21915, 22281, 22646, 23011,	/* 56-63 */
	23376, 23742, 24107, 24472, 24837, 25203, 25568, 25933,	/* 64-71 */
	26298, 26664, 27029, 27394, 27759, 28125, 28490, 28855,	/* 72-79 */
	29220, 29586, 29951, 30316, 30681, 31047, 31412, 31777,	/* 80-87 */
	32142, 32508, 32873, 33238, 33603, 33969, 34334, 34699,	/* 88-95 */
	35064, 35430, 35795, 36160, 36525, 36891, 37256, 37621,
	37986, 38352, 38717, 39082, 39447, 39813, 40178, 40543,
40908, 41274, 41639, 42004, 42369, 42735, 43100, 43465};


/*
** get_dattim calls the DOS functions for get_date and get_time,
** and fills 2 REGS structures with the return values.  It is used
** by the dattim() function to calculate the current date and time
** values.
*/

void get_dattim(union REGS *timeregs, union REGS *dateregs)
{
	union REGS inregs;
	inregs.h.ah = 44;
	intdos(&inregs, timeregs);
	inregs.h.ah = 42;
	intdos(&inregs, dateregs);
}

/*
** The next function will place the current system time
** in both a binary formatted internal format, and in
** string format.  The timestamp structure contains
** the template for both
**
*/

#define year t->ts_year
#define month t->ts_month
#define day t->ts_day
#define hour t->ts_hour
#define minute t->ts_minute
#define tstr t->ts_str

/*
** dattim() fills a timestamp structure with the current date and
** time.
*/

void dattim(struct timestamp *t)
{
	union REGS timeregs, dateregs;

	get_dattim(&timeregs, &dateregs);
	year = (unsigned int) dateregs.x.cx;	/* year  */
	month = (unsigned int) dateregs.h.dh;	/* month */
	day = (unsigned int) dateregs.h.dl;	/* day */
	cur_dayofweek = (unsigned int) dateregs.h.al + 1;	/* day of week */
	hour = (unsigned int) timeregs.h.ch;	/* hours */
	minute = (unsigned int) timeregs.h.cl;	/* minutes */
	datetonum(t);
	t->ts_time = (hour << 7) + minute;
	timestring(t);
}

/*
** timestring fills the ts_str portion of a timestamp structure
** with a printable character string representing the date and time.
*/

void timestring(struct timestamp *t)
{
	sprintf(tstr, "%02d/%02d/%02d %02d:%02d", month, day,
			year - 1900, hour, minute);
}

/*
***************************************************
**
** Given the bit fields in ts_date and ts_time, the
** next function calculates the integer values
** for the time fields, and creates the string.
**
***************************************************
*/

void timeexpand(struct timestamp *t)
{
	if (t->ts_date == 0)
	{
		dattim(t);
	}
	else
	{
		numtodate(t);
		minute = t->ts_time & 63;
		hour = (t->ts_time >> 7) & 31;
		timestring(t);
	}
}

/*
** filetimeexpand() converts a date and time in DOS format (such as
** the time stamp for a file), to Falken's internal format.
** To use this function, put theDOS date and time stamps in the
** ts_date and ts_time fields of a timestamp structure, and call the
** function with a pointer to the timestamp structure.  The structure
** will be filled with Falken's internal date and time format
** corresponding to the time stamp originally in the structure.
*/

void filetimeexpand(struct timestamp *t)
{
	year = ((t->ts_date >> 9) & 127) + 1980;
	month = (t->ts_date >> 5) & 15;
	day = t->ts_date & 31;
	minute = (t->ts_time >> 5) & 63;
	hour = (t->ts_time >> 11) & 31;
	timestring(t);
}

/*
** Given the ts_year, ts_month, and ts_day fields in a timestamp
** structure, calculate the ts_date value.
*/

void datetonum(struct timestamp *t)
{
  unsigned int j, k;
	k = year;
	if (k >= 1900)
		k -= 1900;
    if (k > 118)
    {
        t->ts_date=0;
        return;
    }
	if ((k & 3) == 0)
		dc_monthsiz[1] = 29;
	else
		dc_monthsiz[1] = 28;
	j = dc_yearstart[k];
  for (k = 0; k < (unsigned) (month - 1); k++)
		j += dc_monthsiz[k];
	j += day;
	t->ts_date = j;
}

/*
** Given a date in ts_date, fill in ts_year, ts_month, and ts_day
*/

void numtodate(struct timestamp *t)
{
	unsigned int j, k, l, n;
	n = t->ts_date;
	n--;
	if (n > 43800)
        return;               /* error return */
	j = 0;
	while (dc_yearstart[j + 1] <= n)
		j++;
	if ((j & 3) == 0)
		dc_monthsiz[1] = 29;
	else
		dc_monthsiz[1] = 28;
	k = n - dc_yearstart[j];
	l = 0;
  while (k >= (unsigned) dc_monthsiz[l])
	{
		k -= dc_monthsiz[l++];
	}
	day = k + 1;
	month = l + 1;
	year = j + 1900;
}

/*
** Given a string in the format mm/dd/yyyy, calculate the internal
** date value, suitable for placing into the ts_date field of a
** timestamp structure.  Returns the date value.
*/

unsigned int strtonum(char *str)
{
	unsigned j, k;
	unsigned kmon, kday, kyear;
	j = sscanf(str, "%d/%d/%d", &kmon, &kday, &kyear);
	if (j != 3)
		return 0;
	k = kyear;
	if (k >= 1900)
		k -= 1900;
	if (k > 118)
		return 0;
	if ((k & 3) == 0)
		dc_monthsiz[1] = 29;
	else
		dc_monthsiz[1] = 28;
	j = dc_yearstart[k];
	for (k = 0; k < (kmon - 1); k++)
		j += dc_monthsiz[k];
	j += kday;
	return j;
}
