(*****************************************************************************)
(*                                                                           *)
(*        filename        : XPRINTER.PAS                                     *)
(*        autor           : Stefan Boether / Compuserve Id : 100023,275      *)
(*                                                 FidoNet :  2:243/91.331   *)
(*                                  Internet: 100023.275@CompuServe.COM      *)
(*        system          : BP 7.0 / DOS 5.0                                 *)
(*        changes         :                                                  *)
(*        when    what                                                who    *)
(*---------------------------------------------------------------------------*)
(*****************************************************************************)
(*  Description :  Printer seq. access to Turbo Vision                       *)
(*****************************************************************************)
{Header-End}

UNIT XPrinter; {$D-,O+,N+}

INTERFACE

USES Objects {$ifdef dpmi},WinAPI{$endif};

TYPE dll_PrintError  = procedure ( Error, Status : Integer );

     KeyStr  = String[20];
     SeqStr  = String[128];

     {- Only for better class browsing !-}
     TPrinterObjects = object( TObject )
     end;

     {- The base object for the printer sequences and data -}
     PPrnItem = ^TPrnItem;
     TPrnItem = object( TPrinterObjects )
       Key  : PString;
       constructor Init( AKey:KeyStr );
       constructor Load( var S:TStream );
       procedure   Store( var S:TStream );
       destructor  Done; virtual;
       procedure   GetData( var Rec ); virtual;
       procedure   SetData( var Rec ); virtual;
     end;

     {- A printer sequence to put a mode-attribute on/off -}
     PPrnMode = ^TPrnMode;
     TPrnMode = object( TPrnItem )
       On  : PString;
       Off : PString;
       constructor Init( AKey:KeyStr; AOn,AOff:SeqStr );
       constructor Load( var S:TStream );
       procedure   Store( var S:TStream );
       destructor  Done; virtual;
       procedure   GetData( var Rec ); virtual;
       procedure   SetData( var Rec ); virtual;
     end;

     {- A normal printer sequence -}
     PPrnSeq  = ^TPrnSeq;
     TPrnSeq = object( TPrnItem )
       Seq : PString;
       constructor Init( AKey:KeyStr; ASeq:SeqStr );
       constructor Load( var S:TStream );
       procedure   Store( var S:TStream );
       destructor  Done; virtual;
       procedure   GetData( var Rec ); virtual;
       procedure   SetData( var Rec ); virtual;
     end;

     {- A printer sequence for font selection -}
     PPrnFont = ^TPrnFont;
     TPrnFont = object( TPrnSeq )
       Proport : PString;
       Cpi     : Real;
       constructor Init( AKey:KeyStr; ASeq,AProp:SeqStr; ACpi:Real );
       constructor Load( var S:TStream );
       procedure   Store( var S:TStream );
       destructor  Done; virtual;
       procedure   GetData( var Rec ); virtual;
       procedure   SetData( var Rec ); virtual;
     end;

     {- Some general options -}
     PPrnOptions = ^TPrnOptions;
     TPrnOptions = object( TPrnItem )
       LineValue  : Integer;
       RelTabs    : Boolean;
       Reposition : Boolean;
       constructor Init( AKey:KeyStr; ALineValue:Integer; AOption:Word);
       constructor Load( var S:TStream);
       procedure   Store( var S:TStream);
       procedure   GetData( var Rec ); virtual;
     end;

     {- A collection sorted on PPrnItem -}
     PPrnCollection = ^TPrnCollection;
     TPrnCollection = object( TSortedCollection )
       function Compare(Key1, Key2: Pointer): Integer; virtual;
       function KeyOf(Item: Pointer): Pointer; virtual;
       function FindItem(Key:KeyStr): PPrnItem;
     end;

     {- A container for all the printer codes -}
     PPrinter   = ^TPrinter;
     TPrinter   = object ( TPrinterObjects )
       Version : Word;
       Fonts   : PPrnCollection;
       Modes   : PPrnCollection;
       Feeder  : PPrnCollection;
       Chars   : PPrnCollection;
       Others  : PPrnCollection;
       Reserved: PPrnCollection;
       constructor Init;
       destructor  Done; virtual;
       constructor Load ( var S:TStream );
       procedure   Store( var S:TStream );
     end;

     PPrnDriver = ^TPrnDriver;
     TPrnDriver = object( TPrinterObjects )
       constructor Init( DllName: PChar; StreamPort: PStream );
       destructor  Done; virtual;
       procedure   FormFeed;
       procedure   StrOut( St:String );
       procedure   DataOut( Data:Pointer; Count:Word );
       procedure   EnumFonts( CallBack:Pointer );
       procedure   SwitchPort( P:PMemoryStream );
     private
      {$ifdef Dpmi}
       PrnHandle     : THandle;
       xInitPrinter  : procedure;
       xResetPrinter : procedure;
       xEnumFonts    : procedure ( CallBack:Pointer);
       xFormFeed     : procedure;
       xStrOut       : procedure (St:String);
       xDataOut      : procedure (Data:Pointer;Count:Word);
       xSwitchPort   : procedure (P:PMemoryStream );
       procedure   DllFncMap( Name:PChar;var P );
      {$endif}
     end;

CONST
     xpReset     = 'Reset';
     xpTabs      = 'Tabs';
     xpFormLen   = 'FormLen';
     xpLeftMar   = 'LeftMar';
     xpLinieDist = 'LineDist';
     xpOptions   = 'Options';
     xpFormFeed  = 'FormFeed';
     xpBold      = 'Bold';
     xpUnderline = 'Underline';
     xpItalics   = 'Italics';

CONST
     RPrinter: TStreamRec = (
       ObjType: 20050;
       VmtLink: Ofs(TypeOf(TPrinter)^);
       Load: @TPrinter.Load;
       Store: @TPrinter.Store);

     RPrnItem: TStreamRec = (
       ObjType: 20051;
       VmtLink: Ofs(TypeOf(TPrnItem)^);
       Load: @TPrnItem.Load;
       Store: @TPrnItem.Store);

     RPrnMode: TStreamRec = (
       ObjType: 20052;
       VmtLink: Ofs(TypeOf(TPrnMode)^);
       Load: @TPrnMode.Load;
       Store: @TPrnMode.Store);

     RPrnSeq: TStreamRec = (
       ObjType: 20053;
       VmtLink: Ofs(TypeOf(TPrnSeq)^);
       Load: @TPrnSeq.Load;
       Store: @TPrnSeq.Store);

     RPrnFont: TStreamRec = (
       ObjType: 20054;
       VmtLink: Ofs(TypeOf(TPrnFont)^);
       Load: @TPrnFont.Load;
       Store: @TPrnFont.Store);

     RPrnCollection: TStreamRec = (
       ObjType: 20055;
       VmtLink: Ofs(TypeOf(TPrnCollection)^);
       Load: @TPrnCollection.Load;
       Store: @TPrnCollection.Store);

     RPrnOptions: TStreamRec = (
       ObjType: 20056;
       VmtLink: Ofs(TypeOf(TPrnOptions)^);
       Load: @TPrnOptions.Load;
       Store: @TPrnOptions.Store);

procedure RegisterXPrinter;

IMPLEMENTATION

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnItem                                             *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrnItem.Init( AKey:KeyStr );
begin
  TObject.Init;
  Key := NewStr( AKey );
end;

destructor TPrnItem.Done;
begin
  IF Key <> nil then DisposeStr( Key );
  TObject.Done;
end;

constructor TPrnItem.Load( var S:TStream );
begin
  Key := S.ReadStr;
end;

procedure TPrnItem.Store( var S:TStream );
begin
  S.WriteStr(Key);
end;

procedure TPrnItem.GetData( var Rec );
begin end;

procedure TPrnItem.SetData( var Rec );
begin end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnMode                                             *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrnMode.Init( AKey:KeyStr; AOn,AOff:SeqStr );
begin
  TPrnItem.Init(AKey);
  On  := NewStr(AOn);
  Off := NewStr(AOff);
end;

destructor TPrnMode.Done;
begin
  IF On <> nil then DisposeStr( On );
  IF Off <> nil then DisposeStr( Off );
  TPrnItem.Done;
end;

constructor TPrnMode.Load( var S:TStream );
begin
  TPrnItem.Load( S );
  On := S.ReadStr;
  Off := S.ReadStr;
end;

procedure TPrnMode.Store( var S:TStream );
begin
  TPrnItem.Store( S );
  S.WriteStr(On);
  S.WriteStr(Off);
end;

procedure TPrnMode.GetData( var Rec );
  type ModeType = record
         StOn, StOff : SeqStr;
       end;
begin
  With ModeType( Rec ) Do Begin
    If On <> nil then StOn  := On^
                 else StOn  := '';
    If Off<> nil then StOff := Off^
                 else StOff := '';
  end;
end;

procedure TPrnMode.SetData( var Rec );
  type ModeType = record
         StOn, StOff : SeqStr;
       end;
begin
  With ModeType(Rec) Do Begin
    IF On <> Nil then DisposeStr( On );
    On := NewStr( StOn );
    IF Off <> Nil then DisposeStr(Off);
    Off := NewStr( StOff );
  end; (* With *)
end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnSeq                                              *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrnSeq.Init( AKey:KeyStr; ASeq:SeqStr );
begin
  TPrnItem.Init(AKey);
  Seq := NewStr(ASeq);
end;

destructor TPrnSeq.Done;
begin
  IF Seq <> nil then DisposeStr( Seq );
  TPrnItem.Done;
end;

constructor TPrnSeq.Load( var S:TStream );
begin
  TPrnItem.Load( S );
  Seq := S.ReadStr;
end;

procedure TPrnSeq.Store( var S:TStream );
begin
  TPrnItem.Store( S );
  S.WriteStr(Seq);
end;

procedure TPrnSeq.GetData( var Rec );
begin
  IF Seq <> nil then
     String(Rec) := Seq^
  else
     String(Rec) := '';
end;

procedure TPrnSeq.SetData( var Rec );
begin
  IF Seq <> Nil then DisposeStr( Seq );
  Seq := NewStr( String(Rec));
end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnFont                                             *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrnFont.Init( AKey:KeyStr; ASeq,AProp:SeqStr; ACpi:Real );
begin
  TPrnSeq.Init(AKey,ASeq);
  Proport := NewStr(AProp);
  Cpi := ACpi;
end;

destructor TPrnFont.Done;
begin
  IF Proport <> nil then DisposeStr( Proport );
  TPrnSeq.Done;
end;

constructor TPrnFont.Load( var S:TStream );
begin
  TPrnSeq.Load( S );
  Proport := S.ReadStr;
  S.Read(Cpi,Sizeof(Cpi));
end;

procedure TPrnFont.Store( var S:TStream );
begin
  TPrnSeq.Store( S );
  S.WriteStr(Proport);
  S.Write(Cpi,Sizeof(Cpi));
end;

procedure TPrnFont.GetData( var Rec );
  type FontType = record
         StSeq  : SeqStr;
         ACpi   : Real;
         StProp : SeqStr;
       end;
begin
  TPrnSeq.GetData(Rec);
  With FontType( Rec ) Do Begin
    If Proport <> nil then StProp:= Proport^
                      else StProp := '';
    ACpi := Cpi;
  end;
end;

procedure TPrnFont.SetData( var Rec );
  type FontType = record
         StSeq  : SeqStr;
         ACpi   : Real;
         StProp : SeqStr;
       end;
begin
  TPrnSeq.SetData(Rec);
  With FontType(Rec) Do Begin
    IF Proport<> Nil then DisposeStr( Proport );
    Proport := NewStr( StProp );
    Cpi := ACpi;
  end; (* With *)
end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnOptions                                          *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrnOptions.Init( AKey:KeyStr; ALineValue:Integer; AOption:Word );
begin
  TPrnItem.Init(AKey);
  LineValue := ALineValue;
  RelTabs   := (AOption and 1) <> 0;
  Reposition:= (AOption and 2) <> 0;
end;

constructor TPrnOptions.Load( var S:TStream );
begin
  TPrnItem.Load( S );
  S.Read(LineValue,Sizeof(LineValue)+Sizeof(RelTabs)+Sizeof(Reposition));
end;

procedure TPrnOptions.Store( var S:TStream );
begin
  TPrnItem.Store( S );
  S.Write(LineValue,Sizeof(LineValue)+Sizeof(RelTabs)+Sizeof(Reposition));
end;

procedure TPrnOptions.GetData( var Rec );
  type OptType= record
                  Value : String[5];
                  Opt   : Word;
                end;
begin
  With OptType(Rec) Do
    begin
      Str(LineValue,Value);
      Opt := 0;
      IF RelTabs then
         Opt := Opt or 1;
      IF Reposition then
         Opt := Opt or 2;
    end;
end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnCollection                                       *)
   (*                                                                      *)
    (************************************************************************)

function TPrnCollection.Compare(Key1, Key2: Pointer): Integer;
  var P1,P2 : KeyStr;
begin
  P1 := KeyStr( Key1^ );
  P2 := KeyStr( Key2^ );
  if P1 > P2 then Compare := 1
  else if P1 < P2 then Compare := -1
  else Compare := 0;
end;

function TPrnCollection.KeyOf(Item: Pointer): Pointer;
begin
  KeyOf := @PPrnItem(Item)^.Key^;
end;

function TPrnCollection.FindItem( Key:KeyStr ):PPrnItem;
    var I,C  : Integer;
        Item : PPrnItem;
begin
  if Search(@Key,I) then
     FindItem := PPrnItem(At(I))
  else
     FindItem := nil;
end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrinter                                             *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrinter.Init;
begin
  TObject.Init;
  Version := $0100;
  New( Fonts,    Init( 1,1 ));
  New( Modes,    Init( 1,1 ));
  New( Feeder,   Init( 1,1 ));
  New( Chars,    Init( 1,1 ));
  New( Others,   Init( 1,1 ));
  New( Reserved, Init( 1,1 ));
end;

destructor TPrinter.Done;
begin
  Dispose(Fonts,    Done );
  Dispose(Modes,    Done );
  Dispose(Feeder,   Done );
  Dispose(Chars,    Done );
  Dispose(Others,   Done );
  Dispose(Reserved, Done );
  TObject.Done;
end;

constructor TPrinter.Load( var S:TStream );
begin
  S.Read( Version, Sizeof( Version ));
  New( Fonts,    Load( S ));
  New( Modes,    Load( S ));
  New( Feeder,   Load( S ));
  New( Chars,    Load( S ));
  New( Others,   Load( S ));
  New( Reserved, Load( S ));
end;

procedure TPrinter.Store( var S:TStream );
begin
  S.Write( Version, Sizeof( Version ));
  Fonts^.   Store( S );
  Modes^.   Store( S );
  Feeder^.  Store( S );
  Chars^.   Store( S );
  Others^.  Store( S );
  Reserved^.Store( S );
end;

(************************************************************************)
 (*                                                                      *)
  (*        Object : TPrnDriver                                           *)
   (*                                                                      *)
    (************************************************************************)

constructor TPrnDriver.Init( DllName: PChar; StreamPort:PStream );
  var Shared: TMemoryStream;
begin
  Inherited Init;
 {$ifdef Dpmi}
  PrnHandle := LoadLibrary(DllName);
  IF PrnHandle < 32 then fail;
  DllFncMap('InitPrinter', xInitPrinter  );
  DllFncMap('ResetPrinter',xResetPrinter );
  DllFncMap('EnumFonts',   xEnumFonts    );
  DllFncMap('FormFeed',    xFormFeed     );
  DllFncMap('StrOut',      xStrOut       );
  DllFncMap('DataOut',     xDataOut      );
  DllFncMap('SwitchPort',  xSwitchPort   );

  if StreamPort <> nil then begin
     Shared.Init(1024,1024); { Send Port object over stream to DLL ! }
     Shared.Put(StreamPort);
     xSwitchPort( @Shared );
     Shared.Done;
     Dispose(StreamPort,Done);  { not need any longer ! }
  end;

  xInitPrinter;
 {$endif}
end;

destructor TPrnDriver.Done;
begin
 {$ifdef Dpmi}
  xResetPrinter;
  FreeLibrary(PrnHandle);
 {$endif}
  inherited Done;
end;

{$ifdef Dpmi}
procedure TPrnDriver.DllFncMap( Name:PChar;var P );
begin
  TFarProc(P) := GetProcAddress(PrnHandle,Name );
end;
{$endif}

procedure TPrnDriver.FormFeed;
begin xFormFeed; end;

procedure TPrnDriver.StrOut( St:String );
begin xStrOut( St ); end;

procedure TPrnDriver.DataOut( Data:Pointer; Count:Word );
begin xDataOut( Data, Count ); end;

procedure TPrnDriver.EnumFonts( CallBack:Pointer );
begin xEnumFonts( CallBack ); end;

procedure TPrnDriver.SwitchPort(P:PMemoryStream );
begin xSwitchPort(P); end;

procedure RegisterXPrinter;
begin
  RegisterType(RPrinter);
  RegisterType(RPrnItem);
  RegisterType(RPrnMode);
  RegisterType(RPrnSeq);
  RegisterType(RPrnFont);
  RegisterType(RPrnCollection);
  RegisterType(RPrnOptions);
end;

END.