/*
 * TemplatePortPrivate.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

/**
 * This class serves as a private per port record for each port
 * on a device. This handles most driver duties for both
 * serial and parallel ports.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 */
class TemplatePortPrivate extends PortPrivate {
    private TemplatePortDriver driver;  // associated driver
    private boolean typeSerialPort;     // for quickie port type determination
    private boolean eof;                // indicates end of file condition

    private SerialPortParams params;    // current port params
    private PortStatus stat;            // current port status values
    private PortStatus lastStat;        // last status values returned
    private boolean statChgPending;     // thread(s) waiting for status change

    // output stuff
    private byte outputBuffer[];
    private int outBufSize;             // maximum buffer size set on device
    private int outputCount;            // number of bytes in outputBuffer
    private boolean outputDrained;      // complete output drainout detected
    private boolean writeWaiting;       // thread(s) waiting to put output data

    // input stuff
    private byte inputBuffer[];         // circle queue data buffer
    private static final int IN_BUF_SIZE = 2048;
    private int inputCount;             // number of bytes in inputBuffer
    private int inputFill;              // filling pointer into circle queue
    private int inputEmpty;             // emptying pointer into circle queue
    private boolean readWaiting;        // thread(s) waiting for input data

    /**
     * Instantiates a new PortPrivate record. Note that we avoid keeping
     * a reference around to the Port so as to allow garbage collection
     * to properly reclaim ports the application is no longer referencing.
     *
     * @param portNumber number used both to index driver tables
     * @param port associated Port object
     */
    TemplatePortPrivate( int portNumber, Port port )
    {
        this.portNumber = (byte)portNumber;
        this.driver = (TemplatePortDriver)port.driver;
        typeSerialPort = port instanceof SerialPort;
        inputBuffer = new byte[IN_BUF_SIZE];
        // for template simplicity, this is the number of bytes that can
        // be stuffed in an empty output fifo; actual implementation would
        // buffer more data here and feed the fifo in its small chunks
        outBufSize = 16;
        outputBuffer = new byte[outBufSize];
        statChgPending = false;
    }

    /**
     * Sets default params, gets receive and status chains started.
     */
    synchronized void open()
        throws PortIOException
    {
        SerialPortParams startingParams = new SerialPortParams();

        if( driver.messageLevel >= PortDriver.PORTALL ) System.out.println( "open()" );

        // reset the port (device-specific)

        // enable the port (device-specific)

        // if it's a serial port, set params and start receive chain
        if( typeSerialPort ) {
            params = new SerialPortParams();
            // force current params to invalid value so that setParams will do something
            params.outBaud = 0;
            setParams( startingParams );

            stat = new SerialPortStatus();
            lastStat = new SerialPortStatus();
            // use unused bits in lastStatus to set the modem lines
            ((SerialPortStatus)lastStat).RTS = true;
            ((SerialPortStatus)lastStat).DTR = true;
            setStatus( (SerialPortStatus)lastStat );
        }
        // if it's a parallel port, just create the status objects
        else {
            stat = new ParallelPortStatus();
            lastStat = new ParallelPortStatus();
        }

        // find out what the current status of the port is (device-specific)

        // put something in place to check for status changes periodically
        // via polling or interrupts
    }

    /**
     * Closes a port.
     */
    synchronized void close()
        throws PortIOException
    {
        // stop any more data flow that may sneak in
        eof = true;
        notifyAll();

        if( driver.messageLevel >= PortDriver.PORTALL ) System.out.println( "close()" );

        // do whatever is necessary to disable the port (device-specific)
    }

    /**
     * This method should be called when port status has (or may have) changed.
     * The flags passed here somehow represent status bits at the port.
     */
    synchronized void statChgCallback( byte flags )
        throws PortIOException
    {
        byte CTS = 1;
        byte DSR = 2;
        byte DCD = 4;
        byte PE = 1;
        byte ERR = 2;
        byte SEL = 4;

        if( driver.messageLevel >= PortDriver.PORTALL ) System.out.println( "statChgCallback " + this );
        if( typeSerialPort ) {
            ((SerialPortStatus)stat).CTS = (flags & CTS) != 0;
            ((SerialPortStatus)stat).DSR = (flags & DSR) != 0;
            ((SerialPortStatus)stat).DCD = (flags & DCD) != 0;

            // look for carrier drop if carrier sense mode is set
            if( params.carrierSense && ( ( flags & DCD ) == 0 ) ) eof = true;
            // note that any blocked read is notified below...
        }

        // must be parallel port
        else {
            ((ParallelPortStatus)stat).paperEmpty = (flags & PE) != 0;
            ((ParallelPortStatus)stat).error = (flags & ERR) != 0;
            ((ParallelPortStatus)stat).select = (flags & SEL) != 0;
        }

        boolean wasPending = statChgPending;
        statChgPending = false;
        // if we were waiting on this, wake up waiter(s), including blocked reads
        if( wasPending || ( eof == true ) ) notifyAll();
    }

    /**
     * Verifies the baud rate is valid for the port.
     * @param baud baud rate
     * @return divider or something else needed by driver to configure
     * at the chosen baud rate
     * @exception PortIOException if baud rate not supported
     */
    private int getBaudCode( int baud )
        throws PortIOException
    {
        switch( baud ) {
        case 9600:
        case 19200:
        case 38400:
        case 57600:
        case 115200:
        case 230400:
            // of course, this is device-specific
            return baud / 16;
        default:
            break;
        }
        throw new PortIOException( "Baud rate " + baud + " not supported" );
    }

    /**
     * Check params for legal and supported values. Does device-specific stuff to
     * establish params at port, and then saves param block for future use. Only called
     * for serial ports.
     */
    synchronized void setParams( SerialPortParams newParams )
        throws PortIOException
    {
        // look up the baud code
        int inBaud = getBaudCode( newParams.inBaud );
        int outBaud = getBaudCode( newParams.outBaud );
        if( driver.messageLevel >= PortDriver.PORT ) System.out.println( "Port " + portNumber + " Output baud=" + outBaud + ", Input baud=" + inBaud );

        // translate character size (device-specific); throw exception if invalid

        // translate framing (device-specific); throw exception if invalid

        // translate parity (device-specific); throw exception if invalid

        // translate receiveEnable and loopback (device-specific);
        // throw exception if invalid

        // throw exception for unsupported and illegal flow control modes

        // translate flow control (device-specific)

        // made it to here: the params must be valid, so copy them over
        params = (SerialPortParams) newParams.clone();
    }

    /**
     * Returns params for this port. Only called if this is a serial port.
     * @return SerialPortParams for this port
     */
    synchronized PortParams getParams()
        throws PortIOException
    {
        return (PortParams)params.clone();
    }

    /**
     * Sets modem signal for serial ports. Not called for parallel ports.
     */
    synchronized void setStatus( SerialPortStatus status )
        throws PortIOException
    {
        // save new status
        ((SerialPortStatus)stat).RTS = status.RTS;
        ((SerialPortStatus)stat).DTR = status.DTR;

        // do device-specific stuff to twiddle the bits on the port
    }

    /**
     * Returns appropriate type status block for port
     * @param wait waits for something to change
     * @return PortStatus with current port states
     */
    synchronized PortStatus getStatus( boolean wait )
        throws PortIOException
    {
        // see if wait-for-change mode requested
        if (typeSerialPort) {
            while( wait &&
                ( ((SerialPortStatus)stat).CTS == ((SerialPortStatus)lastStat).CTS ) &&
                ( ((SerialPortStatus)stat).DSR == ((SerialPortStatus)lastStat).DSR ) &&
                ( ((SerialPortStatus)stat).DCD == ((SerialPortStatus)lastStat).DCD ) ) {

                statChgPending = true;
                while( statChgPending ) {
                    try { wait(); } catch( InterruptedException e ) {}
                }
            }
            lastStat = (SerialPortStatus)((SerialPortStatus)stat).clone();
            return (SerialPortStatus)((SerialPortStatus)stat).clone();
        }
        // must be parallel
        while( wait &&
            ( ((ParallelPortStatus)stat).paperEmpty == ((ParallelPortStatus)lastStat).paperEmpty ) &&
            ( ((ParallelPortStatus)stat).error == ((ParallelPortStatus)lastStat).error ) &&
            ( ((ParallelPortStatus)stat).select == ((ParallelPortStatus)lastStat).select ) ) {

            statChgPending = true;
            while( statChgPending ) {
                try { wait(); } catch( InterruptedException e ) {}
            }
        }
        lastStat = (ParallelPortStatus)((ParallelPortStatus)stat).clone();
        return (ParallelPortStatus)((ParallelPortStatus)stat).clone();
    }

    /**
     * Handles writes to this port. If eof occurs, any remaining bytes dumped.
     * Never writes more than outBufSize bytes to the device in one shot.
     * @param buf data buffer to be written
     * @param off offset into buffer
     * @param length length to write
     * @return length written or -1 if eof
     */
    synchronized int write( byte buf[], int off, int length )
        throws PortIOException
    {
        if( driver.messageLevel >= PortDriver.PORTALL ) System.out.println( "write()" );
        int remaining = length;
        int moved = 0;

        // copy to output buffer
        while( remaining != 0 && !eof ) {
            int amt = outBufSize - outputCount;
            amt = amt < remaining ? amt : remaining;
            if( amt != 0 ) {
                System.arraycopy( buf, off + moved, outputBuffer, outputCount, amt );
                outputCount += amt;
                moved += amt;
                remaining -= amt;
                startSend();
                continue;
            }
            // wait for drain if we still have characters
 	      else {
                writeWaiting = true;
                try { wait(); } catch( InterruptedException e ) {}
            }
            writeWaiting = false;
        }
        // check for end of file indication
        if( eof ) {
            outputCount = 0;
            return -1;
        }
   	  return length;
    }

    /**
     * Initiates a FAS send for a port.
     */
    private synchronized void startSend()
        throws PortIOException
    {
        if( driver.messageLevel >= PortDriver.PORTALL ) System.out.println( "startSend()" );

        // device-specific stuff to write the output fifo with the outputBuffer;
        // it should wait here if the fifo cannot take the bytes and
        // should only write outputCount bytes to the fifo when it is ready for them

        outputCount = 0;
        outputDrained = false;
        if( writeWaiting ) notifyAll();
    }

    /**
     * Reads data from the input buffer, or blocks if none is available.
     */
    synchronized int read( byte buf[], int off, int length )
        throws PortIOException
    {
        // wait for input or end of file
        while( length != 0 && inputCount == 0 && !eof ) {
            readWaiting = true;
            try { wait(); } catch( InterruptedException e ) {}
        }
        readWaiting = false;

        // see if we merely got end of file indication
        if( eof ) {
            inputCount = inputFill = inputEmpty = 0;
            return -1;
        }

        // we must have data to transfer since no eof
        int total = 0;
        while( length != 0 && inputCount != 0 ) {
            int amt = IN_BUF_SIZE - inputEmpty;
            amt = inputCount < amt ? inputCount : amt;
            amt = length < amt ? length : amt;
            System.arraycopy( inputBuffer, inputEmpty, buf, off, amt );
            total += amt;
            off += amt;
            length -= amt;
            inputCount -= amt;
            inputEmpty += amt;
            if( inputEmpty == IN_BUF_SIZE ) inputEmpty = 0;
        }

        return total;
    }

    /**
     * Reads input data for this port from the device. Called only by the receiver thread
     * in the driver. This must read all data from the device, discarding
     * any that can't be immediately buffered.
     */
    synchronized void callback( byte[] data, int count )
        throws PortIOException
    {
        // read as much as we can into input circle queue
        while( inputCount != IN_BUF_SIZE && count != 0 ) {
            int amt = IN_BUF_SIZE - inputFill;
            amt = count < amt ? count : amt;
            System.arraycopy( data, 0, inputBuffer, inputFill, amt );
            inputCount += amt;
            inputFill += amt;
            if( inputFill == IN_BUF_SIZE ) inputFill = 0;
            count -= amt;
        }

        // if count is still not zero, we just dump data

        // see if a thread is blocked on read
        if( readWaiting ) notifyAll();
    }

    /**
     * Waits for output to drain output completely both at local buffer and
     * at hardware. If wait time is non-zero, only waits for spec'd number of
     * seconds before giving up.
     */
    synchronized void flush( int waitTime )
        throws PortIOException
    {
        long startTime = System.currentTimeMillis();

        // if end of file, just return
        if( eof ) return;

        // if wait is zero, make it a big value (one week) to simulate "forever"
        if( waitTime == 0 ) waitTime = 60 * 60 * 24 * 7;
        // convert to millisecs
        // if wait is over about 24 days or so, this will go negative
        waitTime *= 1000;

        // device-specific polling to determine when last character went out
        while( !outputDrained && System.currentTimeMillis() - startTime < waitTime ) {

            // issue a wait send if one is not already posted

            // sleep for a bit
            try { wait( 200 ); } catch( InterruptedException e ) {}
        }
    }

    /**
     * Returns the number of bytes of input currently available to be read (without
     * blocking).
     */
    int available()
        throws PortIOException
    {
        return inputCount;
    }

    /**
     * Dumps input/output both locally and at device.
     */
    synchronized void purge( boolean write, boolean read )
        throws PortIOException
    {
        if( write ) {
            outputCount = 0;

            // device-specific code to flush output fifo
        }
        if( read ) {
            inputCount = 0;
            inputEmpty = inputFill;

            // device-specific code to flush input fifo
        }
    }

    /**
     * Sends a break indication.
     */
    synchronized void sendBreak( int duration )
        throws PortIOException
    {
        // device-specific implementation
    }

    /**
     * Returns a short string representation.
     */
    public String toString()
    {
        return " Port " + portNumber;
    }

}
