/*
 * PortDriver.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

/**
 * This abstract class specifies the functionality to be provided
 * by a serial and/or parallel port driver.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 */
abstract public class PortDriver {

    /**
     * Options of NONE means the PortDriver will display no messages for any reason.
     */
    public static final int NONE = 0;
    /**
     * Options of DEV means the PortDriver will display minimal device-related messages.
     */
    public static final int DEV = 1;
    /**
     * Options of DEVALL means the PortDriver will display all device-related messages.
     */
    public static final int DEVALL = 2;
    /**
     * Options of PORT means the PortDriver will display all device-related and
     * minimal port-related messages.
     */
    public static final int PORT = 4;
    /**
     * Options of PORTALL means the PortDriver will display  all device-related and
     * port-related messages.
     */
    public static final int PORTALL = 5;
    /**
     * Options of ALL means the PortDriver will display everything imaginable.
     */
    public static final int ALL = 7;

    // how verbose the driver should be (see above values)
    int messageLevel;
    PortDriverInfo info = new PortDriverInfo();

    // holds PortPrivate objects for each open port
    PortPrivate portPrivate[];

    static final int PORT_UNOPENED = 0;
    static final int PORT_OPENING = 1;
    static final int PORT_OPENED = 2;
    static final int PORT_LOST_CONTACT = 3;
    static final int PORT_CLOSING = 4;
    int portState[];

    /**
     * Returns an object describing the ports supported by the particular
     * instance of the driver.
     *
     * @return PortDriverInfo
     */
    public PortDriverInfo getDriverInfo()
    {
        return (PortDriverInfo)info.clone();
    }

    /**
     * Not called directly, used only indirectly through port objects.
     * Opens a port of the appropriate type with specified open options.
     * This must be included in the subclass, which must then call
     * super.open( port, (PortPrivate)new EtherLitePortPrivate( port.portNumber, port ) );
     * (substituting the proper subclass for EtherLitePortPrivate).
     *
     * @param port port object of desired type
     * @exception PortIOException if <i>number</i> is out of range
     */
    abstract void open( Port port )
        throws PortIOException;

    /**
     * Never should be called except by subclasses open( port ) method.
     */
    void open( Port port, PortPrivate priv )
        throws PortIOException
    {
        int portNumber = priv.portNumber;

        // create PortPrivate object and enter into array
        addToPortList( port, priv );

        // attempt to init the port at the device
        try { port.driverPrivate.open(); }
        catch( PortIOException e ) {
            removeFromPortList( port );
            throw e;
        }
    }

    /**
     * Not called directly, used only indirectly through port objects.
     * Closes the specified serial port waiting first for output to drain.
     *
     * @param port the port object to be closed
     * @exception PortIOException if a problem occurs during the close
     */
    void close( Port port )
        throws PortIOException
    {
        int portNumber = port.driverPrivate.portNumber;
        synchronized( this ) {
            if( portState[portNumber] < PORT_OPENED ) throw new PortIOException ( "Port not opened yet" );
            portState[portNumber] = PORT_CLOSING;
        }

        try {
            // wait for output to complete
            port.driverPrivate.flush( 0 );

            // shut down the port at the device
            port.driverPrivate.close();
        }
        catch( PortIOException e ) {
            // let someone else use this port number despite an error
            removeFromPortList( port );
            throw e;
        }
        removeFromPortList( port );
    }

    /**
     * Adds port to list
     *
     * @param portNumber FAS port number on device
     * @param port port object for port
     * @exception PortIOException if driver already instantiated for device
     */
    private synchronized void addToPortList ( Port port, PortPrivate priv )
        throws PortIOException
    {
        int portNumber = priv.portNumber;
        if( portState[portNumber] != PORT_OPENING ) throw new PortIOException ( "Port being opened by another" );
        port.driverPrivate = priv;
        // let port know who its private driver is too
        portPrivate[portNumber] = priv;
        portState[portNumber] = PORT_OPENED;
    }

    /**
     * Removes port from list
     *
     * @param portNumber FAS port number on device
     * @exception PortIOException if driver already instantiated for device
     */
    private synchronized void removeFromPortList ( Port port )
    {
        int portNumber = port.driverPrivate.portNumber;
        port.driverPrivate = null;
        portPrivate[portNumber] = null;
        portState[portNumber] = PORT_UNOPENED;
        notifyAll();
    }
}