/*
 * Port.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

import  java.io.*;

/**
 * This is a superclass for SerialPort and ParallelPort (and any other
 * future port types).
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 * @see portio.SerialPort
 * @see portio.ParallelPort
 */
public class Port {
    /**
     * Keeps a copy of the driver for this port around.
     */
    PortDriver driver;

    /**
     * Port number on driver for this port. Serial and parallel ports are
     * numbered separately for each port driver, starting with zero.
     */
    int portNumber;

    /**
     * Used by the port driver to store the private object which manages this port.
     * The driver.open() method creates this object and stores a reference to it here.
     */
    PortPrivate driverPrivate;

    /**
     * This instantiates a new port by calling the driver's open routine.
     *
     * @param driver the specific driver instance which will provide the port services
     * @param portNumber the port number to open
     * @exception PortIOException problems encountered during instantiation/open
     */
    Port( PortDriver driver, int portNumber )
        throws PortIOException
    {
        this.driver = driver;
        this.portNumber = portNumber;
        driver.open( this );
    }

    /**
     * Disables the port and makes it available for someone else to use.
     */
    public void close()
        throws PortIOException
    {
        driver.close( this );
    }

    /**
     * Returns an input stream for the port.
     */
    public InputStream getInputStream()
    {
        return new PortInputStream( this );
    }

    /**
     * Returns an output stream for the port.
     */
    public OutputStream getOutputStream()
    {
        return new PortOutputStream( this );
    }

    /**
     * Dumps any buffered data on the input or output streams of the port.
     *
     * @param write if true, dumps any buffered outgoing data
     * @param read if true, dumps any buffered incoming data
     */
    public void purge( boolean write, boolean read )
        throws PortIOException
    {
        driverPrivate.purge( write, read );
    }

    /**
     * Gets the state of the port. Will actually return SerialPortStatus
     * or ParallelPortStatus object. Serial status is modem signals RTS, CTS,
     * DSR, DCD, and DTR. Parallel status is SEL, ERR, and PE.
     *
     * @param wait when true, waits until status changes from last returned value
     * @see SerialPortStatus
     * @see ParallelPortStatus
     * @exception PortIOException if a problem occurs
     */
    public PortStatus getStatus( boolean wait )
        throws PortIOException
    {
        return driverPrivate.getStatus( wait );
    }
}
