// PNDlg.cpp : implementation file

// PrintNow version 1.0.
// Copyright  1997 Ziff-Davis Publishing
// First published in PC Magazine April 22, 1997
// Written by Gregory A. Wolking.

#include "stdafx.h"
#include "PrintNow.h"
#include "PNDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// Custom Window message identifiers
#define WM_MY_TASKBAR_NOTIFY	(WM_USER + 0)	// Used for Taskbar Notification messages.
#define WM_MY_PRINT_CLIPBOARD	(WM_USER + 1)	// Signals PrintScreen or Alt+PrintScreen.
#define WM_MY_UPDATE_BUTTON		(WM_USER + 2)	// Signals change in clipboard contents.
#define WM_MY_POPUP				(WM_USER + 3)	// Signals menu was requested from tray icon.
#define WM_MY_TERMINATE			(WM_USER + 4)	// Allows other instances to shut us down.

/////////////////////////////////////////////////////////////////////////////
// CPNDlg dialog

CPNDlg::CPNDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CPNDlg::IDD, pParent)
{
	// Note that all variables are initialized in Init_Dialog_Data().
	//{{AFX_DATA_INIT(CPNDlg)
	m_bAlways = TRUE;
	m_bCenterImage = TRUE;
	m_bEnabled = TRUE;
	m_intScaleMode = 0;
	//}}AFX_DATA_INIT
	// Note that LoadIcon does not require a subsequent DestroyIcon in Win32
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
}

void CPNDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPNDlg)
	DDX_Control(pDX, IDC_cmdUnload, m_cmdUnload);
	DDX_Control(pDX, IDC_lblPrintWait, m_lblPrintWait);
	DDX_Control(pDX, IDC_lblCurrent, m_lblCurrent);
	DDX_Control(pDX, IDC_lblStatus2, m_lblStatus);
	DDX_Control(pDX, IDC_cmdPrintClipboard, m_cmdPrintClipboard);
	DDX_Control(pDX, IDC_cmdApply, m_cmdApply);
	DDX_Check(pDX, IDC_chkAlways, m_bAlways);
	DDX_Check(pDX, IDC_chkCenter, m_bCenterImage);
	DDX_Check(pDX, IDC_chkEnabled, m_bEnabled);
	DDX_Radio(pDX, IDC_optScale0, m_intScaleMode);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CPNDlg, CDialog)
	//{{AFX_MSG_MAP(CPNDlg)
	ON_WM_DESTROY()
	ON_BN_CLICKED(IDC_cmdApply, OncmdApply)
	ON_BN_CLICKED(IDC_cmdUnload, OncmdUnload)
	ON_WM_DRAWCLIPBOARD()
	ON_WM_CHANGECBCHAIN()
	ON_WM_CONTEXTMENU()
	ON_MESSAGE(WM_MY_TASKBAR_NOTIFY, On_Taskbar_Notify)
	ON_MESSAGE(WM_MY_PRINT_CLIPBOARD, On_Print_Clipboard)
	ON_MESSAGE(WM_MY_UPDATE_BUTTON, On_Update_Button)
	ON_MESSAGE(WM_MY_POPUP, On_Popup)
	ON_MESSAGE(WM_MY_TERMINATE, On_Terminate)
	ON_WM_TIMER()
	ON_BN_CLICKED(IDC_cmdPrintClipboard, Do_Print_Clipboard)
	ON_BN_CLICKED(IDC_cmdPrintSetup, Do_Print_Setup)
	ON_BN_CLICKED(IDC_cmdUnloadAll, OncmdUnloadAll)
	//}}AFX_MSG_MAP
	// Use one handler for check boxes and radio buttons.
	ON_CONTROL_RANGE(BN_CLICKED, IDC_chkEnabled, IDC_optScale3, Update_Apply_Button)
	// Use one handler for pop-up menu commands.
	ON_COMMAND_RANGE(ID_MY_CAPTURE, ID_MY_KILL_ALL, On_My_Command)
	// Help support.
	ON_COMMAND(ID_CONTEXT_HELP, OnHelpContext)	// Right-click on dialog.
	ON_COMMAND(ID_HELP, OnHelp)					// F1 help.
	ON_COMMAND(ID_DEFAULT_HELP, OnHelpFinder)	// Help button or menu Help command.
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPNDlg message handlers for standard dialog controls and events.
/////////////////////////////////////////////////////////////////////////////
BOOL CPNDlg::OnInitDialog()
{
	CDialog::OnInitDialog();

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon
	
	UINT result;

	// Perform bulk of initialization. If result is non-zero,
	// display error message and quit.
	if ((result = Init_Dialog_Data()))
	{
		AfxMessageBox(result, MB_ICONSTOP, 0);
		EndDialog(IDCANCEL);
	}
	else
	{
		SetWindowText(m_strCaption);		// Display instance counter in our caption.
		UpdateData(FALSE);					// Make sure controls match our settings.
		Set_Tooltip_Text();					// Update tooltip display.
		// This is a workaround becase DoModal() forces the dialog to be visible when launched.
		// First, move the dialog completely off the screen without changing its size
		{
			CRect dlg_rect;

			GetWindowRect(dlg_rect);
			dlg_rect.NormalizeRect();
			dlg_rect.left = -(dlg_rect.Width() + 1);
			dlg_rect.top = -(dlg_rect.Height() + 1);
			dlg_rect.right = dlg_rect.bottom = -1;
			MoveWindow(dlg_rect, FALSE);
		}
		// Next, set a timer. When timer fires, OnTimer() kills the timer,
		// hides the dialog, and centers it on the screen. The net effect is that the user
		// never sees the dialog as the program starts.
		m_intMyTimer = SetTimer(1, 100, NULL);
	}
	return TRUE;  // Let Windows handle initial focus
}

// Called immediately before our main window is destroyed (i.e., program is terminating).
void CPNDlg::OnDestroy()
{
	// If needed, unhook from clipboard viewer chain and taskbar notification area..
	if (m_bClipboardHooked)
		Hook_Clipboard(FALSE);
	if (m_bTaskbarHooked)
		Do_Taskbar_Icon(NIM_DELETE, NULL);
	// Clean up common dialog box and task bar pop-up menu.
	if (m_pPrintDialog)
	{
		// If common print dialog object exists, make sure to
		// clean up printer DC and/or memory it may be holding.
		PRINTDLG& ppd = m_pPrintDialog->m_pd;	// Get a reference to the dialog's m_pd structure.
		if (ppd.hDC)				// If Printer DC exists,
		{
			::DeleteDC(ppd.hDC);	// Delete it.
			ppd.hDC = NULL;
		}
		if (ppd.hDevMode)				// If memory is allocated to DEVMODE structure,
		{
			GlobalFree(ppd.hDevMode);	// release it
			ppd.hDevMode = NULL;		// and NULL the pointer.
		}
		if (ppd.hDevNames)				// If memory is allocated to DEVNAMES structure,
		{
			GlobalFree(ppd.hDevNames);	// release it
			ppd.hDevNames = NULL;		// and NULL the pointer.
		}
		delete m_pPrintDialog;			// Finally, delete the dialog object
		m_pPrintDialog = NULL;			// and NULL its pointer.
	}
	if (m_pMainMenu)					// If pop-up menu object still exists,
	{
		delete m_pMainMenu;				// delete it
		m_pMainMenu = NULL;				// and null its pointer.
	}
	WinHelp(0L, HELP_QUIT);		// Make sure WINHELP.EXE is shut down.
	CDialog::OnDestroy();		// Call base class to finish up.
}

// "Apply" command button.
void CPNDlg::OncmdApply() 
{
	UpdateData(TRUE);
	if (!Check_Clipboard_Hook())	// Hook/unhook clipboard if necessary.
		return;						// Return immediately if call fails.
	m_cmdApply.EnableWindow(FALSE);	// Disable the button.
}

// "Unload PrintNow" command button.
void CPNDlg::OncmdUnload() 
{
	m_bShutdown = TRUE;		// Set flag so OnCancel knows we're shutting down.
	OnCancel();				// Do it!
}

// "OK" command button.
// Note that we do _not_ call the base class for this hander as
// we don't want OK to close the dialog, just hide it.
void CPNDlg::OnOK() 
{
	OncmdApply();			// Update settings from controls.
	Hide_Dialog();			// Hide the dialog.
}

// "Cancel" command button (or {Esc} key).
void CPNDlg::OnCancel() 
{
	if (m_bShutdown)			// Flag set?
		CDialog::OnCancel();	// If so, call base class to terminate.
	else						// Otherwise,
		Hide_Dialog();			// just hide the dialog.
}

// Handles WM_DRAWCLIPBOARD messages.
// Called by the clipboard when either the clipboard contents change
// or when Hook_Clipboard() adds us to the chain. In the latter case,
// Hook_Clipboard() sets m_bFirstPass to TRUE so we know that the WM_DRAWCLIPBOARD
// message did not represent a change in the clipboard's contents.
// Note: The clipboard waits for us to process this message, so we must spend as little
// time in this handler as possible to avoid hanging the system.
void CPNDlg::OnDrawClipboard() 
{
	if (m_bFirstPass)			// If flag is set,
		m_bFirstPass = FALSE;	// clear it.
	else
	{
		// Make sure other viewers (if any) process the message before we continue.
		if (m_hWndNextViewer != NULL)
			::SendMessage(m_hWndNextViewer, WM_DRAWCLIPBOARD, 0, 0);
		// If main dialog is active, post message to update the Print Clipboard Image button.
		if (m_bDialogActive)
			PostMessage(WM_MY_UPDATE_BUTTON);
		// If we're not already printing,
		if (!m_bIsPrinting)
			// and the clipboard is in the proper format,
			if (IsClipboardFormatAvailable(CF_DIB))
				// and the clipboard owner is the desktop under Win 95
				// or NULL under Win NT,
				if (GetClipboardOwner() == (m_dwPlatform == VER_PLATFORM_WIN32_WINDOWS ? GetDesktopWindow() : NULL))
					// Copy clipboard data and post a message to ourselves to print it after we leave this handler.
					// Note: It's important to copy the data _now_. If we put the copy function in our
					// Do_Print_Clipboard handler, another application could place different data in the
					// clipboard before our window gets a chance to handle the message.
					PostMessage(WM_MY_PRINT_CLIPBOARD, Copy_Clipboard_Image(), 0);
	}
	// Call base class to let MFC clean up internals.
	CDialog::OnDrawClipboard();
}

// Called when the Clipboard removes a viewer from the chain.
void CPNDlg::OnChangeCbChain(HWND hWndRemove, HWND hWndAfter) 
{
	// If the window being removed is _our_ next viewer,
	if (m_hWndNextViewer == hWndRemove)
		m_hWndNextViewer = hWndAfter;	// Store new next viewer.
	else								// Otherwise,
		if (m_hWndNextViewer)			// if there is a viewer after us, pass the message along.
			::SendMessage(m_hWndNextViewer, WM_CHANGECBCHAIN, (WPARAM) hWndRemove, (LPARAM) hWndAfter);
	// Call base class to let MFC take care of internals.
	CDialog::OnChangeCbChain(hWndRemove, hWndAfter);
}

// Called when user presses F1
void CPNDlg::OnHelp()
{
	// Handle special case for Print Dialogs.
	HWND hWnd = ::GetActiveWindow();			// Get the active window.
	if (hWnd != NULL && hWnd != m_hWnd)			// If there is one and it's not the main dialog,
		WinHelp(0x37008, HELP_CONTEXTPOPUP);	// show popup topic for print dialogs.
	else
		CDialog::OnHelp();						// Otherwise lauch default help.
}

// Called when user selects "What's this?" from the dialog's context menu.
// Note that m_dwPopupContext is set by the OnContextMenu handler before it launches the menu.
void CPNDlg::OnHelpContext()
{
	WinHelp(m_dwPopupContext, HELP_CONTEXTPOPUP);
}

// "Help.." button or pop-up menu "Help..." command.
void CPNDlg::OnHelpFinder()
{
	WinHelp(0, HELP_FINDER);
}

// Called when user clicks the right mouse button on the main dialog.
void CPNDlg::OnContextMenu(CWnd* pWnd, CPoint point) 
{
	CPoint local(point);
	HWND target;

	// Convert mouse cursor location from screen to client coordinates.
	ScreenToClient(&local);
	// See if mouse is within the dialog's client area.
	// Note that all static frames are set as transparent, otherwise ChildWindowFromPointEx
	// would find a frame before it finds a control within that frame's area.
	if ((target = ::ChildWindowFromPointEx(m_hWnd, local, CWP_SKIPTRANSPARENT | CWP_SKIPINVISIBLE)) == NULL) 
		return;
	// See if target window is a control, not the dialog itself.
	if ((m_dwPopupContext = ::GetDlgCtrlID(target)) == 0)
	{
		// Try again, including all child windows in the search.
		if ((target = ::ChildWindowFromPointEx(m_hWnd, local, CWP_SKIPINVISIBLE)) == NULL)
			return;
		else
			// If still no control identifier, nothing else to do.
			if ((m_dwPopupContext = ::GetDlgCtrlID(target)) == 0)
					return;
	}
	// Handle special case for icon and labels within the About frame; these
	// use the default IDC_STATIC identifier, so change it to the About frame's identifier.
	// This saves overhead in the Help file since we don't need a unique identifier for
	// every static control within the frame.
	if (m_dwPopupContext == IDC_STATIC)
		m_dwPopupContext = IDC_fraAbout;
	// Adjust control ID to match the Help Popup context IDs generated by MAKEHELP.BAT.
	m_dwPopupContext += 0x30000;
	// Create and launch the "What's this?" pop-up menu.
	{
		CMenu help_menu;
		CMenu * pPopUp;
		help_menu.LoadMenu(IDR_MENU2);
		pPopUp = help_menu.GetSubMenu(0);
		pPopUp->TrackPopupMenu(TPM_CENTERALIGN | TPM_LEFTBUTTON, point.x, point.y, this);
	}
}

// Called when the timer (set by OnInitDialog) fires.
void CPNDlg::OnTimer(UINT nIDEvent) 
{
	KillTimer(m_intMyTimer);			// Kill the timer.
	m_intMyTimer = 0;
	Hide_Dialog();						// Hide the dialog.
	CenterWindow(GetDesktopWindow());	// Center dialog on the screen.
}

// "Unload all instances" command
void CPNDlg::OncmdUnloadAll() 
{
	HWND target = NULL;
	target = FindWindowEx(NULL, NULL, "#32770", NULL);			// Look for any top-level dialogs.
	while (target)												// If found,
	{
		if (target != m_hWnd)									// and it's not our dialog,
			::PostMessage(target, WM_MY_TERMINATE, 0, 0);		// post our private shutdown message.
		target = FindWindowEx(NULL, target, "#32770", NULL);	// Look for next dialog.
	}
	OncmdUnload();												// Shut ourselves down.
}

////////////////////////////
// Custom message handlers
////////////////////////////

// Handles notification messages from our task bar icon.
// Note: We only have one icon on the taskbar, so we can
// ignore the icon identifier (wParam).
LONG CPNDlg::On_Taskbar_Notify(WPARAM wParam, LPARAM lParam)
{
	UINT mouse_msg = (UINT) lParam;

	if (mouse_msg != WM_MOUSEMOVE)	// Ignore mouse movement messages.
	{
		switch(mouse_msg)
		{
		case WM_LBUTTONDBLCLK:		// Left double-click.
			if (!Bring_To_Top())	// If visible, bring print or main dialog to foreground.
				Show_Dialog();		// Otherwise, activate main dialog.
			break;
		case WM_LBUTTONUP:			// Left button released.
			Bring_To_Top();			// If visible, bring print or main dialog to foreground.
			break;
		case WM_RBUTTONUP:			// Right button released.
			if (!Bring_To_Top())						// If main or print dialog is not active,
				if (!m_bMenuActive)						// and menu is not already open,
					PostMessage(WM_MY_POPUP, 0, 0);		// launch the menu.
		}
	}
	return 0L;	// Let the system know that we processed the message.
}

// Handles the WM_MY_PRINT_CLIPBOARD message posted by OnDrawClipboard().
// Note that OnDrawClipboard passes the result of Copy_Clipboard_Image()
// as the wParam parameter.
LONG CPNDlg::On_Print_Clipboard(WPARAM wParam, LPARAM lParam)
{
	if (wParam)		// Was copy successful?
	{
		if (m_bDialogActive)			// Is dialog active?
			SetForegroundWindow();		// Put dialog in foreground.
		else							// Otherwise, show the window but don't set the active flag
			ShowWindow(SW_RESTORE);		// so we know to hide it again when printing finishes.
		Do_Print(m_bAlways);			// Print the image.
		Discard_Image();				// Release memory allocated to hold our copy of the image.
	}
	else
		AfxMessageBox(IDS_COPY_FAIL, MB_ICONINFORMATION, 0);	// Otherwise, display error message.
	return 0L;	// Let the system know that we processed the message.
}

// Handles the WM_MY_UPDATE_BUTTON message posted by OnDrawClipboard().
LONG CPNDlg::On_Update_Button(WPARAM wParam, LPARAM lParam)
{
	m_cmdPrintClipboard.EnableWindow(IsClipboardFormatAvailable(CF_DIB));
	return 0L;
}

// Handles the WM_MY_POPUP message posted by On_Taskbar_Notify().
LONG CPNDlg::On_Popup(WPARAM wParam, LPARAM lParam)
{
	Do_Taskbar_Menu();
	return 0L;
}

// Handles the WM_MY_TERMINATE message posted by another instance's
// OncmdUnloadAll() function.
LONG CPNDlg::On_Terminate(WPARAM wParam, LPARAM lParam)
{
	OncmdUnload();
	return 0L;
}

// Handles all commands (except Help...) from the task bar pop-up menu.
// Note that the order of the menu's command identifiers is important!
void CPNDlg::On_My_Command(UINT nID)
{
	switch (nID)							// Branch based on the command selected.
	{
	case ID_MY_CAPTURE:						// Screen Capture Enabled (check box)
		m_bEnabled = !m_bEnabled;			// Toggle flag
		if (!Check_Clipboard_Hook())		// Hook or unhook clipboard as appropriate.
			return;							// Return immediately if call failed.
		break;
	case ID_MY_ALWAYS:						// Always show Print Dialog (check box)
		m_bAlways = !m_bAlways;
		break;
	case ID_MY_CENTER:						// Center Image (check box)
		m_bCenterImage = !m_bCenterImage;
		break;
	case ID_SCALE0:							// Scaling sub-menu (radio buttons)
	case ID_SCALE1:
	case ID_SCALE2:
	case ID_SCALE3:
		m_intScaleMode = nID - ID_SCALE0;
		break;
	case ID_MY_PRINT_SETUP:					// Print Setup (command)
		Do_Print_Setup();
		break;
	case ID_MY_PRINT_LAST:					// Print Clipboard Image (command)
		Do_Print_Clipboard();
		break;
	case ID_MY_ACTIVATE:					// Open PrintNow (command)
		Show_Dialog();
		break;
	case ID_MY_EXIT:						// Unload this instance (command)
		m_bShutdown = TRUE;
		OnCancel();
		break;
	case ID_MY_KILL_ALL:					// Unload all instances (command).
		OncmdUnloadAll();
	case ID_MY_CLOSE_MENU:					// Close Menu (command)
		break;
	}
	// If choice was an option setting, launch the menu again.
	if (nID <= ID_SCALE3)
		PostMessage(WM_MY_POPUP, 0, 0);
}

// Called when user clicks any of the check boxes or radio buttons in the dialog.
// Again, note that the value of those controls' identifiers (like those
// in the task bar menu) is important; they must be contiguous.
void CPNDlg::Update_Apply_Button(UINT nID)
{
	m_cmdApply.EnableWindow(TRUE);
}

//////////////////////////////////
// Helper functions.
//////////////////////////////////

// Checks status flags, then adds us to or removes us from the
// clipboard viewer chain appropriately.
// Displays error message, shuts program down, and returns FALSE on failure.
BOOL CPNDlg::Check_Clipboard_Hook()
{
	if (m_bEnabled != m_bClipboardHooked)	// Check status flags. If they don't match,
	{
		if (!Hook_Clipboard(m_bEnabled))	// try to hook/unhook accordingly.
		{	// If call fails, display error message and shut down.
			AfxMessageBox(IDS_CLIP_HOOK_FAIL, MB_ICONSTOP, 0);
			m_bShutdown = TRUE;
			OnCancel();
			return FALSE;
		}
	}
	return TRUE;
}

// Hooks or unhooks us from the clipboard viewer chain.
// Parameter: TRUE = add us to the chain.
//            FALSE = remove us from the chain.
// Return value: TRUE = success
//				 FALSE = failure
// NOTE: Status flags have already been checked before this function
//       is called to ensure that we don't try to hook the clipboard
//       again if we're already hooked and vice versa.
// NOTE: Calling functions will terminate the program if this function returns FALSE.
//       It's futile to continue using the clipboard viewer chain if the
//       API calls are failing!
BOOL CPNDlg::Hook_Clipboard(BOOL hook)
{
	HWND first, next;
	DWORD error_code = 0;
	BOOL status = FALSE;

	if (hook)
	{
		first = ::GetClipboardViewer();			// Get handle to first viewer in chain.
		status = (first != m_hWnd);				// Make sure it's not us!
		if (status)
		{
			// Set flag to ignore WM_DRAWCLIPBOARD message resulting from SetClipboardViewer().
			m_bFirstPass = TRUE;
			next = ::SetClipboardViewer(m_hWnd);	// Add our window to the chain, get handle to next viewer.
			error_code = GetLastError();			// Get result code from SetClipboardViewer().
			// Test all possible error conditions:
			if (SUCCEEDED(error_code) &&		// SetClipboardViewer must have succeeded, and the handle
				(next == first))				// it returned must have been the first window in the chain.
			{									// If so,
				m_hWndNextViewer = next;		// save handle to next viewer and
				m_bClipboardHooked = TRUE;		// set flag so we know we're hooked into the chain.
			}
			else								// Otherwise,
			{
				m_bFirstPass = FALSE;			// Clear flags.
				m_bClipboardHooked = FALSE;
				status = FALSE;					// Signal failure to caller.
			}
		}
	}
	else
	{
		// Note: Contrary to the MFC documentation, the return value of CWnd::ChangeClipboardChain does
		// not indicate success or failure. If there are other windows in the chain, the
		// return value is usually (but not always) FALSE, because those other windows will usually
		// (but not always) return FALSE when they process the message. If we are the only window in the chain,
		// the return value is usually TRUE. The documentation for ::ChangeClipboardChain (the base
		// class API call that I'm using here) is correct. Note the use of GetLastError() to determine
		// success or failure instead of relying on ::ChangeClipboardChain's return value.
		::ChangeClipboardChain(m_hWnd, m_hWndNextViewer);	// Remove us from the chain.
		if ( (status = SUCCEEDED(GetLastError())) )	// Set return status for caller.
		{											// If successful,
			m_hWndNextViewer = NULL;				// zap handle to next viewer
			m_bClipboardHooked = FALSE;				// and clear flag so we know we're unhooked.
		}
	}
	return status;
}

// Adds, modifies, or deletes our task bar icon.
// Input: func = NIM_ADD, NIM_MODIFY, or NIM_DELETE.
//		  text = string for tooltip text.
// Returns: TRUE = success
//			FALSE = failure.
BOOL CPNDlg::Do_Taskbar_Icon(UINT func, LPCTSTR text)
{
	NOTIFYICONDATA tnid;
	BOOL result;

	// Initialize common members of TNID structure.
	tnid.cbSize = sizeof(tnid);
	tnid.hWnd = m_hWnd;
	tnid.uID = IDR_MAINFRAME;
	tnid.uFlags = 0;
	// Set remaining members based on task to be performed.
	switch (func)
	{
	case NIM_ADD:		// Add new icon.
		tnid.uFlags = NIF_MESSAGE | NIF_ICON | NIF_TIP;		// Signal all members valid.
		tnid.uCallbackMessage = WM_MY_TASKBAR_NOTIFY;		// Set callback message identifier.
		// Note: Use LoadImage to make sure we get the small icon.
		tnid.hIcon = (HICON) LoadImage(	AfxGetApp()->m_hInstance,
										MAKEINTRESOURCE(IDR_MAINFRAME),
										IMAGE_ICON,
										16,
										16,
										LR_DEFAULTCOLOR);
		break;
	case NIM_MODIFY:	// Modify existing icon.
		tnid.uFlags = NIF_TIP;								// Signal only tooltip text is valid.
	case NIM_DELETE:	// Remove existing icon.
		tnid.hIcon = NULL;									// Don't need icon handle for removal.
	}
	// Make sure tooltip text pointer is initialized.
	if ((func != NIM_DELETE) && (text != NULL))
		lstrcpyn(tnid.szTip, text, sizeof(tnid.szTip));
	else
		tnid.szTip[0] = '\0';
	// Call API, return success or failure to caller.
	result = Shell_NotifyIcon(func, &tnid);
	// Discard the icon if it was loaded
	// (Shell_NotifyIcon makes its own copy of the image, so we don't have to waste
	// memory for it).
	if (tnid.hIcon)
		DestroyIcon(tnid.hIcon);
	return result;
}

// Sets pop-up menu's check boxes and radio buttons.
// Called immediately before menu is launched.
void CPNDlg::Prepare_Menu()
{
	m_pPopUpMenu->CheckMenuItem(ID_MY_CAPTURE, m_bEnabled ? MF_CHECKED : MF_UNCHECKED);
	m_pPopUpMenu->CheckMenuItem(ID_MY_ALWAYS, m_bAlways ? MF_CHECKED : MF_UNCHECKED);
	m_pPopUpMenu->CheckMenuItem(ID_MY_CENTER, m_bCenterImage ? MF_CHECKED : MF_UNCHECKED);
	m_pPopUpMenu->EnableMenuItem(ID_MY_PRINT_LAST, IsClipboardFormatAvailable(CF_DIB) ? MF_ENABLED : MF_GRAYED);
	m_pScaleSubMenu->CheckMenuRadioItem(0, 3, m_intScaleMode, MF_BYPOSITION);
}

// Handles the bulk of our initialization.
// Called by OnInitDialog.
// Returns 0 if successful, otherwise returns identifier of
// string to be displayed in error message box before program terminates.
UINT CPNDlg::Init_Dialog_Data()
{
	// Create common dialog object.
	m_pPrintDialog = new CPrintDialog(FALSE, PD_ALLPAGES, this);
	if (m_pPrintDialog == NULL)		// Return error code if creation fails.
		return IDS_CMN_DLG_FAIL;
	else
	{
		// Note: GetDefaults() has side effects. First, it alters the m_pd.Flags to retrieve the Windows
		// defaults without displaying the dialog. If we don't save and restore m_pd.Flags, subsequent calls
		// to DoModal() will return the Windows default printer instead of displaying a dialog.
		// Second, since we called CPrintDialog's constructor with bPrintSetupOnly set to FALSE, it creates
		// a printer device context that we don't need.
		UINT flags = m_pPrintDialog->m_pd.Flags;	// Save flags.
		m_pPrintDialog->GetDefaults();				// Retrieve Windows default printer settings.
		m_pPrintDialog->m_pd.Flags = flags;			// Restore flags.
		if (m_pPrintDialog->m_pd.hDC)				// If context was created,
			DeleteDC(m_pPrintDialog->m_pd.hDC);		// delete it so it doesn't waste memory.
		m_pPrintDialog->m_pd.nCopies = 1;			// Set number of copies.
		Show_Printer_Name();						// Update printer name caption.
	}
	// Load and initialize the task bar popup menu.
	m_pMainMenu = new CMenu;
	if (m_pMainMenu->LoadMenu(IDR_MENU1))					// Load the menu.
		if ((m_pPopUpMenu = m_pMainMenu->GetSubMenu(0)));	// If successful, get pointer to main menu.
			m_pScaleSubMenu = m_pPopUpMenu->GetSubMenu(6);	// If successful, get pointer to scale sub-menu.
	if (!(m_pMainMenu && m_pPopUpMenu && m_pScaleSubMenu))	// If failure, return error code.
		return IDS_MENU_FAIL;
	// Modify the "Open" item caption to include the instance counter.
	CString scratch;
	scratch = "&Open " + m_strCaption;
	m_pPopUpMenu->ModifyMenu(ID_MY_ACTIVATE, MF_BYCOMMAND, ID_MY_ACTIVATE, scratch);
	// Do the same for the "Unload" menu item and dialog button.
	scratch = "&Unload " + m_strCaption;
	m_pPopUpMenu->ModifyMenu(ID_MY_EXIT, MF_BYCOMMAND, ID_MY_EXIT, scratch);
	m_cmdUnload.SetWindowText(scratch);
	// Set the "Open" command as the default.
	SetMenuDefaultItem(m_pPopUpMenu->m_hMenu, ID_MY_ACTIVATE, FALSE);
	// Try to place our icon in the system tray, report error on failure.
	if (!(m_bTaskbarHooked = Do_Taskbar_Icon(NIM_ADD, m_strCaption)))
		return IDS_TASKBAR_FAIL;
	// Now that the critical stuff is out of the way, we can take care of initializing the rest of our data.
	// Handles and pointers
	m_hWndNextViewer = NULL;
	m_hDIB = NULL;
	m_pPal = NULL;
	// Flags
	m_bDialogActive = FALSE;
	m_bClipboardHooked = FALSE;
	m_bIsPrinting = FALSE;
	m_bInPrintLoop = FALSE;
	m_bMenuActive = FALSE;
	m_bShutdown = FALSE;
	// Settings
	m_bAlways = TRUE;
	m_bEnabled = TRUE;
	m_bCenterImage = TRUE;
	m_intScaleMode = 0;
	m_cmdPrintClipboard.EnableWindow(IsClipboardFormatAvailable(CF_DIB));
	Parse_Command_Line();				// Process command-line parameters
	if (m_bEnabled)						// If Screen Capture Enabled setting was not disabled,
		if (!Hook_Clipboard(TRUE))		// Try to hook the clipboard.
			return IDS_CLIP_HOOK_FAIL;	// Return error code if hook fails.
	return 0;
}

// Hides the main dialog.
void CPNDlg::Hide_Dialog()
{
	ShowWindow(SW_HIDE);		// Hide the window.
	m_bDialogActive = FALSE;	// Clear the active flag.
	Set_Tooltip_Text();			// Update tray icon tooltip.
}

// Activates the main dialog.
void CPNDlg::Show_Dialog()
{
	UpdateData(FALSE);					// Make sure checkboxes and radio buttons are in synch.
	m_cmdApply.EnableWindow(FALSE);		// Update the "Apply" and "PrintClipboard Image" buttons.
	m_cmdPrintClipboard.EnableWindow(IsClipboardFormatAvailable(CF_DIB));
	m_lblStatus.SetWindowText("Ready");	// Set status text.
	ShowWindow(SW_RESTORE);				// Activate the dialog.
	m_bDialogActive = TRUE;				// Set active flag.
	Set_Tooltip_Text();					// Update tray icon's tooltip.
}

// Called by On_Popup()
// Calculates proper position and displays task bar pop-up menu.
void CPNDlg::Do_Taskbar_Menu()
{
	HWND tray_wnd;

	// If the pop-up menu overlaps the system tray window, the tray's tooltips may "stall"
	// after the menu closes; that is, the tray will stop showing tooltips until you activate
	// another window, then click somewhere inside the tray window.
	// First, locate the taskbar.
	if ((tray_wnd = FindWindowEx(NULL, NULL, "Shell_TrayWnd", NULL)) != NULL)
	{
		// Next, locate the taskbar notification area window.
		if ((tray_wnd = FindWindowEx(tray_wnd, NULL, "TrayNotifyWnd", NULL)) != NULL)
		{
			CRect tray_rect;
			UINT flags;
			int x, y;

			::GetWindowRect(tray_wnd, tray_rect);		// Get taskbar notification rectangle.
			tray_rect.NormalizeRect();					
			if (tray_rect.left < 100)			// Taskbar docked at left side of screen?
			{
				flags = TPM_LEFTALIGN;			// Put menu to right.
				x = tray_rect.right + 5;
			}
			else								// Otherwise,
			{
				flags = TPM_RIGHTALIGN;			// Put menu to left.
				x = tray_rect.left - 5;
			}
			if (tray_rect.bottom < 100)			// Taskbar docked at top of screen?
				y = tray_rect.bottom + 5;		// If so, put menu below.
			else								// Otherwise,
				y = tray_rect.top - 5;			// put menu above.
			flags |= TPM_LEFTBUTTON;			// Menu tracks the left mouse button.
			Prepare_Menu();						// Set check boxes and radio buttons.
			m_bMenuActive = TRUE;				// Signal that menu is open.
			Set_Tooltip_Text();					// Update tooltip.
			SetForegroundWindow();				// Make dialog the active window.
												// Note: this is a workaround required to make the menu
												// operate correctly. See article Q135788 in the MSKB
												// for further information on this behavior.
			m_pPopUpMenu->TrackPopupMenu(flags, x, y, this, NULL);		// Launch the menu.
			PostMessage(WM_NULL, 0, 0);			// Second part of the workaround.
			m_bMenuActive = FALSE;				// Signal that menu is closed.
			Hide_Dialog();						// Hide the dialog.
		}
	}
}

// Sets Tooltip text for the Task bar icon.
void CPNDlg::Set_Tooltip_Text()
{
	CString text(m_strCaption);			// Start with the instance counter.

	text += " is ";							// "PrintNow (n) is "
	if (m_bInPrintLoop)						// If printing an image,
		text += "busy printing an image";	// say so.
	else									// Otherwise,
	{
		text += m_bEnabled ? "enabled" : "disabled";			// Display Screen Capture status.
		if (m_bDialogActive || m_bMenuActive || m_bIsPrinting)	// If waiting for a dialog or menu,
		{
			text += ", waiting for ";							
			if (m_bMenuActive)									// Indicate which one we're waiting for.
				text += "a menu command";
			else
			{
				if (m_bIsPrinting)
					text += "a print dialog";
				else
					text += "the main dialog";
			}
		}
	}
	Do_Taskbar_Icon(NIM_MODIFY, text);		// Install the new tooltip text.
}

// Makes a local copy of the clipboard's contents.
// Called only when clipboard contains data in CF_DIB format.
BOOL CPNDlg::Copy_Clipboard_Image()
{
	HDIB hNewDIB = NULL;

	if (OpenClipboard())				// Try to open the clipboard.
	{
		hNewDIB = (HDIB) CopyHandle(::GetClipboardData(CF_DIB));	// Make a copy of its data.
		CloseClipboard();											// Close the clipboard.
		if (m_hDIB != NULL)											// If we already have an image,
			::GlobalFree((HGLOBAL) m_hDIB);							// discard it.
		m_hDIB = hNewDIB;											// Point to new image.
		Init_DIB_Data();											// Intialize DIB palette.
	}
	return m_hDIB != NULL;				// Return FALSE if unable to copy or initialize the DIB.
}

// Handles palette creation for a new DIB image.
// If initialization fails, sets m_hDIB to NULL.
void CPNDlg::Init_DIB_Data()
{
	if (m_pPal != NULL)		// If palette exists,
	{
		delete m_pPal;		// discard it.
		m_pPal = NULL;
	}
	if (m_hDIB == NULL)		// If DIB handle is already NULL,
		return;				// nothing to do.
	// Lock the image's memory.
	LPSTR lpDIB = (LPSTR) ::GlobalLock((HGLOBAL) m_hDIB);
	// Check image size. If it's too big, release the memory and display appropriate message.
	if (::DIBWidth(lpDIB) > INT_MAX ||::DIBHeight(lpDIB) > INT_MAX)
	{
		::GlobalUnlock((HGLOBAL) m_hDIB);
		::GlobalFree((HGLOBAL) m_hDIB);
		m_hDIB = NULL;
		AfxMessageBox(IDS_DIB_TOO_BIG, MB_ICONINFORMATION, 0);
		return;
	}
	::GlobalUnlock((HGLOBAL) m_hDIB);
	m_pPal = new CPalette;	// Create a new palette
	if (m_pPal == NULL)		// Free image memory and report error if unable to create palette.
	{
		::GlobalFree((HGLOBAL) m_hDIB);
		m_hDIB = NULL;
		return;
	}
	if (::CreateDIBPalette(m_hDIB, m_pPal) == NULL)	// Initialize palette from the DIB.
	{
		// DIB may not have a palette
		delete m_pPal;
		m_pPal = NULL;
		return;
	}
}

// Scales image as user requested and sets
// m_rcDIB and m_rcDest appropriately.
void CPNDlg::Prepare_Image(HDC hDC)
{
	if (m_hDIB != NULL)
	{
		LPSTR lpDIB = (LPSTR) ::GlobalLock((HGLOBAL) m_hDIB);
		int cxDIB = (int) ::DIBWidth(lpDIB);         // Size of DIB - x
		int cyDIB = (int) ::DIBHeight(lpDIB);        // Size of DIB - y
		::GlobalUnlock((HGLOBAL) m_hDIB);
		m_rcDIB.top = m_rcDIB.left = 0;
		m_rcDIB.right = cxDIB;
		m_rcDIB.bottom = cyDIB;
		// get size of printer page (in pixels)
		int cxPage = ::GetDeviceCaps(hDC, HORZRES);
		int max_x = cxPage;
		int cyPage = ::GetDeviceCaps(hDC, VERTRES);
		int max_y = cyPage;
		// get printer pixels per inch
		int cxInch = ::GetDeviceCaps(hDC, LOGPIXELSX);
		int cyInch = ::GetDeviceCaps(hDC, LOGPIXELSY);

		// Adjust printable area if necessary.
		switch (m_intScaleMode)
		{
		case 1:
			cxPage = (cxPage * 3) / 4;
			cyPage = (cyPage * 3) / 4;
			break;
		case 2:
			cxPage /= 2;
			cyPage /= 2;
		}
		m_rcDest.top = m_rcDest.left = 0;
		if (m_intScaleMode < 3)
		{
			//Best fit scaling.
			// Scale Y axis to match X axis.
			m_rcDest.bottom = (int)(((double)cyDIB * cxPage * cyInch)
					/ ((double)cxDIB * cxInch));
			// If image is too tall,
			if (m_rcDest.bottom > cyPage)
			{
				// Scale X axis to match Y axis.
				m_rcDest.right = (int)(((double)cxDIB * cyPage * cxInch)
					/ ((double)cyDIB * cyInch));
				m_rcDest.bottom = cyPage;
			}
			else
				// Otherwise, set image height.
				m_rcDest.right = cxPage;
		}
		else
		{	// No scaling -- shrink to fit page if necessary.
			m_rcDest.right = (m_rcDIB.right > cxPage) ? cxPage : m_rcDIB.right;
			m_rcDest.bottom = (m_rcDIB.bottom > cyPage) ? cyPage : m_rcDIB.bottom;
		}
		// Handle centering if necessary.
		if (m_bCenterImage)
		{
			int offset_x = 0, offset_y = 0;

			if (m_rcDest.right < max_x)
				offset_x = (max_x - m_rcDest.right) / 2;
			if (m_rcDest.bottom < max_y)
				offset_y = (max_y - m_rcDest.bottom) / 2;
			m_rcDest.OffsetRect(offset_x, offset_y);
		}
	}
}

// Retrieves printer and port name from common dialog
// and displays it in the Current Printer label.
void CPNDlg::Show_Printer_Name()
{
	CString printer, port;

	printer = m_pPrintDialog->GetDeviceName();	// Get the printer name.
	port = m_pPrintDialog->GetPortName();		// Get port name.
	printer += " on ";							// Put the strings together
	printer += port;
	m_lblCurrent.SetWindowText(printer);		// and set the label's caption.
}

// Displays a Print Setup dialog.
void CPNDlg::Do_Print_Setup()
{
	DWORD flags;

	flags = m_pPrintDialog->m_pd.Flags;					// Save common dialog flags.
	m_pPrintDialog->m_pd.Flags |= (PD_PRINTSETUP);		// Set "Print Setup" flag.
	m_bIsPrinting = TRUE;								// Set printing flag.
	Set_Tooltip_Text();									// Update task bar tooltip and dialog status display.
	m_lblStatus.SetWindowText("Waiting for Print Setup dialog");
	if (!m_bDialogActive)					// If dialog isn't already visible,
		ShowWindow(SW_RESTORE);				// make it visible but don't set our "active" flag.
	UpdateWindow();							// Force immediate window update.
	m_pPrintDialog->DoModal();				// Launch the print dialog.
	m_pPrintDialog->m_pd.Flags = flags;		// Restore original flags.
	m_bIsPrinting = FALSE;					// Clear printing flag.
	Show_Printer_Name();					// Update printer name display.
	Set_Tooltip_Text();						// Update tooltip text.
	m_lblStatus.SetWindowText("Ready");		// Update status text.
	if (!m_bDialogActive)					// If dialog wasn't visible before we started,
		Hide_Dialog();						// hide it.
}

// Prints the current DIB image.
// Note that this function is not called unless the DIB image
// and palette pointed to by m_hDIB and m_pPal are both valid.
void CPNDlg::Do_Print(BOOL ask_first)
{
	// Note: Must use static buffer for document name. Some printer drivers 
	// may modify the contents of this buffer. (For example, the Microsoft HTML driver
	// replaces the document name with the path to a temporary file it uses in creating
	// an HTML document from the printed output).
	// If we use a CString instead of a static buffer, the new contents
	// may overflow the CString and clobber the stack.
	char doc_name[MAX_PATH];
	HDC hDC = NULL;
	DOCINFO di;
	int copies;

	m_bIsPrinting = TRUE;			// Set printing status flag.
	if (ask_first)					// If necessary,
	{
		m_lblStatus.SetWindowText("Waiting for Print dialog...");	// Update status text.
		if (!m_bDialogActive)			// If dialog isn't already visible,
			ShowWindow(SW_RESTORE);		// make it visible but don't set internal flag.
		Set_Tooltip_Text();				// Update tooltip.
		UpdateWindow();					// Force immediate window update.
		if (m_pPrintDialog->DoModal() != IDOK)	// Launch print dialog.
		{										// If user cancelled,
			m_bIsPrinting = FALSE;				// Clear printing status flag.
			m_lblStatus.SetWindowText("Ready");	// Update status text.
			if (!m_bDialogActive)				// If dialog wasn't already active,
				Hide_Dialog();					// hide it again.
			return;								// Done.
		}
		Show_Printer_Name();			// Update printer name display.
	}
	m_bInPrintLoop = TRUE;				// Set printing flag.
	Hide_Controls(TRUE);				// Hide controls and display warning prompt.
	UpdateWindow();						// Force immediate window update.
	Set_Tooltip_Text();					// Update tooltip display.
	hDC = m_pPrintDialog->CreatePrinterDC();		// Create new printer device context.
	if (hDC)	// If printer DC is valid,
	{
		// Get copy count and make sure it's not zero.
		if ((copies = m_pPrintDialog->GetCopies()) == 0)
			copies = 1;
		// Set up document information for the spooler.
		lstrcpyn(doc_name, m_strCaption, MAX_PATH);
		lstrcat(doc_name, " DIB Image");
		di.cbSize = sizeof(DOCINFO);
		di.lpszDocName = doc_name;
		di.lpszOutput = NULL;
		di.lpszDatatype = NULL;
		di.fwType = 0;
		Prepare_Image(hDC);	// Scale image, set source and destination rectangles.
		BeginWaitCursor();	// Turn wait cursor on.
		while (copies)		// For each copy,
		{
			if (!::StartDoc(hDC, &di))	// Start document; display error and exit if call fails.
			{
				AfxMessageBox(IDS_DOC_INIT_FAIL, MB_ICONINFORMATION, 0);
				break;
			}
			RestoreWaitCursor();	// Turn wait cursor back on in case file selection dialog turned it off.
			::StartPage(hDC);		// Start a page and draw the image on it.
			::PaintDIB(hDC, &m_rcDest, m_hDIB, m_rcDIB, m_pPal);
			::EndPage(hDC);			// End the page.
			::EndDoc(hDC);			// End the document.
			--copies;				// Next copy.
		}
		DeleteDC(hDC);				// Delete printer device context.
		EndWaitCursor();			// Restore normal mouse pointer.
	}
	else	// Printer DC is invalid, so display error message.
	{
		AfxMessageBox(IDS_PDC_INIT_FAIL, MB_ICONINFORMATION, 0);
	}
	m_bInPrintLoop = FALSE;					// Reset status flags.
	m_bIsPrinting = FALSE;
	Hide_Controls(FALSE);					// Hide warning prompt and restore dialog controls.
	m_lblStatus.SetWindowText("Ready");		// Reset status text.
	if (m_bDialogActive)					// If dialog was already visible,
		Set_Tooltip_Text();					// just update the tooltip,
	else
		Hide_Dialog();						// otherwise hide the dialog (which also updates the tooltip).
}

// Prints the image currently stored in the clipboard.
void CPNDlg::Do_Print_Clipboard()
{
	// Error if clipboard data is not in proper format.
	// (Command button/Menu item may not be in synch with the clipboard!)
	if (!IsClipboardFormatAvailable(CF_DIB))
	{
		AfxMessageBox(IDS_BAD_FORMAT, MB_ICONINFORMATION, 0);
		return;
	}
	// Error if unable to copy the data.
	if (!Copy_Clipboard_Image())
	{
		AfxMessageBox(IDS_COPY_FAIL, MB_ICONINFORMATION, 0);
		return;
	}
	Do_Print(TRUE);		// Print the image, forcing display of the Print dialog.
	Discard_Image();	// Release image memory when printing is done.
}

// Discard DIB image and palette.
// Frees the memory used to hold the objects and NULLs their pointers.
void CPNDlg::Discard_Image()
{
	if (m_hDIB)								// If handle is not null,
	{
		::GlobalFree((HGLOBAL) m_hDIB);		// release the image memory
		m_hDIB = NULL;						// and NULL the pointer.
	}
	if (m_pPal)								// If palette exists,
	{
		delete m_pPal;						// delete it
		m_pPal = NULL;						// and NULL the pointer.
	}
}

// Decodes command-line parameters and sets options accordingly.
void CPNDlg::Parse_Command_Line()
{
	CString command_line;
	
	command_line = AfxGetApp()->m_lpCmdLine;	// Make copy of command line.
	command_line.TrimLeft();					// Trim leading and trailing spaces.
	command_line.TrimRight();
	if (command_line.IsEmpty())					// Nothing to do if blank.
		return;
	command_line.MakeUpper();					// Force uppercase
	// Look for keywords and set flags accordingly.
	if (command_line.Find("DISABLED") >= 0)
		m_bEnabled = FALSE;
	if (command_line.Find("NOPRINTDIALOG") >= 0)
		m_bAlways = FALSE;
	if (command_line.Find("NOCENTER") >= 0)
		m_bCenterImage = FALSE;
	if (command_line.Find("3/4PAGE") >= 0)
		m_intScaleMode = 1;
	else
	{
		if (command_line.Find("1/2PAGE") >= 0)
			m_intScaleMode = 2;
		else
		{
			if (command_line.Find("NOSCALING") >= 0)
				m_intScaleMode = 3;
		}
	}
}

// Hide/restore dialog controls and
// display/hide warning message
// before/after printing.
void CPNDlg::Hide_Controls(BOOL hide)
{
	CWnd* cur_wnd;

	cur_wnd = GetWindow(GW_CHILD);		// Find dialog's first child window.
	while (cur_wnd != NULL)				// For each child window,
	{
		cur_wnd->ShowWindow(hide ? SW_HIDE : SW_RESTORE);	// Hide or show it.
		cur_wnd = cur_wnd->GetNextWindow();					// Next window.
	}
	// Hide or show warning prompt.
	m_lblPrintWait.ShowWindow(hide ? SW_RESTORE : SW_HIDE);
}

// Handles activation from the taskbar icon.
// If main, Print, Print Setup, or Print to File dialog is visible,
// sets it as foreground window and returns TRUE.
// Otherwise, returns FALSE.
BOOL CPNDlg::Bring_To_Top()
{
	if (m_bInPrintLoop)		// Are we printing?
	{
		HWND target;		// See if the "Print To File" dialog is present.
							// Note that it is a child window of the desktop, not our main dialog.
		target = FindWindowEx(NULL, NULL, "#32770", "Print To File");
		if (target)			// If it is,
			::SetForegroundWindow(target);	// Set it as foreground
		else
			SetForegroundWindow();	// Otherwise, bring main dialog to foreground.
		return TRUE;
	}
	else	// Otherwise,
	{
		if (m_bIsPrinting)							// Waiting for print dialog?
			m_pPrintDialog->SetForegroundWindow();	// Bring it to foreground if so.
		else										// Otherwise,
			if (m_bDialogActive || m_bInPrintLoop)	// are we waiting for main dialog?
				SetForegroundWindow();				// Bring it to foreground.
	}
	return (m_bIsPrinting || m_bDialogActive);	// Return window status to caller.
}
