/* PLPPM.C
*/

#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "piclab.h"

static int subformat, maxval;

static int pgethc(int tf)	/* Get char from header */
{
	int c;

	while ((c = p_getc(tf)) == '#') {
		while (p_getc(tf) != '\n')
			;
	}
	return c;
}

static pgetint(int tf)
{
	int v, c;

	while (isspace(c = pgethc(tf)))
		;
	v = c - '0';

	while (isdigit(c = p_getc(tf))) {
		v = v * 10 + (c - '0');
	}
	return v;
}

/* Handler for LOAD command.
*/
int loadppm(int ac, argument *av)
{
	char *path;
	U8 FAR *tbuf, *fbuf, *lbuf[3];
	int tfile, i, r, np, line, h, w, x;
	struct _plane *p[3];
	U32 (*gpfunc)();

	if (ac < 2) {
		pl_printf("Syntax is LOAD <filename>\n");
		return 0;
	}
	if (ac > 2) pl_warn(1);

	histvalid = 0;
	if ((tbuf = (U8 *)talloc(BUFSIZE + 6144)) == NULL) return 2;
	path = makepath(picdir, av[1].cval, "PPM");
	pl_printf(reading, path);
	if ((tfile = p_open(path, tbuf, BUFSIZE, 6144, READ)) < 0) return 10;

	if (pgethc(tfile) != 'P') return 6;
	if ((subformat = pgethc(tfile) - '0') <= 1 || subformat == 4
		|| subformat  > 6) return 6;

	w = pgetint(tfile);
	h = pgetint(tfile);
	maxval = pgetint(tfile);

	if (subformat > 3 && maxval != 255) return 6;
	if (maxval != 255) subformat += 100;
	if (subformat == 2 || subformat == 5) np = 1; else np = 3;

	if ((r = begintransform()) != 0) return r;
	new->flags &= ~1;
	new->planes = (U8)np;
	new->width = w;
	new->height = h;

	for (i=0; i<np; ++i) {
		if ((p[i] = openplane(i, new, WRITE)) == NULL) return 3;
	}
	for (line = 0; line < h; ++line) {
		for (i=0; i<np; ++i) lbuf[i] = p[i]->linebuf;
		switch (subformat) {
			case 2:			/* ASCII Gray */
				for (x=0; x<w; ++x) {
					lbuf[0][x] = (U8)pgetint(tfile);
				}
				break;
			case 3:			/* ASCII RGB */
				for (x=0; x<w; ++x) {
					lbuf[0][x] = (U8)pgetint(tfile);
					lbuf[1][x] = (U8)pgetint(tfile);
					lbuf[2][x] = (U8)pgetint(tfile);
				}
				break;
			case 5:			/* Packed Gray */
				if (p_getline(tfile, w) != w) return 4;
				memcpy(lbuf[0], linebuf(tfile), w);
				break;
			case 6:			/* Packed RGB */
				if (p_getline(tfile, 3*w) != 3*w) return 4;
				fbuf = linebuf(tfile);
				for (x=0; x<w; ++x) {
					lbuf[0][x] = *fbuf++;
					lbuf[1][x] = *fbuf++;
					lbuf[2][x] = *fbuf++;
				}
				break;
			case 102:		/* Scaled ASCII Gray */
				for (x=0; x<w; ++x) {
					lbuf[0][x] = (U8)(pgetint(tfile) * 255L + 128L) / maxval;
				}
				break;
			case 103:		/* Scaled ASCII RGB */
				for (x=0; x<w; ++x) {
					lbuf[0][x] = (U8)(pgetint(tfile) * 255L + 128L) / maxval;
					lbuf[1][x] = (U8)(pgetint(tfile) * 255L + 128L) / maxval;
					lbuf[2][x] = (U8)(pgetint(tfile) * 255L + 128L) / maxval;
				}
				break;
		}
		for (i=0; i<np; ++i) {
			if (putline(p[i]) == 0) return 3;
		}
		pl_trace(line);
	}
	p_close(tfile);
	for (i=0; i<np; ++i) closeplane(p[i]);
	pl_printf(done);
	return 0;
}

/* Handler for SAVE command.
*/
int saveppm(int ac, argument *av)
{
	struct _plane *p[3];
	int i, tfile, np, ascii, line, x, h, w, s, linesize;
	U8 *fbuf, FAR *tbuf, *lbuf[3];
	char *path;

	if (transpend) return 8;
	if (new->planes == 0) return 7;
	if (new->flags & 2) {
		pl_printf("PPM format does not support mapped images.\n");
		return 0;
	}
    if ((new->flags & 1) == 1) {
        pl_printf("Can only save top-down raster to PPM file.  Issue\r\n");
        pl_printf("REVERSE command before continuing.\r\n");
        return 0;
    }
	ascii = 0;

	if (ac > 3) pl_warn(1);
	if (ac < 2) {
		pl_printf("Syntax is SAVE <filename> [ASCII]\n");
		return 0;
	} else {
		path = makepath(picdir, (*++av).cval, "PPM");
		if (ac > 2 && *(*++av).cval == 'A') ascii = 1;
	}
	pl_printf(writing, path);
	if ((tbuf = talloc(BUFSIZE + 6144)) == NULL) return 2;
	if ((tfile = p_open(path, tbuf, BUFSIZE, 6144, WRITE)) < 0) return 3;

	np = new->planes;
	for (i=0; i<np; ++i) {
		if ((p[i] = openplane(i, new, READ)) == NULL) return 3;
	}
	if (np == 1) subformat = 5; else subformat = 6;
	if (ascii) subformat -= 3;
	w = new->width;
	h = new->height;

	fbuf = linebuf(tfile);
	pl_sprintf(fbuf, "P%1d\n%d %d\n255\n", subformat, w, h);
	s = strlen(fbuf);
	if (p_putline(tfile, s) != s) return 3;

	for (line = 0; line < h; ++line) {
		linesize = 0;
		fbuf = linebuf(tfile);

		for (i=0; i<np; ++i) {
			if (getline(p[i]) == 0) return 4;
			lbuf[i] = p[i]->linebuf;
		}
		switch (subformat) {
			case 2:
				for (x=0; x<w-1; ++x) {
					pl_sprintf(fbuf, "%d ", lbuf[0][x]);
					s = strlen(fbuf);

					if ((linesize += s) > 70) {
						linesize = 0;
						fbuf[s-1] = '\n';
					}
					if (p_putline(tfile, s) != s) return 3;
					fbuf = linebuf(tfile);
				}
				pl_sprintf(fbuf, "%d\n\n", lbuf[0][w-1]);
				s = strlen(fbuf);
				if (p_putline(tfile, s) != s) return 3;
				break;
			case 3:
				for (x=0; x<w-1; ++x) {
					pl_sprintf(fbuf, "%d %d %d ",
						lbuf[0][x], lbuf[1][x], lbuf[2][x]);
					s = strlen(fbuf);

					if ((linesize += s) > 60) {
						linesize = 0;
						fbuf[s-1] = '\n';
					}
					if (p_putline(tfile, s) != s) return 3;
					fbuf = linebuf(tfile);
				}
				pl_sprintf(fbuf, "%d %d %d\n\n",
					lbuf[0][w-1], lbuf[1][w-1], lbuf[2][w-1]);
				s = strlen(fbuf);
				if (p_putline(tfile, s) != s) return 3;
				break;
			case 5:
				memcpy(fbuf, lbuf[0], w);
				if (p_putline(tfile, w) != w) return 3;
				break;
			case 6:
				for (x=0; x<w; ++x) {
					*fbuf++ = lbuf[0][x];
					*fbuf++ = lbuf[1][x];
					*fbuf++ = lbuf[2][x];
				}
				if (p_putline(tfile, 3*w) != 3*w) return 3;
				break;
			default:
				break;
		}
		pl_trace(line);
	}
	for (i=0; i<np; ++i) {
		closeplane(p[i]);
	}
	pl_printf(done);
	p_close(tfile);

	return 0;
}
