/* PLDITHER.C : General-purpose dithering code for PICLAB.	Called by
** printpic(), savegif(), and a few other places.
*/

#include <stdlib.h>
#include <memory.h>

#include "piclab.h"
#include "sierra"

/* The ditherline() routine takes an array of pointers to RGB values and
** produces an array of color indices by error-distribution dithering with
** the Sierra filter.  The pointers rv, gv, and bv must be set to point to
** arrays containing the red, green, and blue values of the available
** palette.  Table2 points to an array of 4096 indices, each entry indicat-
** ing which of the available colors is closest to the center of the subcube
** of RGB-space given by the formula ((r&0xF0)<<4)+(g&0xF0)+((b&0xF0)>>4).
** This method is used to avoid time-consuming color comparisons inside the
** dithering loop.	Pre-made tables are provided for the HP PaintJet printer
** in the file PJTBLS.	Other tables are created on-the-fly.
**
** To use this routine, set the pointers as mentioned above, set the flag
** variable startdither to 1, then call this routine for each line.
*/
void ditherline(U16 pwidth, U8 *rgb[3], U8 *indices)
{
	static S16 *this[3], *next[3], *next2[3], *temp[3];
	static U8 *line, *up, c;
	static int p, x, c1[3], e[3], e1[3], e2[3];
	int w2, t3, t4, t5;
    register int d;
	S8 *sp;

	if (startdither) {			/* Here for the first time, intialize	*/
		startdither = 0;
		w2 = 2 * pwidth + 8;
		for (p=0; p<3; ++p) {
			this[p] = (S16 *)talloc(w2) + 2;	memset(this[p]-2, 0, w2);
			next[p] = (S16 *)talloc(w2) + 2;	memset(next[p]-2, 0, w2);
			next2[p] = (S16 *)talloc(w2) + 2;	memset(next2[p]-2, 0, w2);
		}
		if ((line = (U8 *)talloc(pwidth)) == NULL) return;
		memset(line, 0, pwidth);
	}

    e1[0] = e1[1] = e1[2] =
    e2[0] = e2[1] = e2[2] = 0;
	up = line;
	for (p=0; p<3; ++p) {
		next[p][0] = next[p][1] = next2[p][0] = next2[p][1] = 0;
	}
	for (x=0; x<pwidth; ++x) {
		for (p=0; p<3; ++p) {
			c1[p] = rgb[p][x] + this[p][x] + e1[p];
			if (c1[p] < 0) c1[p] = 0;
			if (c1[p] > 255) c1[p] = 255;
		}

		c = table2[ ((c1[0]&0xF0)<<4) + (c1[1]&0xF0) + ((c1[2]&0xF0)>>4) ];
		*up++ = c;
		e[0] = c1[0] - rv[c];
		e[1] = c1[1] - gv[c];
		e[2] = c1[2] - bv[c];

		for (p=0; p<3; ++p) if (e[p] != 0) {
			d = e[p]+255;	sp = sierra[d];
			t3 = *sp++; 	t4 = *sp++; 	t5 = *sp++;
			e1[p] = e2[p] + *sp++;
			e2[p] = t4;
			next[p][x-2] += t5;
			next[p][x-1] += t3; 	next2[p][x-1] += t5;
			next[p][x] += *sp;		next2[p][x] += t4;
			next[p][x+1] += t3; 	next2[p][x+1] = t5;
			next[p][x+2] = t5;
		}
	}
	for (p=0; p<3; ++p) {
		temp[p] = this[p]; this[p] = next[p];
		next[p] = next2[p]; next2[p] = temp[p];
	}
	memcpy(indices, line, pwidth);
}

static U8 masks[] = { 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };
#define setbit(x,p) (p)[(x)>>3]|=masks[(x)&7]

/* Bitmapline() takes an array of gray values and returns two arrays of
** bits, appropriate for printing on a LaserJet or storing in an IMG file.
** The bitmap pointers passed must both point to an area large enough for
** (pwidth * 2) bits.
*/
void bitmapline(U16 pwidth, U8 *gray, U8 *bitmap[2])
{
	/* The arrays er1..er3 hold error values for the current line, next line
	** and line after next.  The pointers this, next, and next2 are rotated
	** on each pass through this routine so that arrays need not be copied.
	*/

	static S16 *this, *next, *next2, *temp;
	static U8 *thisline, *lastline, *tp;
	 register U8 *bp, *lp;
	static int x, c1, e, e1, e2;
	int d, t3, t4, t5;
    S8 *sp;

	if (startdither) {
		startdither = 0;
		this = (S16 *)talloc(4808) + 2;		memset(this-2, 0, 4808);
		next = (S16 *)talloc(4808) + 2;		memset(next-2, 0, 4808);
		next2 = (S16 *)talloc(4808) + 2;	memset(next2-2, 0, 4808);
		thisline = (U8 *)talloc(2404) + 2;	memset(thisline-2, 0, 2404);
		lastline = (U8 *)talloc(2404) + 2;	memset(lastline-2, 0, 2404);
	}

	memset(bitmap[0], 0, (2*pwidth+7)>>3);
	memset(bitmap[1], 0, (2*pwidth+7)>>3);

	e1 = e2 = 0;
	next[0] = next[1] = next2[0] = next2[1] = 0;
	for (x=0; x < 2*pwidth; ++x) {
		c1 = gray[x >> 1] + this[x] + e1;
		if (c1 < 0) c1 = 0;
		if (c1 > 255) c1 = 255;
		if (c1 > 127) {
			e = c1 - 255;
			if (lastline[x]) e += bleed;
			if (thisline[x-1]) e += bleed;
			thisline[x] = 0;
		} else {
			e = c1;
			thisline[x] = 1;
		}

		d = e+255;			sp = sierra[d];
		t3 = *sp++; 		t4 = *sp++; 	t5 = *sp++;
		e1 = e2 + *sp++;	e2 = t4;

		next[x-2] += t5;
		next[x-1] += t3;	next2[x-1] += t5;
		next[x] += *sp; 	next2[x] += t4;
		next[x+1] += t3;	next2[x+1] = t5;
		next[x+2] = t5;
	}
	bp = bitmap[0];
	if ((new->flags & 1) == 1) {
		lp = thisline + 2*pwidth;
		for (x=0; x < 2*pwidth; ++x) if (*--lp) setbit(x,bp);
	} else {
		lp = thisline;
		for (x=0; x < 2*pwidth; ++x) if (*lp++) setbit(x,bp);
	}
	temp = this; this = next; next = next2; next2 = temp;
	tp = thisline; thisline = lastline; lastline = tp;

	e2 = 0;
	next[0] = next[1] = next2[0] = next2[1] = 0;
	for (x=0; x < 2*pwidth; ++x) {
		c1 = gray[x >> 1] + this[x] + e1;
		if (c1 < 0) c1 = 0;
		if (c1 > 255) c1 = 255;
		if (c1 > 127) {
			e = c1 - 255;
			if (lastline[x]) e += bleed;
			if (thisline[x-1]) e += bleed;
			thisline[x] = 0;
		} else {
			e = c1;
			thisline[x] = 1;
		}

		d = e+255;			sp = sierra[d];
		t3 = *sp++; 		t4 = *sp++; 	t5 = *sp++;
		e1 = e2 + *sp++;	e2 = t4;

		next[x-2] += t5;
		next[x-1] += t3;	next2[x-1] += t5;
		next[x] += *sp; 	next2[x] += t4;
		next[x+1] += t3;	next2[x+1] = t5;
		next[x+2] = t5;
	}
	bp = bitmap[1];
	if ((new->flags & 1) == 1) {
		lp = thisline + 2*pwidth;
		for (x=0; x < 2*pwidth; ++x) if (*--lp) setbit(x,bp);
	} else {
		lp = thisline;
		for (x=0; x < 2*pwidth; ++x) if (*lp++) setbit(x,bp);
	}
	temp = this; this = next; next = next2; next2 = temp;
	tp = thisline; thisline = lastline; lastline = tp;
}
