/*
 * Spline routine for equally spaced points
 * A. Schiffler, 1995, andreas@karlsberg.usask.ca
 *
 * It was modified for pmod by Jaroslav Kysela
 * Rewrote 'oversample' routines - A.S., 16 Sept. 1995 - unchecked
 * Checked 'oversample' routines - J.K., 21 Sept. 1995 - debugged
 *
 */

#include <math.h>
#include <malloc.h>
#include <string.h>
#include "pmod.h"

/*
 *  defines
 */

#define SFLOAT		float	/* data type for calculations */
#define SNUMPOINTS	64	/* number of data points to process at once */
#define COPYPOINTS	2

/*
 *  global variables
 */

static SFLOAT spline_y[SNUMPOINTS];	/* spline input */

static SFLOAT spline_b[SNUMPOINTS];	/* spline coefficients */
static SFLOAT spline_c[SNUMPOINTS];
static SFLOAT spline_d[SNUMPOINTS];

/*
 *  calculate the coefficients
 *  expects y values in spline_y[] up to and including 'numpoints'
 */
 
static void createSpline( int numpoints )
{
  int index;
  SFLOAT l[ SNUMPOINTS ];
  SFLOAT mu[ SNUMPOINTS ];
  SFLOAT z[ SNUMPOINTS ];
 
  /* calculate c */
  l[ 0 ] = mu[ 0 ] = z[ 0 ] = 0.0;
  for ( index = 1; index < numpoints; index++ ) 
    {
      l[ index ] = 4.0-mu[index-1];
      mu[ index ] = 1.0/l[index];
      z[ index ] = (3.0*(spline_y[index+1]-2.0*spline_y[index]+spline_y[index-1])
      		   -z[index-1])/l[index];
   }
  spline_c[ numpoints ] = 0.0;
  for ( index = (numpoints - 1); index >= 0; index-- )
    spline_c[ index ] = z[index]-mu[index]*spline_c[index+1];
  /* calculate b and d */
  for ( index = (numpoints - 1); index >= 0; index-- ) 
    {
      spline_b[index] = (spline_y[index+1]-spline_y[index])
   		        -(spline_c[index+1]+2.0*spline_c[index])/3.0;
      spline_d[index] = (spline_c[index+1]-spline_c[index])/3.0;	
    }
}

/*
 *  calculate spline interpolator at position 'xpt'
 */

static SFLOAT spline( SFLOAT xpt ) 
{
  int loc;
  SFLOAT cx;
 
  loc=(int)xpt;
  cx=xpt-(SFLOAT)loc;
  return( ((spline_d[loc]*cx+spline_c[loc])*cx+spline_b[loc])*cx+spline_y[loc]);
}

/*
 *
 */
 
void oversamplingS8( signed char *ptr, long length, float factor )
{
  long newlength;
  int i,j,c,first;	
  signed char *tmp,*tmp1,*ptr1;
  float position,step,maximum;
  SFLOAT res;
  
  position=0.0;  /* x-position for interpolation */
  step=1.0/factor;  /* x-step for interpolation */
  newlength=(long)(factor*(float)length);      /* New length of sample */
  if ( ( tmp = malloc( newlength ) ) == NULL )
    fatal( "oversamplingS8: malloc error\n" ); /* Allocate new memory */
  tmp1=tmp;
  ptr1=ptr;
  i=0; /* Sample byte position */
  first=1;
  while (i<length) {
   if ( !first )
     {
       for ( c = 0; c < COPYPOINTS; c++ )
         spline_y[ c ] = spline_y[ ( SNUMPOINTS - 2 * COPYPOINTS ) + c ];
     }
   j = first ? 0 : COPYPOINTS;
   position += (float)j;
   while ((i<length) && (j<SNUMPOINTS-COPYPOINTS)) {
    spline_y[ j++ ] = *(ptr1++);  /* Load spline array */
    i++;
   }
   for ( c = 0; (c < COPYPOINTS) && (i + c < length); c++ ) {
    spline_y[ j++ ] = *(ptr1 + c);
   }
   /* Interpolate */
   createSpline (j);
   if ( j > SNUMPOINTS - COPYPOINTS ) j = SNUMPOINTS - COPYPOINTS;
   maximum=(float)j; /* Maximum floating point number for this chunk */
   while (position<maximum) {
    /* Calculate interpolation value */
    res = spline( position );
    if ( res < -128 ) *(tmp1++) = -128; else
    if ( res > 127 ) *(tmp1++) = 127; else
    *(tmp1++) = rint( res );
    position += step; /* Update position */
   }
   position -= maximum; /* Subtract last maximum */
   first = 0;
  }
  memmove( ptr, tmp, newlength );
  free( tmp );
}

void oversamplingS16( signed short *ptr, long length, float factor )
{
  long newlength;
  int i,j,c,first;	
  signed short *tmp,*tmp1,*ptr1;
  float position,step,maximum;
  SFLOAT res;
  
  position=0.0;  /* x-position for interpolation */
  step=1.0/factor;  /* x-step for interpolation */
  newlength=(long)(factor*(float)length)*(sizeof(short));  
  			/* New length of sample */
  if ( ( tmp = malloc( newlength ) ) == NULL )
    fatal( "oversamplingS16: malloc error\n" ); /* Allocate new memory */
  tmp1=tmp;
  ptr1=ptr;
  i=0; /* Sample byte position */
  first=1;
  while (i<length) {
   if ( !first )
     {
       for ( c = 0; c < COPYPOINTS; c++ )
         spline_y[ c ] = spline_y[ ( SNUMPOINTS - 2 * COPYPOINTS ) + c ];
     }
   j = first ? 0 : COPYPOINTS;
   position += (float)j;
   while ((i<length) && (j<SNUMPOINTS-COPYPOINTS)) {
    spline_y[ j++ ] = *(ptr1++);  /* Load spline array */
    i++;
   }
   for ( c = 0; (c < COPYPOINTS) && (i + c < length); c++ ) {
    spline_y[ j++ ] = *(ptr1 + c);
   }
   /* Interpolate */
   createSpline (j);
   if ( j > SNUMPOINTS - COPYPOINTS ) j = SNUMPOINTS - COPYPOINTS;
   maximum=(float)j; /* Maximum floating point number for this chunk */
   while (position<maximum) {
    /* Calculate interpolation value */
    res = spline( position );
    if ( res < -32768 ) *(tmp1++) = -32768; else
    if ( res > 32767 ) *(tmp1++) = 32767; else
    *(tmp1++) = rint( res );
    position += step; /* Update position */
   }
   position -= maximum; /* Subtract last maximum */
   first = 0;
  }
  memmove( ptr, tmp, newlength );
  free( tmp );
}
