/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Native synthesizer support for GF1 chip
 */

#include <asm/byteorder.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  defines
 */
 
#define GUS_WAVE_EXPANDED	0x01000000
#define GUS_WAVE_TOTAL_EXPANDED 0x02000000
#define GUS_WAVE_EXPAND		0xf0000000
#define GUS_WAVE_EXPAND_S8	0x10000000
#define GUS_WAVE_EXPAND_S16	0x20000000
 
/*
 *  structures
 */

struct gus_sample_t {
  int smpno;		/* real sample number in GUS memory or -1 */
  unsigned int type;	/* sample type - GUS_WAVE_XXXX constants */
  int size;		/* sample size */
  int lstart;		/* sample loop start */
  int lend;		/* sample loop end */
  float factor;		/* expand factor */
};

/*
 *  variables
 */

static int gus_handle = -1;
static int gus_freq_type;
static unsigned char *gus_queue_buffer = NULL;
static size_t gus_queue_buffer_size;
static unsigned char *gus_queue_ptr;
static size_t gus_queue_len;
static int gus_info_flag;
static struct GUS_STRU_INFO gus_info_data;
static int gus_reg_flag;
static int gus_reg_flag_download;
static struct gus_sample_t *gus_samples;
static int *gus_samples_index;
static int gus_active_voices;
static int gus_active_voice;
static int gus_active_sample[ 32 ];

/*
 *  local functions
 */

static void gus_reset_variables( void )
{
  int i;

  gus_info_flag = 0;
  gus_queue_ptr = gus_queue_buffer;
  gus_queue_len = gus_queue_buffer_size;
  gus_reg_flag = gus_reg_flag_download = 0;
  for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
    gus_samples[ i ].smpno = -1;
  gus_active_voice = -1;
  for ( i = 0; i < 32; i++ )
    gus_active_sample[ i ] = -1;
}

/*
 *  EXPORTED FUNCTIONS
 */

/*
 *  ----------------- open / close
 */

int gus_open( size_t queue_buffer_size )
{
  int handle;
  int i;

  errno = 0;
  
  if ( gus_handle >= 0 )
    {
      gus_dprintf( "gus_open: device already open" );
      return -1;
    }
  if ( queue_buffer_size < 512 || queue_buffer_size > 1024 * 1024 )
    {
      gus_dprintf( "gus_open: queue buffer size isn't valid" );
      return -1;
    }
    
  if ( ( handle = open( "/dev/gus", O_WRONLY ) ) < 0 )
    {
      gus_dprintf( "gus_open: file open error %i", errno );
      return -1;
    }
  if ( ( i = ioctl( handle, GUS_IOCTL_VERSION ) ) < 0 )
    {
      gus_dprintf( "gusOpen: gus ioctl - VERSION" );
      close( handle );
      return -1;
    }
  
  if ( ( i >> 8 ) != ( GUS_SYNTH_VERSION >> 8 ) ||
       ( GUS_SYNTH_VERSION & 0xff ) > ( i & 0xff ) )
    {
      gus_dprintf( "gus_open: uncompatible version of gus driver" );
      gus_dprintf( "gus_open: required version of synth protocol %i.%02i (driver reports %i.%02i)",
		 	      GUS_SYNTH_VERSION >> 8, GUS_SYNTH_VERSION & 0xff,
			  i >> 8, i & 0xff );
      close( handle );
      return -1;
    }
 
  /* init variables */

  if ( ( gus_queue_buffer = (unsigned char *)malloc( queue_buffer_size ) ) == NULL ||
       ( gus_samples = (struct gus_sample_t *)malloc( sizeof( struct gus_sample_t ) * GUS_MAX_SAMPLES ) ) == NULL ||
       ( gus_samples_index = (int *)malloc( sizeof( int ) * GUS_MAX_SAMPLES ) ) == NULL )
    {
      gus_dprintf( "gus_open: allocation error" );
      close( handle );
      free( gus_queue_buffer );
      free( gus_samples );
      free( gus_samples_index );
      return -1;
    }
  
  gus_handle = handle;
  gus_queue_buffer_size = queue_buffer_size;
  gus_queue_ptr = gus_queue_buffer;
  gus_queue_len = gus_queue_buffer_size;
  gus_reset_variables();

  /* default setup */
  
  gus_setup_volume_type( GUS_VOL_TYPE_LINEAR );
  gus_setup_frequency_type( GUS_FREQ_TYPE_HZ );
  
  return 0;
}

int gus_close( void )
{
  int res;
  
  res = close( gus_handle ) < 0 ? -1 : 0;
  gus_handle = -1;
  free( gus_queue_buffer );
  free( gus_samples );
  free( gus_samples_index );
  return res;
}

/*
 *  ----------------- info
 */

int gus_info( struct GUS_STRU_INFO *info )
{
  int null;

  null = info == NULL;
  if ( !gus_info_flag )
    {
      if ( null ) info = &gus_info_data;
      if ( ioctl( gus_handle, GUS_IOCTL_INFO, info ) < 0 )
        {
          gus_dprintf( "gus_read_info: gus ioctl - INFO" );
          return -1;
        }
      if ( !null )
        memcpy( &gus_info_data, info, sizeof( struct GUS_STRU_INFO ) );
      gus_info_flag = 1;
    }
   else
    if ( !null )
      memcpy( info, &gus_info_data, sizeof( struct GUS_STRU_INFO ) );
  return 0;
}

int gus_info_port( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.port;
}

int gus_info_irq( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.irq;
}

int gus_info_dma1( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.dma1;
}

int gus_info_dma2( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.dma2;
}

int gus_info_version( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.version;
}

int gus_info_noise_channel( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.noise_channel;
}

/*
 *  ----------------- setup
 */

int gus_setup_volume_type( int type )
{
  if ( ioctl( gus_handle, GUS_IOCTL_VOL_TYPE, type ) < 0 )
    {
      gus_dprintf( "gus_setup_volume_type: gus ioctl - VOL_TYPE\n" );
      return -1;
    }
  return 0;
}

int gus_setup_frequency_type( int type )
{
  if ( ioctl( gus_handle, GUS_IOCTL_FREQ_TYPE, type ) < 0 )
    {
      gus_dprintf( "gus_setup_frequency_type: gus ioctl - FREQ_TYPE\n" );
      return -1;
    }
  gus_freq_type = type;
  return 0;
}

int gus_setup_set_auto_ramp( int speed )
{
  if ( ioctl( gus_handle, GUS_IOCTL_SET_AUTO_RAMP, speed ) < 0 )
    {
      gus_dprintf( "gus_setup_set_auto_ramp: gus ioctl - SET_AUTO_RAMP\n" );
      return -1;
    }
  return 0;
}

int gus_setup_get_auto_ramp( void )
{
  int res;

  if ( ( res = ioctl( gus_handle, GUS_IOCTL_GET_AUTO_RAMP ) ) < 0 )
    {
      gus_dprintf( "gus_setup_get_auto_ramp: gus ioctl - GET_AUTO_RAMP\n" );
      return -1;
    }
  return res;
}

int gus_setup_set_smooth_pan( int speed )
{
  if ( ioctl( gus_handle, GUS_IOCTL_SET_SPAN, speed ) < 0 )
    {
      gus_dprintf( "gus_setup_set_smooth_pan: gus ioctl - SET_SPAN\n" );
      return -1;
    }
  return 0;
}

int gus_setup_get_smooth_pan( void )
{
  int res;

  if ( ( res = ioctl( gus_handle, GUS_IOCTL_GET_SPAN ) ) < 0 )
    {
      gus_dprintf( "gus_setup_get_smooth_pan: gus ioctl - GET_SPAN\n" );
      return -1;
    }
  return res;
}

/*
 *  ----------------- memory
 */

int gus_memory_size( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.memory_size;
}

int gus_memory_free( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.memory_free;
}

int gus_memory_max_bank_free( void )
{
  if ( gus_info( NULL ) ) return -1;
  return gus_info_data.max_bank_free;
}

/*
 *  ----------------- reset
 */

int gus_reset( short voices )
{
  gus_reset_variables();
  if ( ioctl( gus_handle, GUS_IOCTL_RESET, gus_active_voices = voices ) < 0 )
    {
      gus_dprintf( "gus_reset: gus ioctl - RESET\n" );
      return -1;
    }
  return 0;
}

int gus_reset_engine_only( void )
{
  gus_queue_ptr = gus_queue_buffer;
  gus_queue_len = gus_queue_buffer_size;
  if ( ioctl( gus_handle, GUS_IOCTL_RESET0 ) < 0 )
    {
      gus_dprintf( "gus_reset_engine_only: gus ioctl - RESET0\n" );
      return -1;
    }
  return 0;
}

/*
 *  ----------------- timer
 */

int gus_timer_start( int time )
{
  if ( ioctl( gus_handle, GUS_IOCTL_START_TIMER, time ) < 0 )
    {
      gus_dprintf( "gus_timer_start: gus ioctl - START_TIMER\n" );
      return -1;
    }
  return 0;
}

int gus_timer_stop( void )
{
  if ( ioctl( gus_handle, GUS_IOCTL_STOP_TIMER ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - STOP_TIMER\n" );
      return -1;
    }
  return 0;
}

int gus_timer_start1( int time )
{
  if ( ioctl( gus_handle, GUS_IOCTL_START_TIMER1, time ) < 0 )
    {
      gus_dprintf( "gus_timer_start1: gus ioctl - START_TIMER1\n" );
      return -1;
    }
  return 0;
}

int gus_timer_stop1( void )
{
  if ( ioctl( gus_handle, GUS_IOCTL_STOP_TIMER1 ) < 0 )
    {
      gus_dprintf( "gus_timer_stop1: gus ioctl - STOP_TIMER1\n" );
      return -1;
    }
  return 0;
}

int gus_timer_start2( int time )
{
  if ( ioctl( gus_handle, GUS_IOCTL_START_TIMER2, time ) < 0 )
    {
      gus_dprintf( "gus_timer_start2: gus ioctl - START_TIMER2\n" );
      return -1;
    }
  return 0;
}

int gus_timer_stop2( void )
{
  if ( ioctl( gus_handle, GUS_IOCTL_STOP_TIMER2 ) < 0 )
    {
      gus_dprintf( "gus_timer_stop2: gus ioctl - STOP_TIMER2\n" );
      return -1;
    }
  return 0;
}

/*
 *  ----------------- sample
 */

int gus_sample_download( struct GUS_STRU_DOWNLOAD *download )
{
  gus_info_flag = 0;
  if ( gus_reg_flag ) return -EBUSY;	/* you cann't combine this with reg fcns */
  if ( ioctl( gus_handle, GUS_IOCTL_DOWNLOAD, download ) < 0 )
    {
      gus_dprintf( "gus_sample_download: gus ioctl - DOWNLOAD\n" );
      return -1;
    }
  return 0;
}

int gus_sample_download_reset( void )
{
  gus_info_flag = 0;
  if ( gus_reg_flag ) return -EBUSY;	/* you cann't combine this with reg fcns */
  if ( ioctl( gus_handle, GUS_IOCTL_DOWNLOAD_RESET ) < 0 )
    {
      gus_dprintf( "gus_sample_download_reset: gus ioctl - DOWNLOAD_RESET\n" );
      return -1;
    }
  return 0;
}

int gus_sample_download_test( struct GUS_STRU_DOWNLOAD *download )
{
  gus_info_flag = 0;
  if ( gus_reg_flag ) return -EBUSY;	/* you cann't combine this with reg fcns */
  if ( ioctl( gus_handle, GUS_IOCTL_DOWNLOAD_TEST, download ) < 0 )
    {
#if 0
      gus_dprintf( "gus_sample_download_test: gus ioctl - DOWNLOAD_TEST\n" );
#endif
      return -1;
    }
  return 0;
}

int gus_sample_download_test1( struct GUS_STRU_DOWNLOAD *download )
{
  if ( gus_reg_flag ) return -EBUSY;	/* you cann't combine this with reg fcns */
  if ( ioctl( gus_handle, GUS_IOCTL_DOWNLOAD_TEST1, download ) < 0 )
    {
      gus_dprintf( "gus_sample_download_test1: gus ioctl - DOWNLOAD_TEST1\n" );
      return -1;
    }
  return 0;
}

int gus_sample_reg( unsigned short number, unsigned int type,
		    int size, int lstart, int lend )
{
  if ( gus_reg_flag_download ) return -EBUSY;	/* too late */
  if ( number >= GUS_MAX_SAMPLES ) return -EINVAL;
  if ( gus_samples[ number ].smpno >= 0 ) return -EBUSY;
  gus_reg_flag = 1;
  gus_samples[ number ].smpno = number;
  gus_samples[ number ].type = type;
  gus_samples[ number ].size = size;
  gus_samples[ number ].lstart = lstart;
  gus_samples[ number ].lend = lend;
  gus_samples[ number ].factor = (float)1.0;
  return 0;
}

int gus_sample_reg_size( void )
{
  int i, res = 0;

  for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
    if ( gus_samples[ i ].smpno >= 0 )
      res += ( gus_samples[ i ].size + 31 ) & ~31;
  return res;
}

static int sample_compar( const void *ptr1, const void *ptr2 )
{
  if ( ((struct gus_sample_t *)ptr1) -> smpno == -1 )
    return ((struct gus_sample_t *)ptr2) -> smpno == -1 ? 0 : 1;
  if ( ((struct gus_sample_t *)ptr2) -> smpno == -1 )
    return -1;
  if ( ((struct gus_sample_t *)ptr1) -> size == ((struct gus_sample_t *)ptr2) -> size )
    return 0;
  return ((struct gus_sample_t *)ptr1) -> size < ((struct gus_sample_t *)ptr2) -> size ? 1 : -1;
}

static int gus_sample_process( unsigned short number, unsigned char *ptrx )
{
#define ptr16	( (unsigned short *)ptr )
  unsigned int type;
  int size, lend, lstart;
  struct gus_sample_t *smp;
  struct GUS_STRU_DOWNLOAD gs;
  unsigned char *ptr;
  
  ptr = ptrx;
  smp = &gus_samples[ number ];
  type = smp -> type;
  size = smp -> size;
  lstart = smp -> lstart;
  lend = smp -> lend;
#if 0
  gus_dprintf( "size = 0x%x", size );
#endif
  if ( type & GUS_WAVE_DELTA )
    gus_convert_delta( type, ptr, ptr, smp -> size );
  if ( !( type & GUS_WAVE_NO_EXPAND ) && smp -> factor != (float)1.0 ) 
    {
      size_t x;
    
      x = ( type & GUS_WAVE_16BIT ) ? ( size << 1 ) : size;
      x = ( smp -> factor * (float)x ) + 16;
      if ( ( ptr = malloc( x ) ) == NULL )
        {
          ptr = ptrx;
          smp -> factor = (float)1.0;
        }
       else
        { 
#if 0
          gus_dprintf( "new size = %i (0x%x), factor %.4f", x, x, smp -> factor );
#endif
          if ( ( type & GUS_WAVE_EXPAND ) == GUS_WAVE_EXPAND_S16 &&
               !( type & GUS_WAVE_16BIT ) && smp -> factor >= (float)2.0 )
            {
              smp -> factor /= 2;
              gus_convert_spline_oversampling_8bit_to_16bit(
              			(signed short *)ptr, (signed char *)ptrx,
              			size, smp -> factor,
              			type & GUS_WAVE_UNSIGNED );
              smp -> type |= GUS_WAVE_16BIT;
            }
           else
            {
              if ( type & GUS_WAVE_16BIT )
                gus_convert_spline_oversampling_16bit(
	           		(signed short *)ptr, (signed short *)ptrx,
	           		size, smp -> factor, 
	           		type & GUS_WAVE_UNSIGNED );
               else
                gus_convert_spline_oversampling_8bit( 
                		(signed char *)ptr, (signed char *)ptrx,
                		size, smp -> factor,
    				type & GUS_WAVE_UNSIGNED );
            }
          smp -> type &= ~GUS_WAVE_UNSIGNED;
          smp -> type |= GUS_WAVE_EXPANDED;
          type = smp -> type;
          size = (float)size * smp -> factor;
          lstart= (float)lstart * smp -> factor;
          lend = (float)lend * smp -> factor;
          if ( type & GUS_WAVE_16BIT )
            {
              size &= ~1;
              lstart &= ~1;
              lend &= ~1;
            }
        }
    }
  if ( lend > size ) smp -> lend = size;
  if ( type & GUS_WAVE_16BIT )
    {
      unsigned short x;
    
      x = ptr16[ size - 1 ];
      ptr16[ size + 0 ] = x;
      ptr16[ size + 1 ] = x;
    }
   else
    {
      unsigned char x;
      
      x = ptr[ size - 1 ];
      ptr[ size + 0 ] = x;
      ptr[ size + 1 ] = x;
    }
  if ( !( type & GUS_WAVE_NO_LOOP_ADAPT ) && ( type & GUS_WAVE_LOOP ) )
    if ( type & GUS_WAVE_16BIT )
      {
        ptr16[ lend ] = ptr16[ lend + 1 ] = ptr16[ lstart ];
        if ( lend - lstart > 100 )
          {
            if ( smp -> type & GUS_WAVE_UNSIGNED )
              {
                ptr16[ lend - 1 ] = ( ptr16[ lend - 2 ] + ptr16[ lend ] ) >> 1;
                if ( lend - lstart > 200 )
                  ptr16[ lend - 2 ] = ( ptr16[ lend - 3 ] + ptr16[ lend - 1 ] ) >> 1;
              }
             else
              {
                (signed short)ptr16[ lend - 1 ] =
                  ( (signed short)ptr16[ lend - 2 ] + (signed short)ptr16[ lend ] ) / 2;
                if ( lend - lstart > 200 )
                  (signed short)ptr16[ lend - 2 ] =
                    ( (signed short)ptr16[ lend - 3 ] + (signed short)ptr16[ lend - 1 ] ) / 2;
              }
          }
        if ( size == lend ) size++;
      }
     else
      {
        ptr[ lend ] = ptr[ lend + 1 ] = ptr[ lstart ];
        if ( lend - lstart > 100 )
          {
            if ( smp -> type & GUS_WAVE_UNSIGNED )
              {
                ptr[ lend - 1 ] = ( ptr[ lend - 2 ] + ptr[ lend ] ) >> 1;
                if ( lend - lstart > 200 )
                  ptr[ lend - 2 ] = ( ptr[ lend - 3 ] + ptr[ lend - 1 ] ) >> 1;
              }
             else
              {
                (signed char)ptr[ lend - 1 ] =
                  ( (signed char)ptr[ lend - 2 ] + (signed char)ptr[ lend ] ) / 2;
                if ( lend - lstart > 200 )
                  (signed char)ptr[ lend - 2 ] =
                    ( (signed char)ptr[ lend - 3 ] + (signed char)ptr[ lend - 1 ] ) / 2;
              }
          }
        if ( size == lend ) size++;
      }
  if ( type & GUS_WAVE_16BIT )
    {
      size <<= 1;
      lstart <<= 1;
      lend <<= 1;
    }
  gs.voice_cntrl_reg = (unsigned char)type;
  gs.length = size;
  gs.loop_start = lstart;
  gs.loop_end = lend;
  gs.sample_ptr = ptr;
  gus_reg_flag = 0;
  size = gus_sample_download( &gs );
  gus_reg_flag = 1;
  if ( ptr != ptrx ) free( ptr );
  return size < 0 ? -errno : 0;
#undef ptr16
}

int gus_sample_reg_download(
	int (*read_fcn)( unsigned short number, unsigned char *ptr, int size ),
	void (*error_fcn)( unsigned short number, int error ) )
{
  int i, err;
  unsigned char *ptr;

  gus_reg_flag_download = 1;
  qsort( gus_samples, GUS_MAX_SAMPLES, sizeof( struct gus_sample_t ), sample_compar );
  for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
    {
      size_t x;
      struct gus_sample_t *smp;
      
      smp = &gus_samples[ i ];
      if ( smp -> smpno < 0 ) continue;
      err = -ENOMEM;
      x = ( smp -> type & GUS_WAVE_16BIT ) ? ( smp -> size << 1 ) : smp -> size;
      if ( ( ptr = (unsigned char *)malloc( x + 16 ) ) != NULL )
        {
          err = -EIO;
          if ( !read_fcn( smp -> smpno, ptr, x ) )
            err = gus_sample_process( i, ptr );
        }
      if ( err )
        {
          smp -> smpno = -1;		/* error */
          error_fcn( i, err );
        }
      free( ptr );
    }
  for ( i = GUS_MAX_SAMPLES - 1; i >= 0 && gus_samples[ i ].smpno < 0; i-- );
  for ( ; i >= 0; i-- )
    if ( gus_samples[ i ].smpno < 0 )
      {
        if ( i < GUS_MAX_SAMPLES - 1 )
          memmove( &gus_samples[ i ], &gus_samples[ i + 1 ], 
          	   sizeof( struct gus_sample_t ) * ( GUS_MAX_SAMPLES - ( i + 1 ) ) );
      }
  for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
    gus_samples_index[ i ] = -1;
  for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
    if ( ( err = gus_samples[ i ].smpno ) >= 0 )
      gus_samples_index[ err ] = i;
  return 0;
}

static int gus_sample_reg_try_expand( int type, int memory_size_i )
{
  int i, memory_size, memory_used, ok;
  float factor;
  struct gus_sample_t *smp;

  for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
    {
      smp = &gus_samples[ i ];
      if ( smp -> smpno < 0 ) continue;
      smp -> type &= ~GUS_WAVE_TOTAL_EXPANDED;
      smp -> factor = (float)1.0;
    }
  ok = 0;
  factor = 1.0;
  while ( !ok )
    {
      memory_size = memory_size_i;
      memory_used = 0;
      for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
        {
          smp = &gus_samples[ i ];
          if ( smp -> smpno >= 0 )
            {
              size_t x;
              
              x = smp -> size;
              if ( smp -> type & GUS_WAVE_16BIT ) x <<= 1;
              x = (int)((float)x * smp -> factor);
              x = ( x + 31 ) & ~31;
              if ( smp -> type & ( GUS_WAVE_NO_EXPAND | GUS_WAVE_TOTAL_EXPANDED ) )
                memory_size -= x;
               else
                memory_used += x;
           }
        }
      if ( !memory_used ) break;
      factor *= (float)memory_size / (float)memory_used;
      for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
        {
          smp = &gus_samples[ i ];
          if ( smp -> smpno >= 0 && !( smp -> type & ( GUS_WAVE_NO_EXPAND | GUS_WAVE_TOTAL_EXPANDED ) ) )
            {
              smp -> type |= ( type & 0x0000000f ) << 28;
              smp -> factor = factor;
            }
        }
      ok = 1;
      for ( i = 0; i < GUS_MAX_SAMPLES; i++ )
        {
          smp = &gus_samples[ i ];
          if ( smp -> smpno >= 0 && 
               !( smp -> type & ( GUS_WAVE_NO_EXPAND | GUS_WAVE_TOTAL_EXPANDED ) ) )
            {
              if ( ( smp -> type & GUS_WAVE_16BIT ) || ( type == GUS_SAMPLE_EXPAND_S16 ) )
                {
                  if ( (int)(smp -> factor * (float)smp -> size) > 256L * 1024 )
                    {
                      smp -> factor = (float)( 256L * 1024 ) / (float)(smp -> size + 31 );
                      smp -> type |= GUS_WAVE_TOTAL_EXPANDED;
                      ok = 0;
                    }
                }
              if ( smp -> factor > 32.0 )
                {
                  smp -> factor = 32.0;
                  smp -> type |= GUS_WAVE_TOTAL_EXPANDED;
                  ok = 0;
                }
            }
        }
    }
  return 0;
}

static int gus_sample_reg_try_download( int type )
{
  int err, i;
  struct gus_sample_t *smp;
  struct GUS_STRU_DOWNLOAD gs;

  gus_reg_flag = 0;
  err = gus_sample_download_reset();
  if ( !err )
    {
      for ( i = 0; i < GUS_MAX_SAMPLES && !err; i++ )
        {
          smp = &gus_samples[ i ];
          if ( smp -> smpno < 0 ) continue;
          gs.voice_cntrl_reg = 0;
          if ( type == GUS_SAMPLE_EXPAND_S16 && smp -> factor >= 2.0 ) 
            gs.voice_cntrl_reg |= GUS_WAVE_16BIT;
          gs.length = ( (float)smp -> size * smp -> factor ) + 8;
          gs.loop_start = 0;
          gs.loop_end = 0;
          gs.sample_ptr = NULL;
          if ( smp -> type & GUS_WAVE_16BIT ) gs.length &= ~1;
          err = gus_sample_download_test( &gs );
        }
      gus_sample_download_reset();
    }
  gus_reg_flag = 1;
  return err;
}

int gus_sample_reg_expand( int type )
{
  int memory_size, err;

  if ( type <= GUS_SAMPLE_EXPAND_NONE ) return 0;
  if ( ( memory_size = gus_memory_size() ) - 32 != gus_memory_free() )
    return -EBUSY; /* too late */
  gus_reg_flag_download = 1;
  qsort( gus_samples, GUS_MAX_SAMPLES, sizeof( struct gus_sample_t ), sample_compar );
  err = 0;
  while ( !err && memory_size > 0 )
    {
      err = gus_sample_reg_try_expand( type, memory_size );
      if ( !err && !gus_sample_reg_try_download( type ) ) return 0;
      memory_size -= 1024;		/* step */
    }
  return err ? err : -ENOMEM;
}

/*
 *  ----------------- queue
 */

int gus_queue_flush( void )
{
  if ( ioctl( gus_handle, GUS_IOCTL_FLUSH ) < 0 )
    {
      gus_dprintf( "gus_queue_flush: gus ioctl - FLUSH\n" );
      return -1;
    }
  return 0;
}

int gus_queue_abort( void )
{
  if ( ioctl( gus_handle, GUS_IOCTL_ABORT ) < 0 )
    {
      gus_dprintf( "gus_queue_abort: gus ioctl - ABORT\n" );
      return -1;
    }
  return 0;
}

int gus_queue_abort_to_stop( void )
{
  if ( ioctl( gus_handle, GUS_IOCTL_QABORT ) < 0 )
    {
      gus_dprintf( "gus_queue_abort_to_stop: gus ioctl - QABORT\n" );
      return -1;
    }
  return 0;
}

int gus_queue_set_tick( int tick )
{
  if ( ioctl( gus_handle, GUS_IOCTL_SET_TICK, tick ) < 0 )
    {
      gus_dprintf( "gus_queue_set_tick: gus ioctl - SET_TICK\n" );
      return -1;
    }
  return 0;
}

int gus_queue_get_tick( void )
{
  return ioctl( gus_handle, GUS_IOCTL_GET_TICK );
}

/*
 *  ----------------- queue commands
 */

int gus_do_flush( void )
{
  int error;
  size_t size;

  error = 0;
  size = gus_queue_ptr - gus_queue_buffer;
  if ( size > 0 )
    {
      if ( write( gus_handle, gus_queue_buffer, size ) != size )
        {
          gus_dprintf( "gus_do_flush: write error" );
          error = -1;
        }
      gus_queue_ptr = gus_queue_buffer;
      gus_queue_len = gus_queue_buffer_size;
    }
  return error;
}

#define _gus_cmd_( size, cmd ) _gus_cmd__( size, GUS_CMD_##cmd )
#define _gus_vs_cmd_( size, cmd ) \
  gus_active_voice = voice; \
  _gus_vs_cmd__( voice, size, GUS_CMD_VS_##cmd )

static inline void _gus_cmd__( unsigned char size, unsigned char cmd )
{
  if ( gus_queue_len < size + 2 ) gus_do_flush();
  gus_queue_len -= size + 2;
  *gus_queue_ptr++ = size + 2;
  *gus_queue_ptr++ = cmd;
}

static inline void _gus_vs_cmd__( unsigned char voice,
				  unsigned char size,
				  unsigned char cmd )
{
  if ( gus_queue_len < size + 3 ) gus_do_flush();
  gus_queue_len -= size + 3;
  *gus_queue_ptr++ = size + 3;
  *gus_queue_ptr++ = cmd;
  *gus_queue_ptr++ = voice;
}

static inline void _gus_byte_( unsigned char b )
{
  *gus_queue_ptr++ = b;
}

#ifdef __LITTLE_ENDIAN

static inline void _gus_word_( unsigned short w )
{
  *((unsigned short *)gus_queue_ptr)++ = w;
}

static inline void _gus_dword_( unsigned int d )
{
  *((unsigned int *)gus_queue_ptr)++ = d;
}

#else

static inline void _gus_word_( unsigned short w )
{
  *gus_queue_ptr++ = (unsigned char)w;
  *gus_queue_ptr++ = (unsigned char)(w >> 8);
}

static inline void _gus_dword_( unsigned int d )
{
  *gus_queue_ptr++ = (unsigned char)d;
  *gus_queue_ptr++ = (unsigned char)(d >> 8);
  *gus_queue_ptr++ = (unsigned char)(d >> 16);
  *gus_queue_ptr++ = (unsigned char)(d >> 24);
}

#endif

static inline int _gus_expand_( int pos )
{
  if ( gus_reg_flag )
    {
      int sample;
      unsigned int type;
    
      if ( ( sample = gus_active_sample[ gus_active_voice ] ) < 0 ) return -1;
#if 0
      if ( ( sample = gus_samples_index[ sample ] ) < 0 ) return -1;
#endif
      type = gus_samples[ sample ].type;
      if ( type & GUS_WAVE_EXPANDED )
        pos = rint( gus_samples[ sample ].factor * (float)pos );
      if ( type & GUS_WAVE_16BIT )
        pos &= ~1;
    }
  return pos;
}

static inline unsigned int _gus_freq_expand_( unsigned int pos )
{
  if ( gus_freq_type == GUS_FREQ_TYPE_GF1 )
    {
      gus_dprintf( "_gus_freq_expand: you cann't use GF1 frequency type with expand feature of gus library" );
      return 0x200;
    }
  if ( gus_reg_flag )
    {
      int sample;
    
      if ( ( sample = gus_active_sample[ gus_active_voice ] ) < 0 ) return -1;
#if 0
      if ( ( sample = gus_samples_index[ sample ] ) < 0 ) return -1;
#endif
      if ( gus_samples[ sample ].type & GUS_WAVE_EXPANDED )
        pos = rint( gus_samples[ sample ].factor * (float)pos );
    }
  return pos;
}

void gus_do_voice_select( unsigned char voice )
{
#ifdef DEBUG
  if ( voice >= gus_active_voices )
    {
      gus_dprintf( "gus_do_voice_select: voice out of range" );
      return;
    }
#endif
  _gus_cmd_( 1, VOICE_SELECT );
  _gus_byte_( gus_active_voice = voice );
}

void gus_do_sample_select( unsigned short sample )
{
#ifdef DEBUG
  if ( sample >= GUS_MAX_SAMPLES )
    {
      gus_dprintf( "gus_do_sample_select: sample out of range" );
      return;
    }
  if ( gus_reg_flag && gus_samples_index[ sample ] < 0 )
    {
      gus_dprintf( "gus_do_sample_select: wrong sample" );
      return;
    }
  if ( gus_active_voice < 0 )
    {
      gus_dprintf( "gus_do_sample_select: wrong active voice" );
      return;
    }
#endif
  _gus_cmd_( 2, SAMPLE_SELECT );
  _gus_word_( gus_active_sample[ gus_active_voice ] =
  		( gus_reg_flag ? gus_samples_index[ sample ] : sample ) );
}

void gus_do_voice_control( unsigned char cntrl )
{
  _gus_cmd_( 1, VOICE_CONTROL );
  _gus_byte_( cntrl );
}

void gus_do_frequency( unsigned int freq )
{
  _gus_cmd_( 4, FREQUENCY );
  _gus_dword_( _gus_freq_expand_( freq ) );
}

void gus_do_loop_start( unsigned int start )
{
  _gus_cmd_( 4, LOOP_START );
  _gus_dword_( _gus_expand_( start ) );
}

void gus_do_loop_end( unsigned int end )
{
  _gus_cmd_( 4, LOOP_END );
  _gus_dword_( _gus_expand_( end ) );
}

void gus_do_ramp_rate( unsigned char rate )
{
  _gus_cmd_( 1, RAMP_RATE );
  _gus_byte_( rate );
}

void gus_do_ramp_start( unsigned char start )
{
  _gus_cmd_( 1, RAMP_START );
  _gus_byte_( start );
}

void gus_do_ramp_end( unsigned char end )
{
  _gus_cmd_( 1, RAMP_END );
  _gus_byte_( end );
}

void gus_do_volume( unsigned short vol )
{
  _gus_cmd_( 2, CURRENT_VOL );
  _gus_word_( vol );
}

void gus_do_position( unsigned int pos )
{
  _gus_cmd_( 4, CURRENT_LOC );
  _gus_dword_( _gus_expand_( pos ) );
}

void gus_do_pan( unsigned char pan )
{
  _gus_cmd_( 1, CURRENT_PAN );
  _gus_byte_( pan );
}

void gus_do_ramp_control( unsigned char cntrl )
{
  _gus_cmd_( 1, RAMP_CONTROL );
  _gus_byte_( cntrl );
}

void gus_do_freq_and_vol( unsigned int freq, unsigned short vol )
{
  _gus_cmd_( 6, FREQ_AND_VOL );
  _gus_dword_( _gus_freq_expand_( freq ) );
  _gus_word_( vol );
}

void gus_do_loop_all( unsigned char cntrl,
		      unsigned int start,
		      unsigned int end )
{
  _gus_cmd_( 9, RAMP_ALL );
  _gus_byte_( cntrl );
  _gus_dword_( _gus_expand_( start ) );
  _gus_dword_( _gus_expand_( end ) );
}

void gus_do_ramp_all( unsigned char cntrl, unsigned char rate,
		      unsigned char start, unsigned char end )
{
  _gus_cmd_( 4, FREQ_AND_VOL );
  _gus_byte_( cntrl );
  _gus_byte_( rate );
  _gus_byte_( start );
  _gus_byte_( end );
}

void gus_do_voice_stop( void )
{
  _gus_cmd_( 0, VOICE_STOP );
}

void gus_do_vs_voice_control( unsigned char voice, unsigned char cntrl )
{
  _gus_vs_cmd_( 1, VOICE_CONTROL );
  _gus_byte_( cntrl );
}

void gus_do_vs_frequency( unsigned char voice, unsigned int freq )
{
  _gus_vs_cmd_( 4, FREQUENCY );
  _gus_dword_( _gus_freq_expand_( freq ) );
}

void gus_do_vs_loop_start( unsigned char voice, unsigned int start )
{
  _gus_vs_cmd_( 4, LOOP_START );
  _gus_dword_( _gus_expand_( start ) );
}

void gus_do_vs_loop_end( unsigned char voice, unsigned int end )
{
  _gus_vs_cmd_( 4, LOOP_END );
  _gus_dword_( _gus_expand_( end ) );
}

void gus_do_vs_ramp_rate( unsigned char voice, unsigned char rate )
{
  _gus_vs_cmd_( 1, RAMP_RATE );
  _gus_byte_( rate );
}

void gus_do_vs_ramp_start( unsigned char voice, unsigned char start )
{
  _gus_vs_cmd_( 1, RAMP_START );
  _gus_byte_( start );
}

void gus_do_vs_ramp_end( unsigned char voice, unsigned char end )
{
  _gus_vs_cmd_( 1, RAMP_END );
  _gus_byte_( end );
}

void gus_do_vs_volume( unsigned char voice, unsigned short vol )
{
  _gus_vs_cmd_( 2, CURRENT_VOL );
  _gus_word_( vol );
}

void gus_do_vs_position( unsigned char voice, unsigned int pos )
{
  _gus_vs_cmd_( 4, CURRENT_LOC );
  _gus_dword_( _gus_expand_( pos ) );
}

void gus_do_vs_pan( unsigned char voice, unsigned char pan )
{
  _gus_vs_cmd_( 1, CURRENT_PAN );
  _gus_byte_( pan );
}

void gus_do_vs_ramp_control( unsigned char voice, unsigned char cntrl )
{
  _gus_vs_cmd_( 1, RAMP_CONTROL );
  _gus_byte_( cntrl );
}

void gus_do_vs_freq_and_vol( unsigned char voice,
			     unsigned int freq,
			     unsigned short vol )
{
  _gus_vs_cmd_( 6, FREQ_AND_VOL );
  _gus_dword_( _gus_freq_expand_( freq ) );
  _gus_word_( vol );
}

void gus_do_vs_loop_all( unsigned char voice,
			 unsigned char cntrl,
			 unsigned int start,
			 unsigned int end )
{
  _gus_vs_cmd_( 9, FREQ_AND_VOL );
  _gus_byte_( cntrl );
  _gus_dword_( _gus_expand_( start ) );
  _gus_dword_( _gus_expand_( end ) );
}

void gus_do_vs_ramp_all( unsigned char voice,
			 unsigned char cntrl, unsigned char rate,
		         unsigned char start, unsigned char end )
{
  _gus_vs_cmd_( 4, RAMP_ALL );
  _gus_byte_( cntrl );
  _gus_byte_( rate );
  _gus_byte_( start );
  _gus_byte_( end );
}

void gus_do_vs_voice_stop( unsigned char voice )
{
  _gus_vs_cmd_( 0, VOICE_STOP );
}

void gus_do_sample_start( unsigned char voice, unsigned short sample,
			  unsigned int freq, unsigned char volume )
{
  gus_active_voice = voice;
  if ( gus_reg_flag )
    {
      if ( gus_samples_index[ sample ] < 0 ) return;
      sample = gus_samples_index[ sample ];
    }
#if 0
  gus_dprintf( "v = %i, s = %i, f = %i, vol = %i", voice, sample, freq, volume );
#endif 
  gus_active_sample[ voice ] = sample;
  _gus_cmd_( 9, SAMPLE_START );
  _gus_byte_( voice );
  _gus_word_( sample );
  _gus_dword_( _gus_freq_expand_( freq ) );
  _gus_word_( volume );
}

void gus_do_sample_start1( void )
{
  _gus_cmd_( 0, SAMPLE_START1 );
}

void gus_do_set_timer( unsigned int time )
{
  _gus_cmd_( 4, SET_TIMER );
  _gus_dword_( time );
}

void gus_do_set_timer1( unsigned short time )
{
  _gus_cmd_( 2, SET_TIMER1 );
  _gus_word_( time );
}

void gus_do_set_timer2( unsigned short time )
{
  _gus_cmd_( 2, SET_TIMER2 );
  _gus_word_( time );
}

void gus_do_wait( unsigned short ticks )
{
  _gus_cmd_( 2, WAIT );
  _gus_word_( ticks );
}

void gus_do_stop( void )
{
  _gus_cmd_( 0, STOP );
}
