/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Mixer support routines
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  variables
 */

static int mixer_handle = -1;
static int mixer_devmask;
static int mixer_stereodevs;

/*
 *  EXPORTED FUNCTIONS
 */ 

int gus_mixer_open( void )
{
  int i, mversion; 

  if ( ( mixer_handle = open( "/dev/mixer", O_RDWR ) ) < 0 )
    {
      gus_dprintf( "gus_mixer_open: file open error %i", errno );
      return -1;
    }
  mversion = GUS_MIXER_NONE;
  if ( ioctl( mixer_handle, SOUND_MIXER_READ_GUS_VERSION, &i ) < 0 )
    {
      gus_dprintf( "gus_mixer_open: read gus version error" );
      close( mixer_handle );
      mixer_handle = -1;
      return -1;
    }
  if ( i >= 0x35 && i <= 0x37 ) mversion = GUS_MIXER_ICS; else
  if ( i == 0xa0 ) mversion = GUS_MIXER_CODEC;
  i = 0;
  if ( ioctl( mixer_handle, SOUND_MIXER_READ_DEVMASK, &mixer_devmask ) < 0 ) i = -1;
  if ( ioctl( mixer_handle, SOUND_MIXER_READ_STEREODEVS, &mixer_stereodevs ) < 0 ) i = -1;
  if ( i < 0 )
    {
      gus_dprintf( "gus_mixer_open: device mask or stereo devices ioctl error" );
      close( mixer_handle );
      mixer_handle = -1;
      return -1;
    }
  return mversion;  
}

int gus_mixer_close( void )
{
  int res;
  
  res = close( mixer_handle ) < 0 ? -1 : 0;
  mixer_handle = -1;
  return mixer_handle;
}

int gus_mixer_read_devmask( void )
{
  return mixer_devmask;
}

int gus_mixer_read_stereodevs( void )
{
  return mixer_stereodevs; 
}

#ifdef DEBUG

#define gus_mixer_read( what, mesg ) \
	_gus_mixer_read_( SOUND_MASK_##what, SOUND_MIXER_READ_##what, mesg )

static int _gus_mixer_read_( int mask, int what, char *mesg )
{
  int res = -1;

  if ( mixer_devmask & mask )
    if ( ioctl( mixer_handle, what, &res ) < 0 )
      gus_dprintf( "gus_mixer_read: %s", mesg );
  return res;
}

#else

#define gus_mixer_read( what, mesg ) \
	_gus_mixer_read_( SOUND_MASK_##what, SOUND_MIXER_READ_##what )

static inline int _gus_mixer_read_( int mask, int what )
{
  if ( mixer_dev_mask & mask )
    return ioctl( mixer_handle, what, &res ) < 0 ? -1 : 0;
   else
    return -1;
}

#endif

int gus_mixer_read_recmask( void )
{
  int res = -1;

  if ( ioctl( mixer_handle, SOUND_MIXER_READ_RECMASK, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_recmask: error" );
  return res;
}

int gus_mixer_read_recsrc( void )
{
  int res = -1;

  if ( ioctl( mixer_handle, SOUND_MIXER_READ_RECSRC, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_recsrc: error" );
  return res;
}

int gus_mixer_read_mic( void )
{
  return gus_mixer_read( MIC, "MIC" );
}

int gus_mixer_read_cd( void )
{
  return gus_mixer_read( CD, "CD" );
}

int gus_mixer_read_line( void )
{
  return gus_mixer_read( LINE, "LINE" );
}

int gus_mixer_read_synth( void )
{
  return gus_mixer_read( SYNTH, "SYNTH" );
}

int gus_mixer_read_pcm( void )
{
  return gus_mixer_read( PCM, "PCM" );
}

int gus_mixer_read_reclev( void )
{
  return gus_mixer_read( RECLEV, "RECLEV" );
}

int gus_mixer_read_volume( void )
{
  return gus_mixer_read( VOLUME, "VOLUME" );
}

int gus_mixer_read_imix( void )
{
  return gus_mixer_read( IMIX, "IMIX" );
}

#ifdef DEBUG

#define gus_mixer_write( what, data, mesg ) \
	_gus_mixer_write_( SOUND_MASK_##what, SOUND_MIXER_WRITE_##what, data, mesg )

static int _gus_mixer_write_( int mask, int what, int data, char *mesg )
{
  int res = -1;

  if ( mixer_devmask & mask )
    if ( ( res = ioctl( mixer_handle, what, &data ) ) < 0 )
      gus_dprintf( "gus_mixer_write: 0x%x, %s", data, mesg );
  return res;
}

#else

#define gus_mixer_write( what, data, mesg ) \
	_gus_mixer_write_( SOUND_MASK_##what, SOUND_MIXER_WRITE_##what, data )

static inline int _gus_mixer_write_( int mask, int what, int data )
{
  if ( mixer_dev_mask & mask )
    return ioctl( mixer_handle, what, &data ) < 0 ? -1 : 0;
   else
    return -1;
}

#endif

int gus_mixer_write_recsrc( int value )
{
  if ( ioctl( mixer_handle, SOUND_MIXER_WRITE_RECSRC, &value ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_recsrc: error" );
      return -1;
    }
  return 0;
}

int gus_mixer_write_mic( int value )
{
  return gus_mixer_write( MIC, value, "MIC" );
}

int gus_mixer_write_cd( int value )
{
  return gus_mixer_write( CD, value, "CD" );
}

int gus_mixer_write_line( int value )
{
  return gus_mixer_write( LINE, value, "LINE" );
}

int gus_mixer_write_synth( int value )
{
  return gus_mixer_write( SYNTH, value, "SYNTH" );
}

int gus_mixer_write_pcm( int value )
{
  return gus_mixer_write( PCM, value, "PCM" );
}

int gus_mixer_write_reclev( int value )
{
  return gus_mixer_write( RECLEV, value, "RECLEV" );
}

int gus_mixer_write_volume( int value )
{
  return gus_mixer_write( VOLUME, value, "VOLUME" );
}

int gus_mixer_write_imix( int value )
{
  return gus_mixer_write( IMIX, value, "IMIX" );
}
