/*
 * Application interface library for the GUS driver
 * Copyright (c) 1994/95 by Jaroslav Kysela (Perex soft)
 */

#ifndef __LIBGUS_H
#define __LIBGUS_H

#include <malloc.h>		/* size_t type needed */
#include <sys/gus.h>

  /*
   * all functions return value -1 if error occured
   */

/***************************************************************************
 *                                                                         *
 *              Native synthesizer support functions                       *
 *                                                                         *
 ***************************************************************************/

int gus_open( size_t queue_buffer_size );
  /*
   * input value:	size of command queue buffer (512-1MB)
   *			buffer is allocated dynamicaly
   * return value:	zero if sucess
   * note:		this function must be called as first
   * 			open file /dev/gus
   */
int gus_close( void );
  /*
   * close file (gus synthesizer) previously opened with gusOpen function
   * return value:	zero if sucess
   */

int gus_info( struct GUS_STRU_INFO *info );
  /*
   * return value:	filled info variable with actual values
   * 			(see to gus.h header file for more informations)
   */
int gus_info_port( void );
  /*
   * return value:	port number (for example 0x220)
   */
int gus_info_irq( void );
  /*
   * return value:	irq number (for example 11)
   */
int gus_info_dma1( void );
  /*
   * return value:	dma1 number (for example 5)
   */
int gus_info_dma2( void );
  /*
   * return value:	dma2 number (for example 6)
   * note:		dma1 and dma2 maybe same if is used only one dma channel
   */
int gus_info_version( void );
  /*
   * return values:	0x24	- gus revision 2.4
   *			0x35	- gus revision 3.7 with flipped mixer channels
   *			0x37	- gus revision 3.7
   *			0xa0	- gus max revision 10?
   */
int gus_info_noise_channel( void );
  /*
   * return value:	0	- noise channel disabled
   *			1	- noise channel enabled
   * note:		if you use frequency type GF1 (GUS_FREQ_TYPE_GF1)
   *			you must compute with noise channel
   * warning:		reset function must be called before
   */

int gus_setup_volume_type( int type );
  /*
   * input value:	see to gus.h header file for GUS_VOL_TYPE_XXXX constants
   * return value:	zero if sucess
   */
int gus_setup_frequency_type( int type );
  /*
   * input value:	see to gus.h header file for GUS_FREQ_TYPE_XXXX constants
   * return value:	zero if sucess
   */
int gus_setup_set_auto_ramp( int speed );
  /*
   * input value:	auto ramp speed - 64 = off, 0 = max speed
   * return value:	zero if sucess
   */
int gus_setup_get_auto_ramp( void );
  /*
   * return value:	auto ramp speed
   */
int gus_setup_set_smooth_pan( int speed );
  /*
   * input value:	smooth pan speed - 0   = off (max speed),
   *					   255 = min speed
   * return value:	zero if sucess
   */
int gus_setup_get_smooth_pan( void );
  /*
   * return value:	smooth pan speed
   */

int gus_memory_size( void );
  /*
   * return value:	gus memory size in bytes
   */
int gus_memory_free( void );
  /*
   * return value:	unused gus memory in bytes
   * warning:		reset function must be called before
   */
int gus_memory_max_bank_free( void );
  /*
   * return value:	current maximum free memory in one bank
   * warning:		reset function must be called before
   */

int gus_reset( short voices );
  /*
   * input value:	active voices (without noise channel)
   * return value:	zero if reset was sucess (GF1 chip active)
   */
int gus_reset_engine_only( void );
  /*
   * return value:	zero if reset was sucess
   * note:		this command doesn't change number of active
   *			voices and doesn't do hardware reset
   */
   
int gus_timer_start( int time );
  /*
   * input value:	time between tick in microseconds
   *			range: 40-81959
   * return value:	zero if sucess
   */
int gus_timer_stop( void );
  /*
   * return value:	zero if timer was stoped
   */
int gus_timer_start1( int time );
  /*
   * input value:	time between tick in 80 microseconds
   *			range: 1-255
   * return value:	zero if sucess
   * note:		timers (1 & 2) cann't be combined!!!
   */
int gus_timer_stop1( void );
  /*
   * return value:	zero if timer1 was stoped
   */
int gus_timer_start2( int time );
  /*
   * input value:	time between tick in 320 microseconds
   *			range: 1-255
   * return value:	zero if sucess
   * note:		timers (1 & 2) cann't be combined!!!
   */
int gus_timer_stop2( void );
  /*
   * return value:	zero if timer2 was stoped
   */

int gus_sample_download( struct GUS_STRU_DOWNLOAD *download );
  /*
   * input value:	see to gus.h for more details about GUS_STRU_DOWNLOAD
   *			structure
   * return value:	zero if sample was sucessfully downloaded
   */
int gus_sample_reset( void );
  /*
   * return value:	zero if samples was sucessfully removed from
   *			GF1 memory manager
   */
int gus_sample_download_test( struct GUS_STRU_DOWNLOAD *download );
  /*
   * input value:	see to gus.h for more details about GUS_STRU_DOWNLOAD
   *			structure
   * return value:	zero if sample !can be! downloaded
   * note:		sample is allocated in GF1 memory manager
   */
int gus_sample_download_test1( struct GUS_STRU_DOWNLOAD *download );
  /*
   * input value:	see to gus.h for more details about GUS_STRU_DOWNLOAD
   *			structure
   * return value:	zero if sample !can be! downloaded
   * note:		sample isn't allocated in GF1 memory manager
   */

#define GUS_WAVE_UNSIGNED	GUS_WAVE_INVERT	/* alias only */
#define GUS_WAVE_NO_LOOP_ADAPT	0x00010000 /* don't adapt sample loop for GF1 chip */
#define GUS_WAVE_DELTA		0x00020000 /* sample is stored in delta format */
#define GUS_WAVE_NO_EXPAND	0x00040000 /* don't allow expanding of this sample */

int gus_sample_reg( unsigned short number, unsigned int type,
                    int size, int lstart, int lend );
  /*
   * input value:	sample number
   *			sample type (see GUS_WAVE_XXXX constants to gus.h)
   *			sample size
   *			sample loop start
   *			sample loop end
   * note: 		gus_sample_reg_XXXX operates only with samples 
   *			registered with this function
   */
int gus_sample_reg_size( void );
  /*
   * output value:	total size of registered samples in gus memory
   */
int gus_sample_reg_download( 
	int (*read_fcn)( unsigned short number, unsigned char *ptr, int size ),
	void (*error_fcn)( unsigned short number, int error ) );
  /*
   * input value:	pointer to function which read sample to *ptr address;
   *			function must return zero if success
   */

#define GUS_SAMPLE_EXPAND_NONE	0
#define GUS_SAMPLE_EXPAND_S8	1	/* spline 8-bit */
#define GUS_SAMPLE_EXPAND_S16	2	/* spline 16-bit */

int gus_sample_reg_expand( int type );
  /*
   * input value:	expand type - see GUS_SAMPLE_EXPAND_XXXX
   * note:		this function expand or shrink all registered
   *			samples to gus memory size
   */

int gus_queue_flush( void );
  /*
   * return value:	zero if command queue was sucessfully flushed
   */
int gus_queue_abort( void );
  /*
   * return value:	zero if command queue was sucessfully aborted and
   *			GF1 chip was stoped
   * note:		reset must be called after this command for
   *			more sound!
   */
int gus_queue_abort_to_stop( void );
  /*
   * return value:	zero if command queue was sucessfully aborted
   *			to queue command STOP; all voices was stoped
   */
int gus_queue_set_tick( int tick );
  /*
   * input value:	tick count (only information value for application)
   * return value:	zero if sucess
   */
int gus_queue_get_tick( void );
  /*
   * return value:	current tick count
   */

/*
 *  Queue commands
 *
 *  This commands are interpreted by block. Blocks are separated by command
 *  gus_do_wait or gus_do_stop (for abort to stop command). Queue command
 *  isn't send to GUS driver in same time as user its use. Queue commands
 *  are stored to internal buffer which gus library allocates dynamicaly
 *  (see to gus_open function). Internal buffer is flushed (sended to
 *  GUS driver) when it is full or user can force buffer flush with 
 *  gus_do_flush function.
 */

int gus_do_flush( void );
  /*
   * return value:	zero if command queue was sucessfully flushed
   */

void gus_do_voice_select( unsigned char voice );
  /*
   * select active voice (0-31)
   */
void gus_do_sample_select( unsigned short sample );
  /*
   * select active sample (0-127)
   */
void gus_do_voice_control( unsigned char cntrl );
  /*
   * set voice control register:
   *	bit 0. - 0 = voice is running, 1 = voice is stopped
   *	bit 1. - 0 = nothing, 1 = stop voice
   *	bit 2. - 0 = 8 bit wave, 1 = 16 bit wave
   *	bit 3. - 0 = nothing, 1 = enable looping
   *	bit 4. - 0 = nothing, 1 = bidirectional looping
   *	bit 5. - 0 = nothing, 1 = enable wavetable IRQ
   *	bit 6. - 0 = increasing addresses, 1 = decreasing addresses
   *	bit 7. - 0 = wavetable IRQ pending
   * note: don't set bits 5 & 7 - these things must be controled by driver
   */
void gus_do_frequency( unsigned int freq );
  /*
   * set voice frequency:
   *	frequency type = GUS_FREQ_TYPE_GF1 - native gus frequency
   *    frequency type = GUS_FREQ_TYPE_HZ2 - frequency in Hz * 2
   *    frequency type = GUS_FREQ_TYPE_HZ  - frequency in Hz (default)
   */
void gus_do_loop_start( unsigned int start );
  /*
   * set voice loop start position in sample
   * value is relative from begin of sample
   */
void gus_do_loop_end( unsigned int end );
  /*
   * set voice loop end position in sample
   * value is relative from begin of sample
   */
void gus_do_ramp_rate( unsigned char rate );
  /*
   * set voice volume ramping rate
   * (see to UltraSound Lowlevel Toolkit document to more details)
   *    bits 7-6: rate
   *	bits 5-0: volume increment
   */
void gus_do_ramp_start( unsigned char start );
  /*
   * set voice ramp start value
   *    bits 7-4: exponent
   *    bits 3-0: mantisa
   */
void gus_do_ramp_end( unsigned char end );
  /*
   * set voice ramp end value
   *    bits 7-4: exponent
   *    bits 3-0: mantisa
   */
void gus_do_volume( unsigned short vol );
  /*
   * set voice volume level:
   *	volume type = GUS_VOL_TYPE_GF1 - native gus volume level
   *    volume type = GUS_VOL_TYPE_LINEAR - linear (0-128)
   */
void gus_do_position( unsigned int pos );
  /*
   * set current voice position
   * value is relative from begin of sample
   */
void gus_do_pan( unsigned char pan );
  /*
   * set voice pan
   */
void gus_do_ramp_control( unsigned char cntrl );
  /*
   * set ramp control register
   *   bit 0. - 0 = volume ramping in progress, 1 = volume ramping is inactive
   *   bit 1. - 0 = nothing, 1 = stop ramp
   *   bit 2. - 0 = nothing, 1 = roll over condition
   *   bit 3. - 0 = nothing, 1 = ramp loop enable
   *   bit 4. - 0 = nothing, 1 = bidirectional ramp loop enable
   *   bit 5. - 0 = nothing, 1 = volume ramp IRQ enable
   *   bit 6. - 0 = increasing direction, 1 = decreasing direction
   *   bit 7. - 0 = nothing, 1 = volume ramp IRQ pending
   * note: don't set bits 2,5,7 - these things must be controled by driver
   */
void gus_do_freq_and_vol( unsigned int freq, unsigned short vol );
  /*
   * combined command - frequency and volume
   */
void gus_do_loop_all( unsigned char cntrl,
                      unsigned int start,
                      unsigned int end );
  /*
   * combined command - voice control, loop start, loop end
   */
void gus_do_ramp_all( unsigned char cntrl, unsigned char rate,
                      unsigned char start, unsigned char end );
  /*
   * combined command - volume ramp control, ramp rate, ramp start, ramp end
   */
void gus_do_voice_stop( void );
  /*
   * voice stop command
   */

   
  /*
   *  gus_do_vs_XXXX commands are equivalents to gus_do_XXXX commands, but
   *  first argument is voice (channel);
   *  combination of gus_do_voice_select + gus_do_XXXX
   */
   
void gus_do_vs_voice_control( unsigned char voice, unsigned char cntrl );
void gus_do_vs_frequency( unsigned char voice, unsigned int freq );
void gus_do_vs_loop_start( unsigned char voice, unsigned int start );
void gus_do_vs_loop_end( unsigned char voice, unsigned int end );
void gus_do_vs_ramp_rate( unsigned char voice, unsigned char rate );
void gus_do_vs_ramp_start( unsigned char voice, unsigned char start );
void gus_do_vs_ramp_end( unsigned char voice, unsigned char end );
void gus_do_vs_volume( unsigned char voice, unsigned short vol );
void gus_do_vs_position( unsigned char voice, unsigned int pos );
void gus_do_vs_pan( unsigned char voice, unsigned char pan );
void gus_do_vs_ramp_control( unsigned char voice, unsigned char cntrl );
void gus_do_vs_freq_and_vol( unsigned char voice,
                             unsigned int freq,
                             unsigned short vol );
void gus_do_vs_loop_all( unsigned char voice,
                         unsigned char cntrl,
                         unsigned int start,
                         unsigned int end );
void gus_do_vs_ramp_all( unsigned char voice,
                         unsigned char cntrl, unsigned char rate,
                         unsigned char start, unsigned char end );
void gus_do_vs_voice_stop( unsigned char voice );


void gus_do_sample_start( unsigned char voice, unsigned short sample,
                          unsigned int freq, unsigned char volume );
  /*
   * start playback of sample
   */
void gus_do_sample_start1( void );
  /*
   * start playback of sample, but all things must be initialized manualy
   * this maybe good after gus_do_stop command
   */
void gus_do_set_timer( unsigned int time );
  /*
   * set timer to x microseconds
   */
void gus_do_set_timer1( unsigned short time );
  /*
   * set timer1 to x * 80 microseconds
   */
void gus_do_set_timer2( unsigned short time );
  /*
   * set timer2 to x * 320 microseconds
   */
void gus_do_wait( unsigned short ticks );
  /*
   * wait x ticks - this command is block separator
   * all commands between blocks are interpreted in begin of one tick
   */
void gus_do_stop( void );
  /*
   * this command is remark for gus_queue_abort_to_stop
   */

/***************************************************************************
 *                                                                         *
 *                    GUS mixer support functions                          *
 *                                                                         *
 ***************************************************************************/

#define GUS_MIXER_NONE		0	/* GUS rev. <3.7 */
#define GUS_MIXER_ICS		1	/* GUS rev. 3.7 & above */
#define GUS_MIXER_CODEC		2	/* GUS MAX */

int gus_mixer_open( void );
  /*
   * return value:	see to GUS_MIXER_XXXX constants
   */
int gus_mixer_close( void );
  /*
   * return value:	zero if success
   */

int gus_mixer_read_devmask( void );
  /*
   * return value:	available devices mask
   *			(see to SOUND_MASK_XXXX - soundcard.h)
   */
int gus_mixer_read_stereodevs( void );
  /*
   * return value:	available stereo devices mask
   *			(see to SOUND_MASK_XXXX - soundcard.h)
   */
int gus_mixer_read_recmask( void );
  /*
   * return value:	available record devices mask
   *			(see to SOUND_MASK_XXXX - soundcard.h)
   */
int gus_mixer_read_recsrc( void );
  /*
   * return value:	current record source
   *			(see to SOUND_MASK_XXXX - soundcard.h)
   */
int gus_mixer_write_recsrc( int value );
  /*
   * input value:	new record source
   *			(see to SOUND_MASK_XXXX - soundcard.h)
   */

  /*
   * functions bellow have input/output values in this form:
   *	bits 0-7 (low byte): left channel volume level
   *	bits 8-15 (high byte): right channel volume level
   */

int gus_mixer_read_mic( void );		/* all (NONE & CODEC - mono) */
int gus_mixer_read_cd( void );		/* ICS, CODEC */
int gus_mixer_read_line( void );	/* all (NONE - mono) */
int gus_mixer_read_synth( void );	/* all (NONE - mono) */
int gus_mixer_read_pcm( void );		/* all */
int gus_mixer_read_reclev( void );	/* CODEC only - mono */
int gus_mixer_read_volume( void );	/* ICS only */
int gus_mixer_read_imix( void );	/* ICS, CODEC - both mono */

int gus_mixer_write_mic( int value );
int gus_mixer_write_cd( int value );
int gus_mixer_write_line( int value );
int gus_mixer_write_synth( int value );
int gus_mixer_write_pcm( int value );
int gus_mixer_write_reclev( int value );
int gus_mixer_write_volume( int value );
int gus_mixer_write_imix( int value );

/***************************************************************************
 *                                                                         *
 *                      Conversion routines                                *
 *                                                                         *
 ***************************************************************************/

void gus_convert_delta( unsigned int type, unsigned char *dest,
			unsigned char *src, size_t size );
  /*
   * note: dest and src pointers can be equal
   */
void gus_convert_8bit_to_16bit( unsigned short *dest, unsigned char *src,
				size_t size );
  /*
   * note: dest and src pointers can be equal
   */
void gus_convert_spline_oversampling_8bit(
			signed char *dest, signed char *src,
			size_t size, float factor, int unsigned_ );
  /*
   * note: output array is always signed
   */
void gus_convert_spline_oversampling_16bit(
			signed short *dest, signed short *src,
			size_t size, float factor, int unsigned_ );
  /*
   * note: output array is always signed
   */
void gus_convert_spline_oversampling_8bit_to_16bit(
			signed short *dest, signed char *src,
			size_t size, float factor, int unsigned_ );
  /*
   * note: output array is always signed
   */

/*
 *
 */
 
#endif /* __LIBGUS_H */
