#include <stdio.h>
#include <bios.h>
#include <time.h>
#include <math.h>
#include <stdlib.h>
#include <i86.h>
#include <string.h>

#include "defines.hpp"

#include "vesa.hpp"
#include "3d.hpp"
#include "vesa3d.hpp"
#include "misc.hpp"
#include "mouse.hpp"

void ShowTitle() {
    printf( "# 3D-BMP-Viewer V1.2   (w) by P.Kayser 1996 \n" );
    printf( "# EMAIL : p.kayser@tu-bs.de | COMPUSERVE: 100517,3516\n" );
};

void ShowUsage() {
    printf( "# usage : 3dview (-abdlmwz) <file(.bmp)> (<mode>)\n" );
    printf( "# \n" );
    printf( "# options\n" );
    printf( "#   -a        : show additional informations.\n" );
    printf( "#   -b        : disable bitmap.\n" );
    printf( "#   -d<dist>  : distance. <dist> from 0.01 (far) to 3 (near).\n" );
    printf( "#   -l        : list available 256-color-modes.\n" );
    printf( "#   -m        : enable mouse-control.\n" );
    printf( "#   -w        : if screen flickers, try this.\n" );
    printf( "#   -z        : enable z-buffer.\n" );
};

void ShowAddInfo() {
    printf( "# \n" );
    printf( "# Needs VBE version 2.0. If your graphic card has an older VBE version,\n" );
    printf( "# get shareware UniVBE 5.1a or newer from www.scitechsoft.com!\n" );
    printf( "# Only 256-color-modes allowed.\n" );
    printf( "# Some modes may not work even when the mode is listed using the\n");
    printf( "# '-l'-option. Because this program uses double-buffered animation,\n" );
    printf( "# you will for example need 2 MB on your graphic card for 1024x768.\n" );
    printf( "# The bitmap must be '256-color-WINDOWS-RGB-encoded' and should have\n" );
    printf( "# a maximum of 192 colors. Use a program like shareware Paintshop\n" );
    printf( "# Pro to convert to .BMP and decrease color depth to 192.\n" );
};

void main( int argc, char *argv[] ) {

    Boolean ListModes = False;
    Boolean ShowBitmap = True;
    Boolean ShowAddInfos = False;
    Boolean WithZBuffer = False;
    Boolean MouseControl = False;
    Boolean WaitVRetrace = False;
    double MinZoom = 0.01;
    double MaxZoom = 3;
    double NormZoom = 1.5;
    double Distance = NormZoom;
    int CmdCount = 1;
    char Buffer [15];

    VESA_SetTextMode();
    ShowTitle();

    if ( VESA_Boot() != 0 ) {
        return;
    };

    if ( argc == CmdCount ) {
        ShowUsage();
        return;
    };

    while ( (argv[CmdCount])[0] == '-' ) {
        int OptCount = 1;
        while ( OptCount < strlen( argv[CmdCount] ) ) {
            switch ( (argv[CmdCount])[OptCount] ) {
                case 'a' :
                    ShowAddInfos = True;
                    break;
                case 'b' :
                    ShowBitmap = False;
                    break;
                case 'd' :
                    Distance = strtod( strncpy( Buffer,
                                                &(argv[CmdCount])[OptCount+1],
                                                strlen( argv[CmdCount] ) - OptCount ),
                                       NULL );
                    if ( Distance < MinZoom || Distance > MaxZoom )
                        Distance = NormZoom;
                    OptCount = strlen ( argv[CmdCount] );                                       
                    break;
                case 'l' :
                    ListModes = True;
                    break;
                case 'm' :
                    MouseControl = True;
                    break;
                case 'w' :
                    WaitVRetrace = True;
                    break;
                case 'z' :
                    WithZBuffer = True;
                    break;
                default:
                    printf( "\n" );
                    printf( "ERROR : '%c' is an unknown option !\n" , argv[CmdCount][OptCount] );
                    return;
            }
            OptCount++;
        };
        CmdCount++;
    };

    if ( ShowAddInfos == True ) {
        ShowUsage();
        ShowAddInfo();
        return;       
    };

    if ( ListModes == True ) {
        printf( "\n" );
        VESA_PrintModeList8B();
        return;
    };

    static VESA3D_Texture TEX;
    char* Filename = new char [256];
    strcpy( Filename, argv[CmdCount] );

    if ( VESA3D_LoadTexture( Filename, TEX ) != 0 )
        if ( VESA3D_LoadTexture( strcat( Filename, ".BMP" ), TEX ) != 0 ) {
            printf( "\n" );
            printf( "ERROR : Could not open .BMP-File !\n" );
            return;
    };

    VESA3D_ConvertTexture( TEX );

    CmdCount++;

    WORD Mode;
    
    if ( argc == CmdCount ) {
        Mode = 0x0101;
    }
    else {
        Mode = WORD( strtoul( argv[CmdCount], NULL, 16 ) );
    };

    if ( VESA_InitMode( Mode ) != 0 ) {
        return;
    };

    if ( VESA_BpP != 8 ) {
        printf( "\n" );
        printf( "ERROR: Not a 256-Color-Mode !\n" );
        return;
    };

    VESA3D_Init();
    if ( WithZBuffer == True )
        VESA3D_MakeZBuffer();

    for ( int i = 46; i >= 0; i-- )
        VESA_SetColor8B( WORD(239-i) ,
                         WORD(64+i*3),
                         WORD(64+i*3),
                         WORD(64+i*3) );

    for ( i = 15; i >= 0; i-- )
        VESA_SetColor8B( WORD(255-i),
                         WORD(128+i*8),
                         WORD(128+i*8),
                         WORD(128+i*8) );

    VESA_SetColor8B( 0, 0, 0, 0 );

    if ( ShowBitmap == True )
        VESA3D_SetTexturePalette( TEX, 192 );
 
    double px, py;

    if ( TEX.TDY > TEX.TDX ) {
        px = 1;
        py = double(TEX.TDY) / TEX.TDX;
    }
    else {
        py = 1;
        px = double(TEX.TDX) / TEX.TDY;
    };

    long p1 = _3D_NewPoint(  0, py, 0 );
    long p2 = _3D_NewPoint( px, py, 0 );
    long p3 = _3D_NewPoint( px,  0, 0 );
    long p4 = _3D_NewPoint(  0,  0, 0 );

    double dx = 0.25, dz = 0.05;

    long p5  = _3D_NewPoint( -dx, -dx, dz );
    long p6  = _3D_NewPoint( -dx, -dx, -dz );
    long p7  = _3D_NewPoint( px+dx, -dx, dz );
    long p8  = _3D_NewPoint( px+dx, -dx, -dz );
    long p9  = _3D_NewPoint( -dx, py+dx, dz );
    long p10 = _3D_NewPoint( -dx, py+dx, -dz );
    long p11 = _3D_NewPoint( px+dx, py+dx, dz );
    long p12 = _3D_NewPoint( px+dx, py+dx, -dz );

    long p13 = _3D_NewPoint( 0, 0, dz );
    long p14 = _3D_NewPoint( 0, 0, -dz );
    long p15 = _3D_NewPoint( px, 0, dz );
    long p16 = _3D_NewPoint( px, 0, -dz );
    long p17 = _3D_NewPoint( 0, py, dz );
    long p18 = _3D_NewPoint( 0, py, -dz );
    long p19 = _3D_NewPoint( px, py, dz );
    long p20 = _3D_NewPoint( px, py, -dz );

    double ds = 0.2;

    long p21 = _3D_NewPoint( px/2-ds, py/2+ds, ds );
    long p22 = _3D_NewPoint( px/2+ds, py/2+ds, ds );
    long p23 = _3D_NewPoint( px/2+ds, py/2+ds, -ds );
    long p24 = _3D_NewPoint( px/2-ds, py/2+ds, -ds );
    long p25 = _3D_NewPoint( px/2-ds, py/2-ds, ds );
    long p26 = _3D_NewPoint( px/2+ds, py/2-ds, ds );
    long p27 = _3D_NewPoint( px/2+ds, py/2-ds, -ds );
    long p28 = _3D_NewPoint( px/2-ds, py/2-ds, -ds );

    DWORD PalStart = 193;

    _3D_Triangle_SideTyp S = Singlesided;
    _3D_Triangle_Typ T;
    long G = 0;
    if ( WithZBuffer == True )
        T = GouraudShadedZBuf;
    else
        T = GouraudShaded;

    if ( ShowBitmap == True )
        if ( WithZBuffer == True )
            VESA3D_New3DTexQuad (  p1,  p2,  p3,  p4, TEX, Doublesided, TexturedZBuf );
        else
            VESA3D_New3DTexQuad (  p1,  p2,  p3,  p4, TEX, Doublesided, Textured );

    VESA3D_New3DQuadx(  p5,  p7,  p8,  p6, PalStart, S, T, G ); 
    VESA3D_New3DQuadx(  p9,  p5,  p6, p10, PalStart, S, T, G ); 
    VESA3D_New3DQuadx( p11,  p9, p10, p12, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p7, p11, p12,  p8, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p9, p17, p13,  p5, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p6, p14, p18, p10, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p9, p11, p19, p17, PalStart, S, T, G );
    VESA3D_New3DQuadx( p18, p20, p12, p10, PalStart, S, T, G );
    VESA3D_New3DQuadx( p19, p11,  p7, p15, PalStart, S, T, G );
    VESA3D_New3DQuadx( p16,  p8, p12, p20, PalStart, S, T, G );
    VESA3D_New3DQuadx( p13, p15,  p7,  p5, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p6,  p8, p16, p14, PalStart, S, T, G );
    VESA3D_New3DQuadx( p17, p19,  p2,  p1, PalStart, S, T, G );
    VESA3D_New3DQuadx( p13, p17,  p1,  p4, PalStart, S, T, G );
    VESA3D_New3DQuadx( p19, p15,  p3,  p2, PalStart, S, T, G );
    VESA3D_New3DQuadx( p15, p13,  p4,  p3, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p1,  p2, p20, p18, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p2,  p3, p16, p20, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p3,  p4, p14, p16, PalStart, S, T, G );
    VESA3D_New3DQuadx(  p4,  p1, p18, p14, PalStart, S, T, G );

    if ( WithZBuffer == True ) {
        VESA3D_New3DQuad( p24, p23, p22, p21, PalStart, Singlesided, GouraudShadedZBuf );
        VESA3D_New3DQuad( p21, p22, p26, p25, PalStart, Singlesided, GouraudShadedZBuf );
        VESA3D_New3DQuad( p22, p23, p27, p26, PalStart, Singlesided, GouraudShadedZBuf );
        VESA3D_New3DQuad( p23, p24, p28, p27, PalStart, Singlesided, GouraudShadedZBuf );
        VESA3D_New3DQuad( p24, p21, p25, p28, PalStart, Singlesided, GouraudShadedZBuf );
        VESA3D_New3DQuad( p25, p26, p27, p28, PalStart, Singlesided, GouraudShadedZBuf );
    };
    
    double MaxDist = sqrt( (px/2+dx)*(px/2+dx) + (py/2+dx)*(py/2+dx) + dz*dz );
    double Rad;
 
    if ( px > py ) {
        Rad = 3 * px;
    }
    else {
        Rad = 3 * py ;
    };

    int N = 0;
    VESA3D_ShadeSub = Rad - MaxDist;
    VESA3D_ShadeMul = 46 / ( 2 * MaxDist );
    
    long *StarsX = new long [300];
    long *StarsY = new long [300];
    char *StarsC = new char [300];

    _3D_SetAlpha( 1.57 );
    _3D_SetPhi( 0 );
    _3D_SetGamma( 0 );
    _3D_SetOmega( 0 );
    _3D_SetEye( 0, 0, 0 );

    for ( i = 0; i < 300; i++ ) {
        StarsX[i] = MISC_Random(VESA_MaxX);
        StarsY[i] = MISC_Random(VESA_MaxY);
        StarsC[i] = char( 240+MISC_Random(15) );
    };

    double A = 0, B = 1.57, C = 0, D = 0;

    _3D_ScreenDistance = Distance;

    _3D_Matrix SpecMatrix;
    _3D_InitMatrix_Rotate( SpecMatrix, sin(0.1), cos(0.1), sin(0.1), cos(0.1),
                           0, 1 ); 

    MOUSE_SMouseMoves Move;
    MOUSE_SMouseButtons Buttons;

    clock_t Start = clock();

    while( _bios_keybrd( _KEYBRD_READY )==0 ) {
        N++;
        if ( WithZBuffer == True )
            _3D_RotatePointsForward( p21, p28, px/2, py/2, 0, SpecMatrix );
        if ( MouseControl == True ) {
            MOUSE_GetMouseMoves( Move );
            MOUSE_GetButtonState( Buttons );
            A += double(Move.MoveDX) / 100;
            B += double(Move.MoveDY) / 100;
            if ( Buttons.RightButton == True ) {
                _3D_ScreenDistance -= 0.1;
                if ( _3D_ScreenDistance < MinZoom )
                    _3D_ScreenDistance = MinZoom;
            };
            if ( Buttons.LeftButton == True ) {
                _3D_ScreenDistance += 0.1;
                if ( _3D_ScreenDistance > MaxZoom )
                    _3D_ScreenDistance = MaxZoom;
            };
        } 
        else { 
            D += 0.1; 
            A += 0.02 + 0.02 * sin( A + D );
            B += 0.04 + 0.04 * sin( B + D );
            C += 0.06 + 0.06 * sin( C + D );
        };
        _3D_SetPhi( A );
        _3D_SetGamma( B );
        _3D_SetOmega( C );
        _3D_SetEye_Globe( px/2, py/2, 0, Rad );
        _3D_InitRotation();
        _3D_ProjectPoints();
        _3D_PrepareObjects();
        if ( WithZBuffer == True ) {
            VESA3D_ClearZBuffer();
            //_3D_SortObjectsZ();
        }
        else
            _3D_SortObjects();
        VESA3D_FlipPageWrite();
        VESA_ClearScreen8B( 0, VESA3D_WriteSelector );
        for ( i = 0; i < 300; i++ ) {
            VESA_Pix8BC( StarsX[i], StarsY[i], StarsC[i], VESA3D_WriteSelector );
        };
        _3D_DrawObjects();
        VESA3D_FlipPageShow();
        if ( WaitVRetrace == True )
            VESA_SyncDisplay();
    };

    clock_t End = clock();
           
    _bios_keybrd( _KEYBRD_READ );

    VESA_SetTextMode();
    ShowTitle();
    printf( "\n" );
    printf( "%f Frames per second.\n", double( N ) / ( End-Start ) * CLOCKS_PER_SEC );

    VESA_Exit(); 
};

