/************************************************
 * packetio.c : a sliding window packet protocol
 *
 * Copyright (C) 1992 Jouni Leppjrvi
 ************************************************/

#include <string.h>

#include "system.h"
#include "bitcode.h"
#include "packetio.h"

#define CH_START '@'        /* packet start mark */

#define _P_EXPECT    0x00   /* response packet */
#define _P_DATA_FIX  0x01   /* fixed length data packet */
#define _P_DATA_VAR  0x02   /* variable length data packet */
#define _P_DATA_MK   0x03   /* variable length, 'marked' data packet */

#define P_EXPECT  _P_EXPECT /* PacketFlush() only sees these */
#define P_DATA    _P_DATA_VAR
#define P_DATA_MK _P_DATA_MK

#define W_START  0          /* ReadPacket() states */
#define W_TYPE   1
#define W_HEADER 2
#define W_DATA   3

#define WINDOW   2          /* window size = 2 packets */
#define CSEQ (2*WINDOW)     /* packet sequence numbers are 0 - 3 */

#define MAX_WAIT (PACKET_INTERVAL * 25)
#define MIN_WAIT (PACKET_INTERVAL *  2)

#define MAX_BC_PACKET (((PACKET_MAX/3) + 1) * 4)

/* use fixed packets with smaller packet sizes than FIXLIMIT */

#define FIXLIMIT 128

typedef struct
{
    unsigned type;
    unsigned seq;
    unsigned chk;
    unsigned size;
}   HEADER;

#define toasc(c) ((c) + 32)
#define toval(c) ((c) - 32)
#define isasc(c) ((c) >= 32 && (c) <= 97)
#define ptype(c) ((toval(c) >> 4) & 0x03)

/* protocol parameters */

static int _fixedSize = 48;
static int _packetSize = 48;
static char _padString[8] = "\r";
static int _padLen = 0;
static int _hSize[4] = {2,2,4,4};
static int _bitCode = 0;
static unsigned char _localId = 0, _remoteId = 0;

#if SYS_PROTOS

static unsigned Chksum(unsigned id,HEADER *header,char *data);
static int BuildHeader(HEADER *header,char *data,unsigned char *buf);
static unsigned ExtractHeader(HEADER *header,unsigned char *buf);
static void SendPacket(HEADER *header,char *data);
static int ReadPacket(unsigned long mtime,HEADER *header,char *data);

#else

static unsigned Chksum();
static int BuildHeader();
static unsigned ExtractHeader();
static void SendPacket();
static int ReadPacket();

#endif

/* Chksum() calculates checksum for a packet */

static unsigned Chksum(id,header,data)
unsigned char id;
HEADER *header;
char *data;
{
  register unsigned sum;
  register unsigned char *p, *end;

  sum = header -> type + header -> seq + header -> size + id;

  if (data != NULL)
  {
   p = (unsigned char *) data;
   end = (unsigned char *) &data[header -> size];

   for (; p < end ;p++)
    sum += *p;
  }

  return(~sum);
}

/* BuildHeader() builds the packet header
 *
 * returns : header size
 */

static int BuildHeader(header,data,buf)
HEADER *header;
char *data;
unsigned char *buf;
{
 int i;

 header -> chk = Chksum(_localId,header,data);

 header -> size--;  /* 0 = 1 bytes, 0x3ff = 0x400 bytes */

 buf[0] = (header -> type << 4) | (header -> seq << 2) | (header -> chk & 0x03);
 buf[1] = (header -> chk >> 2) & 0x3f;
 buf[2] = ((header -> chk >> 4) & 0x30) | (header -> size & 0x0f);
 buf[3] = header -> size >> 4;

 header -> size++;

 for (i = 0; i < 4 ;i++)
  buf[i] = toasc(buf[i]);

 return(_hSize[header -> type]);
}

/* ExtractHeader() extracts packet headers build by BuildHeader
 * Packet type must already be present in the struct at call time !
 *
 * returns : checksum mask
 */

static unsigned ExtractHeader(header,buf)
HEADER *header;
unsigned char *buf;
{
 int i;

 for (i = 0; i < 4 ;i++)
  buf[i] = toval(buf[i]);

 header -> seq = (buf[0] >> 2) & 0x03;
 header -> chk = (buf[0] & 0x03) | (buf[1] << 2);

 if (header -> type == _P_EXPECT)
 {
  header -> size = 0;
  return(0xff);
 }
 else if (header -> type == _P_DATA_FIX)
 {
  header -> size = _fixedSize;
  return(0xff);
 }
 else if (header -> type == _P_DATA_VAR || header -> type == _P_DATA_MK)
 {
  header -> chk |= (buf[2] & 0x30) << 4;
  header -> size = (buf[2] & 0x0f) | (buf[3] << 4);
  header -> size++; /* 0 = 1 bytes, 0x3ff = 0x400 bytes */
  return(0x3ff);
 }
 return(0xffff);
}

/* SendPacket() writes a packet on the communication line */

static void SendPacket(header,data)
HEADER *header;
char *data;
{
 static unsigned char hbuf[5] = {CH_START};
 HEADER _header;
 char _data[MAX_BC_PACKET];
 int hSize;

 _header = *header;

 if (_bitCode && (_header.type == _P_DATA_VAR || _header.type == _P_DATA_MK))
 {
    _header.size = BitCode(_bitCode,data,_data,_header.size);
    data = _data;
 }

 if (_header.type == _P_DATA_VAR && _header.size == _fixedSize)
    _header.type = _P_DATA_FIX;

 hSize = BuildHeader(&_header,data,&hbuf[1]) + 1;
 LineWrite((char *) hbuf,hSize);

 switch(_header.type)
 {
  case _P_DATA_FIX:
  case _P_DATA_VAR:
  case _P_DATA_MK:
   LineWrite(data,_header.size);
 }

 if (_padLen)
  LineWrite(_padString,_padLen);
}


/* ReadPacket() reads packets from the communication line */

static int ReadPacket(mtime,header,data)
unsigned long mtime;
HEADER *header;
char *data;
{
 static HEADER _header;
 static char _data[MAX_BC_PACKET];
 static unsigned long abortTime;
 static int state = W_START, nbytes, rdbytes, mask;
 int i;

 switch(state)
 {
  case W_START:
   abortTime = mtime + 1000L;
   while(LineRead(_data,1))
   {
    if (_data[0] == CH_START)
    {
     state = W_TYPE;
     break;
    }
   }

   if (state != W_TYPE) /* falltrough */
    break;

  case W_TYPE:
   if (LineRead(_data,1))
   {
    _header.type = ptype(_data[0]);
    state = W_HEADER;
    rdbytes = 0;
    nbytes = _hSize[_header.type] - 1;  /* falltrough */
   }
   else
    break;

  case W_HEADER:
   rdbytes += LineRead(&_data[rdbytes + 1],nbytes - rdbytes);
   if (rdbytes == nbytes)
   {
    /* header should be printable characters only */

    for (i = 0; i < _hSize[_header.type] ;i++)
     if (!isasc(_data[i]))
     {
      state = W_START;
      return(0);
     }

    mask = ExtractHeader(&_header,(unsigned char *) _data);

    if (_header.size > PACKET_MAX)
    {
     state = W_START;
     return(0);
    }

    switch(_header.type)
    {
     case _P_DATA_FIX:
     case _P_DATA_VAR:
     case _P_DATA_MK:
      state = W_DATA;
      rdbytes = 0;
      nbytes = _header.size;    /* falltrough */
     break;

     case _P_EXPECT:
      state = W_START;
      if (_header.chk == (Chksum(_remoteId,&_header,NULL) & mask))
      {
       *header = _header;
       return(1);
      }
      return(0);

     default:           /* never, never .. */
      state = W_START;
      return(0);
    }
   }
   else
    break;

  case W_DATA:
   i = LineRead(&_data[rdbytes],nbytes - rdbytes);

   /* The following should abort a packet
    * if a line error has either corrupted
    * the packet size in the header or
    * caused loss of bytes.
    */

   if (!i)
    if (abortTime < mtime)
    {
     state = W_START;
     return(0);
    }

   rdbytes += i;
   if (rdbytes == nbytes)
   {
    state = W_START;

    i = _header.chk == (Chksum(_remoteId,&_header,_data) & mask);

    if (_bitCode)
        _header.size = BitDecode(_bitCode,_data,_data,_header.size);

    if (_header.type == _P_DATA_FIX)
     _header.type = _P_DATA_VAR;

    if (i)
    {
     *header = _header;
     memcpy(data,_data,_header.size);
     return(1);
    }
    return(0);
   }
  break;
 }
 return(0);
}

/* PacketFlush() reads and writes data packets on the
 * communication line. This function should be called
 * periodically with the calling interval of the time
 * it takes to actually write out packet size bytes.
 *
 * PacketFlush() should realize a generic sliding window
 * protocol. Packet format is, however, hardwired for
 * window size 2. This is assumed to be the optimum for
 * interactive use.
 *
 * mtime : real time in milliseconds since program start
 *
 * returns : bitwise or of
 *           PACKET_IN   : read packet(s)
 *           PACKET_QOUT : packets queued for output
 *
 * The return value can be used at a higher level to
 * determine whether the protocol is still alive.
 *
 * The functions LineRead() and LineWrite() are
 * used to read/write the communication line.
 *
 * int LineRead(char *buf,int nbytes)
 *   Reads at max. nbytes bytes from the line into buf.
 * returns : bytes actually read, 0 if none
 *
 * void LineWrite(char *buf,int nbytes)
 *  Writes nbytes bytes from buf to the line.
 *
 * The functions DataRead() and DataWrite() are used
 * to retrive/deliver data written/read on the
 * communication line.
 *
 * int DataRead(char *buf,int nbytes,int *fmark)
 *  Gets at max. nbytes to be sent to buf.
 *  If *fmark !0 after call, the data is sent as a
 *  'marked' packet.
 * returns : bytes actually read, 0 if none.
 *
 * void DataWrite(char *buf,int nbytes,int fmark)
 *  Delivers nbytes of data in buf (as apropriate).
 *  fmark signals a 'marked' packet.
 *
 * The above functions may not block.
 * Also none of them has error return. Errors must
 * be reported to the user some other way, this module
 * does not want to know about them.
 */

/*
 * PacketFlush() statics moved to module globals
 * to enable protocol reset with PacketInit().
 */

static struct
 {
  HEADER header;
  char busy;
  unsigned cResend;
  unsigned long tSend;
  unsigned long tResend;
  char data[PACKET_MAX];
 } in[WINDOW],out[WINDOW];
static unsigned expect = 0, send = 0, _send = 0;
static unsigned long aWait, sNext;

int PacketFlush(mtime)
unsigned long mtime;
{

 HEADER hExp;
 int i, j, k, l, pRx = 0, dRx = 0,qTx = 0;
 unsigned long u;

 /*
  * fill free slots with incoming packets (if available)
  */

 for (k = 1; k ;)
  for (i = 0, k = 0; i < WINDOW ;i++)
  {
   if (!in[i].busy) /* free slot ? */
   {
    if (!ReadPacket(mtime,&in[i].header,in[i].data))    /* packet available ? */
     break;

    k = 1;      /* loop once more after this */

    pRx = 1;    /* take note : packet(s) received */

    if (in[i].header.type == P_DATA || in[i].header.type == P_DATA_MK)
    {
     /*
      * incoming DATA packet
      *
      * store it in the free slot if
      *  - seq is within widow
      *  - this is not already in a buffer
      *
      */

     dRx = 1;   /* take note : data packet(s) received */

     for (j = 0; j < WINDOW ;j++)
      if (in[i].header.seq == (expect + j) % CSEQ)
      {
       for (l = 0; l < WINDOW ;l++)
        if (in[l].busy && in[l].header.seq == in[i].header.seq)
         break;

       if (l == WINDOW)
        in[i].busy = 1;
      }
    }
    else
    {
     /*
      * incoming EXPECT-packet
      *
      * free outgoing packet slots up to the next
      * expected packet, prepare to send the expected
      * packet
      *
      */

     for (j = send; j != in[i].header.seq ;j = (j + 1) % CSEQ)
      for (l = 0; l < WINDOW ;l++)
       if (out[l].busy && out[l].header.seq == j)
       {
        out[l].busy = 0;

        /*
         * Compute actual ack (EXPECT)
         * wait time. Make it the
         * assumed ack wait time if
         * it is greater than the previous
         * one, else decrement the assumed
         * time slightly. This should result
         * in the assumed time going up
         * fast, but down slower.
         */

        u = mtime - out[l].tSend;
        if (u > aWait)
            aWait = u;
        else
            aWait -= aWait >> 3;
       }

     /* do not wait too long though .. */

     if (aWait > MAX_WAIT)
        aWait = MAX_WAIT;

     send = in[i].header.seq;
    }
   }
  }

 /* deliver incoming data - scan slots until all clear */

 for (j = 1; j ;)
  for (i = 0, j = 0; i < WINDOW ;i++)
  {
   if (in[i].busy && in[i].header.seq == expect)
   {
    DataWrite(in[i].data,in[i].header.size,in[i].header.type == P_DATA_MK);
    in[i].busy = 0;
    expect = (expect + 1) % CSEQ;
    j = 1;
   }
  }

 /* if data packet(s) received, tell which seq we would like to see next */

 if (dRx)
 {
  hExp.type = P_EXPECT;
  hExp.seq = expect;
  hExp.size = 0;
  SendPacket(&hExp,NULL);
 }

 /* get more data to send if there are empty slots */

 if (sNext <= mtime)
  for (i = 0; i < WINDOW ;i++)
  {
   if (!out[i].busy)
   {
    k = DataRead(out[i].data,_packetSize,&j);
    if (k > 0 && k <= PACKET_MAX)
    {
     /*
      * Send next only after about
      * a half of this one is out.
      */

     sNext = ((long) k * (long)(PACKET_INTERVAL / 2)) / (long) _packetSize;
     sNext += mtime;

     /* build a packet - note that _send goes ahead of send */

     out[i].header.type = j ? P_DATA_MK : P_DATA;
     out[i].header.seq = _send;
     out[i].header.size = k;

     out[i].busy = 1;
     out[i].cResend = 0;
     out[i].tSend   = mtime;
     out[i].tResend = mtime;    /* send immediately */

     _send = (_send + 1) % CSEQ;
    }
    break;  /* only one at a time */
   }
  }

 /* send pending packets (if any) in original order */

 for (j = 0; j < CSEQ ;j++)
  for (i = 0; i < WINDOW ;i++)
  {
   if (out[i].busy)
   {
    qTx = 1;    /* take note : outgoing packet(s) in queue */
    if (out[i].header.seq == j && out[i].tResend <= mtime)
    {
     /*
      * Use aWait in determining the resend time.
      * If, however, there has been no response
      * after two resends (=theree sends), wait
      * for longer.
      */

     if (out[i].cResend <= 2)
        out[i].tResend = mtime + aWait + MIN_WAIT;
     else out[i].tResend = mtime + MAX_WAIT;

     SendPacket(&out[i].header,out[i].data);

     out[i].cResend++;
    }
   }
  }

 return((qTx ? PACKET_QOUT : 0) | (pRx ? PACKET_IN : 0));
}


/*
 * PacketSetParams() sets the packet protocol parameters
 *
 * size    : packet size
 * bitcode : bit coding scheme
 * pad     : padding string to use between packets, NULL = none
 *
 * returns : 0  : ok
 *           !0 : error (not currently)
 */

int PacketSetParams(size,bitcode,pad)
int size, bitcode;
char *pad;
{
 _bitCode = bitcode;
 _packetSize = size > PACKET_MAX ? PACKET_MAX : size;
 _fixedSize = _packetSize;

 if (_bitCode == (M_BIT5 | M_BIT7))
 {
  _packetSize = (_packetSize / 4) * 3;
  _fixedSize  = (_packetSize / 3) * 4;
 }
 else if (_bitCode & (M_BIT5 | M_BIT7))
 {
  _packetSize = (_packetSize / 8) * 7;
  _fixedSize  = (_packetSize / 7) * 8;
 }

 _fixedSize = _fixedSize < FIXLIMIT ? _fixedSize : -1;

 if (pad != NULL)
 {
  strncpy(_padString,pad,sizeof(_padString));
  _padLen = strlen(pad);
  _padLen = _padLen < sizeof(_padString) ? _padLen : sizeof(_padString);
 }

 return(0);
}


/*
 * PacketInit() initializes the packet protocol.
 *
 * returns : fixed packet size (before bitcoding)
 */

int PacketInit()
{
 int i;

 expect = 0;
 send = 0;
 _send = 0;

 aWait = 0L;
 sNext = 0L;

 _localId = _remoteId = 0;

 for (i = 0; i < WINDOW ;i++)
 {
    in[i].busy = 0;
    out[i].busy = 0;
 }

 return(_packetSize);
}

/*
 * PacketSetId() sets local / remote id's used in checksum calculation.
 *
 * local  : local id
 * remote : remote id
 *
 * returns : nothing
 */

void PacketSetId(local,remote)
unsigned char local, remote;
{
    _localId  = local;
    _remoteId = remote;
}
