/*******************************************
 * mux.c : channel multiplexor/demultiplexor
 *
 * Copyright (C) 1992, Jouni Leppjrvi
 *******************************************/

#include <string.h>

#include "system.h"
#include "usrmsg.h"
#include "packetio.h"
#include "control.h"
#include "channel.h"
#include "file.h"
#include "shell.h"
#include "mux.h"

/*
 * Absolute maximum number of channels
 * is 95, if we want to keep in ascii
 * and describe channel in one byte.
 */

#define MUX_MAXCHAN 95

#define toasc(v) ((v) + 32)
#define toval(c) ((c) - 32)

typedef struct
{
    char *name;
    struct
    {
#if SYS_PROTOS

        int  (*open)(int ch);
        void (*close)(int id);
        int  (*read)(int id,char *buf,int nbytes);
        void (*write)(int id,char *buf,int nbytes);

#else
        int  (*open)();
        void (*close)();
        int  (*read)();
        void (*write)();
#endif
    }   func;

}   CH_TYPE;

#ifdef SYS_DOS

CH_TYPE _localType[] = {
"CONTROL"   ,{NULL,CtrlClose,CtrlRead,CtrlWrite},
"FILE_CTRL" ,{FileOpenCtrl,FileCloseCtrl,FileReadCtrl,FileWriteCtrl},
"FILE_DATA" ,{FileOpenData,FileCloseData,FileReadData,FileWriteData},
"SHELL"     ,{TtyOpen,TtyClose,TtyRead,TtyWrite},
"SHELL_CTRL",{TtyOpenCtrl,TtyCloseCtrl,TtyReadCtrl,TtyWriteCtrl},
NULL        ,{NULL,NULL,NULL,NULL}};

CH_TYPE _remoteType[] = {
NULL,{NULL,NULL,NULL,NULL}};

#else

CH_TYPE _localType[] = {
"CONTROL"   ,{NULL,CtrlClose,CtrlRead,CtrlWrite},
NULL        ,{NULL,NULL,NULL,NULL}};

CH_TYPE _remoteType[] = {
"FILE_CTRL" ,{FileOpenCtrl,FileCloseCtrl,FileReadCtrl,FileWriteCtrl},
"FILE_DATA" ,{FileOpenData,FileCloseData,FileReadData,FileWriteData},
"SHELL"     ,{ShellOpen,ShellClose,ShellRead,ShellWrite},
"SHELL_CTRL",{ShellOpenCtrl,ShellCloseCtrl,ShellReadCtrl,ShellWriteCtrl},
NULL        ,{NULL,NULL,NULL,NULL}};

#endif

static struct
{
    int id;                 /* ch id for the client  */
    CH_TYPE *type;          /* -> _local/_remoteType */
    struct
    {
        char res;           /* flag : entry reserved */
        char act;           /* flag : active (open)  */
    }   flg;

}   _chList[MUX_MAXCHAN] = {
    0,&_localType[0],{1,1}};        /* control channel always exists */


static int _highPrioCh;             /* priorized ch */
static int _maxCh;                  /* highest ch # so far opened */
static int _outCh, _inCh;           /* current channel out/in */

/*
 * MuxInit() initializes the module
 */

void MuxInit()
{
    _highPrioCh = 0;
    _maxCh      = 0;
    _outCh      = 0;
    _inCh       = 0;
}

/*
 * _MuxOpen() performs channel open with explicit channel id
 *
 * returns :  0 : ok
 *           <0 : error
 */

int _MuxOpen(type,ch,name)
int type, ch;
char *name;
{
    CH_TYPE *p;

    p = (type == MUX_OPEN_LOCAL ? &_localType[1] : &_remoteType[0]);
    for (; p -> name != NULL ;p++)
    {
        if (!stricmp(p -> name,name))
        {
            if (!_chList[ch].flg.act && !_chList[ch].flg.res)
            {

                /*
                 * Check the function pointers.
                 * (just in case ..)
                 */

                if (p -> func.open  != NULL ||
                    p -> func.close != NULL ||
                    p -> func.read  != NULL ||
                    p -> func.write != NULL)
                {
                    _chList[ch].flg.res = 1;
                    _chList[ch].flg.act = 0;
                    _chList[ch].type    = p;

                    _chList[ch].id = (*(p -> func.open))(ch);


                    if (_chList[ch].id >= 0)
                    {
                            _chList[ch].flg.act = 1;

                            if (ch > _maxCh)
                                _maxCh = ch;

                            return(0);
                    }

                    _chList[ch].flg.res = 0;
                }
                else
                    MsgDisplay(
                        MSG_ERROR,
                        "unexpected : type %s doesn't support all functions",
                        name);
            }
            else
            {
                ChanClose(ch);
                MsgDisplay(MSG_ERROR,"inconsistent channel %d closed",ch);
                return(-1);
            }
        }
    }

    MsgDisplay(MSG_ERROR,"local channel open failed [%d %s]",ch,name);

    return(-1);
}

/*
 * MuxOpen() performs local channel open
 * with a given channel type.
 *
 * returns : channel descriptor, 0< on error
 */

int MuxOpen(name)
char *name;
{
    int i;

    for (i = 0; i < MUX_MAXCHAN ;i++)
    {
        if (!_chList[i].flg.res)
        {
            if (!_MuxOpen(MUX_OPEN_LOCAL,i,name))
                return(i);
            return(-1);
        }
    }

    MsgError("MuxOpen","no channels left");

    return(-1);
}

/*
 * MuxClose() performs local channel close for
 * a channel by channel descriptor.
 */

void MuxClose(ch)
int ch;
{
    int i;

    if (ch > 0 && ch < MUX_MAXCHAN &&
        _chList[ch].flg.act && _chList[ch].flg.res)
    {
        if (ch == _highPrioCh)
            _highPrioCh = 0;

        _chList[ch].flg.act = 0;

        (*(_chList[ch].type -> func.close))(_chList[ch].id);

        _chList[ch].flg.res = 0;

        return;
    }
    else if (ch == 0)
    {
        for (i = 1; i < MUX_MAXCHAN ;i++)
            if (_chList[i].flg.act)
                MuxClose(i);

        (*(_chList[0].type -> func.close))(_chList[0].id);
        return;
    }

    /* This is usually not an error, but a result from
     * closing a channel currently sending data, hence,
     * this message is in the debug-category ...
     */

    MsgDisplay(MSG_DEBUG,"local channel close failed for %d",ch);
}

/* MuxPriorize sets the locally priorized
 * channel by descriptor. If ch == 0 then
 * no channel is to be priorized.
 *
 * returns : previous priorized channel
 */

int MuxPriorize(ch)
int ch;
{
    int ret = _highPrioCh;

    if (ch < 0 || ch > _maxCh || !_chList[ch].flg.act)
        MsgDisplay(MSG_ERROR,"attempt to priorize a non-existent channel %d",ch);
    else
        _highPrioCh = ch;

    return(ret);
}


/*
 * DataRead() and DataWrite() for packetio to call
 */

int DataRead(buf,nbytes,fmark)
char *buf;
int nbytes, *fmark;
{
    int i, j, ch, rdbytes;

    ch = -1;

    if (nbytes >= PACKET_MAX)
        nbytes = PACKET_MAX - 1;

    /*
     * The following realizes a simple
     * priority scheme : first check
     * the control channel (0), then
     * the priorized channel, then all
     * other channels (round robin).
     */

    rdbytes = (*(_chList[0].type -> func.read))(_chList[0].id,buf,PACKET_MAX - 1);
    if (rdbytes > 0)
        ch = 0;
    else
    {
        /*
         * critical : ch must be saved prior to calling
         * the read function to allow it to close itself
         * (closing clears _highPrioCh)
         */

        if (_highPrioCh > 0 && _chList[_highPrioCh].flg.act)
        {
            ch = _highPrioCh;

            rdbytes = nbytes;
            rdbytes = rdbytes > (PACKET_MAX - 1) ? (PACKET_MAX - 1) : rdbytes;
            rdbytes = (*(_chList[ch].type -> func.read))(_chList[ch].id,buf,rdbytes);

            if (rdbytes <= 0)
                ch = -1;
        }

        if (ch < 0)
        {
            for (j = 1, i = _outCh + 1; j <= _maxCh ;j++, i++)
            {
                i = (i > _maxCh ? 1 : i);

                if (i != _highPrioCh && _chList[i].flg.act)
                {
                    rdbytes = (*(_chList[i].type -> func.read))(_chList[i].id,buf,nbytes);
                    if (rdbytes > 0)
                    {
                        ch = i;
                        break;
                    }
                }
            }
        }
    }

    /*
     * If there was a data to send
     * send it :
     *  - as a regular packet, if
     *    same channel as last time
     *
     *  - as a marked packet, if
     *    channel changed from last
     *    time, append one byte to
     *    the marked packet to specify
     *    the channel
     */

    if (ch != -1)
    {
        if (ch == _outCh)
            *fmark = 0;
        else
        {
            _outCh = ch;
            *fmark = 1;
            buf[rdbytes++] = toasc(_outCh);
        }

        return(rdbytes);
    }

    return(0);
}

void DataWrite(buf,nbytes,fmark)
char *buf;
int nbytes, fmark;
{
    /*
     * Deliver incoming packets.
     * If a packet is marked, then
     * the last byte specifies a
     * channel to switch into.
     *
     */

    if (fmark)
    {
        nbytes--;
        _inCh = toval(buf[nbytes]);

        if (_inCh < 0 || _inCh >= MUX_MAXCHAN)
        {
            MsgError("DataWrite","unexpected : invalid channel descriptor");
            _inCh = -1; /* invalidate */
            return;
        }
    }

    if (_inCh == -1)
        return;
    else if (!_chList[_inCh].flg.act)
    {
        CtrlSendCmd("CLOSE %d",_inCh);
        MsgDisplay(MSG_DEBUG,"ghost channel %d closed",_inCh);
        _inCh = -1; /* invalidate */
        return;
    }

    (*(_chList[_inCh].type -> func.write))(_chList[_inCh].id,buf,nbytes);
}
