/***************************************
 * file.c : pcucp file transfer
 *
 * Copyright (C) 1992, Jouni Leppjrvi
 ***************************************/

#include <stdio.h>
#include <string.h>
#include <time.h>

#define SYS_INCL_STDLIB
#define SYS_INCL_FILEIO
#define SYS_INCL_UTIME
#include "system.h"
#include "usrmsg.h"
#include "packetio.h"
#include "channel.h"
#include "dirread.h"
#include "fnconv.h"
#include "setftime.h"
#include "file.h"

#define FILE_CHKINTERVAL 30

#define FILE_BUFSIZE 0x2000

#define FILE_SIDLE  0
#define FILE_SWAIT  1
#define FILE_SSEEK  2
#define FILE_SBUSY  3
#define FILE_SDONE  4
#define FILE_SERR  -1

#define FILE_RSEND  0
#define FILE_RREC   1

#if SYS_PROTOS

static void FileReport(int which);
static int FileGetNext(void);
static void FileSetTime(void);
static void FileRename(char *file);
static void FileCmdPut(char *name,long size,long ftime);

#else

static void FileReport();
static int FileGetNext();
static void FileSetTime();
static void FileRename();
static void FileCmdPut();

#endif

static struct
{
    int chCtrl, chData;

    struct
    {
        int state;
        int fd;
        int nbytes;
        long todo, done;
        long time;
        long start, trans;
        char dir[SYS_MAXPATH];
        char name[SYS_MAXFILE];
        char buf[FILE_BUFSIZE + PACKET_MAX];
    }   wr;

    struct
    {
        int state;
        int fd;
        int nbytes, index;
        long todo, done;
        long time;
        long start, trans;
        char dir[SYS_MAXPATH];
        char name[SYS_MAXFILE];
        char buf[FILE_BUFSIZE];
    }   rd;


}   _glb = {-1,-1};


static char _delim[] = {SYS_PATHDELIM,0};

static void FileReport(which)
int which;
{
    long l;
    char buf[32];

    if (which == FILE_RSEND)
    {
        if (_glb.rd.done == _glb.rd.todo)
        {
            time(&l);
            l -= _glb.rd.start;
            if (l < 10)
                sprintf(buf,"n/a");
            else
                sprintf(buf,"%ld bytes/s",_glb.rd.trans / l);

            MsgDisplay(
                MSG_USER,
                "send : complete : %s : %ld bytes : %s",
                _glb.rd.name,
                _glb.rd.done,buf);
        }
        else
        {
            MsgDisplay(
                MSG_USER,
                "send : %s : %ld/%ld bytes",
                _glb.rd.name,
                _glb.rd.done,_glb.rd.todo);
        }
    }
    else
    {
        if (_glb.wr.todo == _glb.wr.done)
        {
            time(&l);
            l -= _glb.wr.start;

            if (l < 10L)
                sprintf(buf,"n/a");
            else
                sprintf(buf,"%ld bytes/s",_glb.wr.trans / l);

            MsgDisplay(
                MSG_USER,
                "receive : complete : %s : %ld bytes : %s",
                _glb.wr.name,
                _glb.wr.done,buf);
        }
        else
        {
            MsgDisplay(
                MSG_USER,
                "receive : %s : %ld/%ld bytes",
                _glb.wr.name,
                _glb.wr.done,_glb.wr.todo);
        }
    }
}

static int FileGetNext()
{
    int ret;
    struct stat sstat;

    ret = DirRead(_glb.rd.dir,_glb.rd.name,sizeof(_glb.rd.name));
    while(!ret)
    {
        strcpy(_glb.rd.buf,_glb.rd.dir);
        strcat(_glb.rd.buf,_delim);
        strcat(_glb.rd.buf,_glb.rd.name);

        if (stat(_glb.rd.buf,&sstat))
        {
            DirDone();
            MsgDisplay(MSG_ERROR,"cannot stat %s",_glb.rd.buf);
            return(-1);
        }

        /*
         * Sun seems to create .nfsXXXXX files
         * (at unlink time ?). So, drop all
         * files beginning with a dot.
         */

        if (_glb.rd.name[0] != '.' && (sstat.st_mode & S_IFMT) == S_IFREG)
            break;

        ret = DirRead(NULL,_glb.rd.name,sizeof(_glb.rd.name));
    }

    DirDone();

    if (ret == 1)
        return(1);
    else if (ret < 0)
    {
        MsgDisplay(MSG_ERROR,"error reading directory %s",_glb.rd.dir);
        return(-1);
    }

    /* initialize for new file */

    _glb.rd.todo   = sstat.st_size;
    _glb.rd.done   = 0;
    _glb.rd.time   = sstat.st_mtime;
    _glb.rd.nbytes = 0;
    _glb.rd.index  = 0;

    _glb.rd.fd = open(_glb.rd.buf,O_BINARY | O_RDONLY);
    if (_glb.rd.fd < 0)
    {
        MsgDisplay(MSG_ERROR,"cannot open %s",_glb.rd.buf);
        return(-1);
    }

    return(0);
}

static void FileSetTime()
{
    strcpy(_glb.wr.buf,_glb.wr.dir);
    strcat(_glb.wr.buf,_delim);
    strcat(_glb.wr.buf,_glb.wr.name);

    if (SetFileTime(_glb.wr.buf,_glb.wr.time))
        MsgError("FileSetTime","unexpected : cannot set file time(s)");
}


int FileReadCtrl(id,buf,nbytes)
int id;
char *buf;
int nbytes;
{
    static long _t;
    long t;
    int ret;

    switch(_glb.rd.state)
    {
        case FILE_SIDLE:

            if (_t > time(&t))
                break;

            ret = FileGetNext();
            if (ret < 0)
                _glb.rd.state = FILE_SERR;
            else if (!ret)
            {
                _glb.rd.state = FILE_SWAIT;

                sprintf(
                    buf,
                    "PUT %s %ld %ld",
                    _glb.rd.name,
                    _glb.rd.todo,
                    _glb.rd.time);

                    return(strlen(buf));
            }

            _t = t + FILE_CHKINTERVAL;

        break;

        case FILE_SSEEK:
            if (_glb.rd.done)
            {
                if (lseek(_glb.rd.fd,_glb.rd.done,SEEK_SET) < 0)
                {
                    MsgError("FileReadCtrl","unexpected : seek failed");
                    close(_glb.rd.fd);
                    _glb.rd.state = FILE_SERR;
                    break;
                }
            }

            _glb.rd.start = time(NULL);
            _glb.rd.trans = _glb.rd.todo - _glb.rd.done;

            if (_glb.rd.done >= _glb.rd.todo)
            {
                _glb.rd.state = FILE_SDONE;

                FileReport(FILE_RSEND);

                if (_glb.rd.done > _glb.rd.todo)
                    MsgError("FileReadCtrl","unexpected : done > todo");
            }
            else
                _glb.rd.state = FILE_SBUSY;
        break;

        case FILE_SDONE:
            close(_glb.rd.fd);
            _glb.rd.state = FILE_SIDLE;

            strcpy(_glb.rd.buf,_glb.rd.dir);
            strcat(_glb.rd.buf,_delim);
            strcat(_glb.rd.buf,_glb.rd.name);

            if (unlink(_glb.rd.buf))
                MsgDisplay(MSG_ERROR,"send : cannot remove %s",_glb.rd.buf);
        break;

        case FILE_SERR:
            close(_glb.rd.fd);

            ChanClose(_glb.chCtrl);
            ChanClose(_glb.chData);

            MsgDisplay(MSG_USER,"send : file transfer aborted");
        break;
    }

    switch(_glb.wr.state)
    {
        case FILE_SSEEK:
            if (_glb.wr.done >= _glb.wr.todo)
            {
                _glb.wr.state = FILE_SIDLE;

                if (_glb.wr.done > _glb.wr.todo)
                {
                    _glb.wr.state = FILE_SERR;
                    MsgError("FileReadCtrl","unexpected : done > todo");
                }
            }
            else
                _glb.wr.state = FILE_SBUSY;

            sprintf(buf,"SEEK %ld",_glb.wr.done);

            FileReport(FILE_RREC);

            return(strlen(buf));

        case FILE_SERR:
            close(_glb.wr.fd);
            FileSetTime();

            ChanClose(_glb.chCtrl);
            ChanClose(_glb.chData);

            MsgDisplay(MSG_USER,"receive : file transfer aborted");
        break;
    }
    return(0);
}

static void FileRename(name)
char *name;
{
    int i;
    char buf[SYS_MAXPATH];

    strncpy(buf,name,sizeof(buf));
    buf[sizeof(buf) - 1] = '\0';

    i = strlen(buf);
    buf[i - 1] = '~';

    unlink(buf);
    if (rename(name,buf))
        MsgError("FileRename","rename failed");
    else
        MsgDisplay(MSG_USER,"receive : name collision, existing file renamed");
}

static void FileCmdPut(name,size,ftime)
char *name;
long size, ftime;
{
    struct stat sstat;
    int openFlgs;

    if (_glb.wr.state != FILE_SIDLE)
    {
        MsgDisplay(
            MSG_ERROR,
            "unexpected : file open out of phase : %d : %s",
            _glb.wr.state,name);
        _glb.wr.state = FILE_SERR;
        return;
    }

    _glb.wr.todo   = size;
    _glb.wr.time   = ftime;

    FnConv(name);
    strncpy(_glb.wr.name,name,sizeof(_glb.wr.name));
    _glb.wr.name[sizeof(_glb.wr.name) - 1] = '\0';

    strcpy(_glb.wr.buf,_glb.wr.dir);
    strcat(_glb.wr.buf,_delim);
    strcat(_glb.wr.buf,_glb.wr.name);

    /* does it exist already ? */

    if (access(_glb.wr.buf,0))
    {
        /* nope */

        _glb.wr.done = 0;
        openFlgs = O_CREAT | O_WRONLY | O_BINARY;
    }
    else
    {
        /* If the file exists, it is assumed
         * to be the same file if file times
         * equal and remote file is bigger
         * or equal in size.
         *
         * Transfer continues if same file,
         * otherwise the local file is truncated
         * and the entire file is transferred.
         */

        if (stat(_glb.wr.buf,&sstat) < 0)
        {
            MsgDisplay(MSG_ERROR,"cannot stat %s",_glb.wr.buf);
            return;
        }

       /*
        * DOS file times have only 2 s resolution !
        */

        if (sstat.st_size <= _glb.wr.todo && (_glb.wr.time & ~((long) 1)) == sstat.st_mtime)
        {
            _glb.wr.done = sstat.st_size;
            openFlgs = O_APPEND | O_WRONLY | O_BINARY;

        }
        else
        {
            _glb.wr.done = 0;
            openFlgs = O_CREAT | O_TRUNC | O_WRONLY | O_BINARY;
            FileRename(_glb.wr.buf);
        }
    }

    /* open the file with openFlgs */

    _glb.wr.fd = open(_glb.wr.buf,openFlgs,S_IWRITE | S_IREAD);
    if (_glb.wr.fd < 0)
    {
        MsgDisplay(MSG_ERROR,"cannot open %s",_glb.wr.buf);
        return;
    }

    /*
     * Setting state here triggers SEEK
     * packet sending in FileRead().
     */

    _glb.wr.nbytes = 0;
    _glb.wr.state  = FILE_SSEEK;
    _glb.wr.start  = time(NULL);
    _glb.wr.trans  = _glb.wr.todo - _glb.wr.done;
}

void FileWriteCtrl(id,buf,nbytes)
int id;
char *buf;
int nbytes;
{
    char sbuf[256];
    long size, time;

    buf[nbytes] = '\0';
    if (sscanf(buf,"%s ",sbuf) == 1)
    {
        if (!stricmp("PUT",sbuf))
        {
            if (sscanf(buf,"%*s %s %ld %ld",sbuf,&size,&time) == 3)
            {
                FileCmdPut(sbuf,size,time);
                return;
            }
        }
        else if (!stricmp("SEEK",sbuf))
        {
            if (sscanf(buf,"%*s %ld",&size) == 1)
            {
                _glb.rd.state = FILE_SSEEK;
                _glb.rd.done = size;
                return;
            }
        }
    }

    MsgDisplay(MSG_ERROR,"unexpected : file command [%s]",buf);
}

int FileReadData(id,buf,nbytes)
int id;
char *buf;
int nbytes;
{
    if (_glb.rd.state == FILE_SBUSY)
    {
        if (_glb.rd.nbytes > 0)
        {
            nbytes = nbytes > _glb.rd.nbytes ? _glb.rd.nbytes : nbytes;

            memcpy(
                buf,
                &_glb.rd.buf[_glb.rd.index],
                nbytes);

            _glb.rd.nbytes -= nbytes;
            _glb.rd.index  += nbytes;
            _glb.rd.done   += nbytes;

            return(nbytes);
        }

        _glb.rd.index  = 0;
        _glb.rd.nbytes = read(_glb.rd.fd,_glb.rd.buf,sizeof(_glb.rd.buf));
        if (_glb.rd.nbytes <= 0)
        {
            _glb.rd.state = FILE_SDONE;
            if (_glb.rd.nbytes < 0)
            {
                MsgError("FileReadData","I/O error reading file");
                _glb.rd.state = FILE_SERR;
            }
        }

        FileReport(FILE_RSEND);
    }

    return(0);
}

void FileWriteData(id,buf,nbytes)
int id;
char *buf;
int nbytes;
{
    if (_glb.wr.state != FILE_SBUSY)
    {
        if (_glb.wr.state != FILE_SERR)
            MsgError("FileWriteData","unexpected : incoming data out of phase");
        _glb.wr.state = FILE_SERR;
        return;
    }

    if (nbytes > (sizeof(_glb.wr.buf) - _glb.wr.nbytes))
    {
        MsgError("FileWriteData","unexpected : packet is too long");
        _glb.wr.state = FILE_SERR;
        return;
    }

    memcpy(&_glb.wr.buf[_glb.wr.nbytes],buf,nbytes);
    _glb.wr.nbytes += nbytes;
    _glb.wr.done   += nbytes;

    if (_glb.wr.nbytes >= FILE_BUFSIZE)
    {
        if (write(_glb.wr.fd,_glb.wr.buf,FILE_BUFSIZE) != FILE_BUFSIZE)
        {
            MsgError("FileWriteData","I/O error writing file - disk full (?)");
            _glb.wr.state = FILE_SERR;
        }

        FileSetTime();  /* reset time after each write */

        memcpy(_glb.wr.buf,&_glb.wr.buf[FILE_BUFSIZE],_glb.wr.nbytes - FILE_BUFSIZE);
        _glb.wr.nbytes -= FILE_BUFSIZE;

        if (_glb.wr.done != _glb.wr.todo)
            FileReport(FILE_RREC);
    }

    if (_glb.wr.done == _glb.wr.todo)
    {
        write(_glb.wr.fd,_glb.wr.buf,_glb.wr.nbytes);

        /*
         * critical : state must change at this point
         * since next packet might open a new file
         */

        _glb.wr.state = FILE_SIDLE;

        close(_glb.wr.fd);

        FileSetTime();

        FileReport(FILE_RREC);
    }
}


int FileOpenCtrl(ch)
int ch;
{
    if (_glb.chCtrl >= 0)
    {
        MsgError("FileOpen","unexpected : file control channel already open");
        return(-1);
    }
    _glb.chCtrl = ch;

    _glb.rd.state = FILE_SIDLE;
    _glb.wr.state = FILE_SIDLE;

    return(0);
}

void FileCloseCtrl(id)
int id;
{
    _glb.chCtrl = -1;
}

int FileOpenData(ch)
int ch;
{
    if (_glb.chData >= 0)
    {
        MsgError("_FileOpen","unexpected : file data channel already open");
        return(-1);
    }
    _glb.chData = ch;

    _glb.rd.state = FILE_SIDLE;
    _glb.wr.state = FILE_SIDLE;

    return(0);
}

void FileCloseData(id)
int id;
{
    _glb.chData = -1;

    if (_glb.rd.state == FILE_SBUSY)
        close(_glb.rd.fd);

    if (_glb.wr.state == FILE_SBUSY)
    {
        if (_glb.wr.nbytes > 0)
            write(_glb.wr.fd,_glb.wr.buf,_glb.wr.nbytes);
        close(_glb.wr.fd);
        FileSetTime();
    }
}

int FileInit(in,out)
char *in;
char *out;
{
    int len;

    strncpy(_glb.rd.dir,out,sizeof(_glb.rd.dir));
    _glb.rd.dir[sizeof(_glb.rd.dir) - 1] = '\0';
    strncpy(_glb.wr.dir,in,sizeof(_glb.wr.dir));
    _glb.wr.dir[sizeof(_glb.wr.dir) - 1] = '\0';

    len = strlen(_glb.rd.dir);
    if (_glb.rd.dir[len - 1] == SYS_PATHDELIM)
        _glb.rd.dir[len - 1] = '\0';

    len = strlen(_glb.wr.dir);
    if (_glb.wr.dir[len - 1] == SYS_PATHDELIM)
        _glb.wr.dir[len - 1] = '\0';

    return(0);
}

