/*****************************************************
 * connect.c : connection procedures
 *
 * Note : this module is not actually part of the
 * the packet/multiplexing protocol. It merely provides
 * a way for clean connect and disconnect _using_ the
 * protocol.
 *
 * Copyright (C) 1992 Jouni Lepp{j{rvi
 *****************************************************/

#include <string.h>
#include <time.h>

#include "system.h"
#include "usrmsg.h"
#include "mux.h"
#include "control.h"
#include "packetio.h"
#include "channel.h"
#include "config.h"
#include "connect.h"

#if SYS_PROTOS
static int ConCmp(char *local,char *remote);
#else
static int ConCmp();
#endif

static unsigned long _tSalvage;
static char _salvString[64+1];
static int _salvLen = 0;

/* ConInit() initializes the connect module for a new connection.
 *
 * returns :  0 (always)
 *           !0 (error - not currently used)
 */

int ConInit()
{
    CtrlInit();
    MuxInit();
    PacketInit();

    _tSalvage = 0L;

    return(0);
}

/* ConSetSalvStr() sets the 'salvage string'.
 *
 * salvString : the 'salvage string'
 *
 * returns :  0 (always)
 *           !0 (error - currently unused)
 */

int ConSetSalvStr(salvString)
char *salvString;
{
    strncpy(_salvString,salvString,sizeof(_salvString));
    _salvString[sizeof(_salvString) - 1] = '\0';
    _salvLen = strlen(_salvString);
    return(0);
}

/* The functions ConFlush(), ConConnect() and ConDisconnect()
 * should be called on an interval at highest equal to PACKET_INTERVAL
 * (from packetio.h). The packet timings, however, are not based on
 * the calling interval, but instead on real-time from function mtime().
 *
 *
 * ConFlush() performs the packet flush by calling PacketFlush()
 * with the millisecond real-time argument on which packet timings
 * are based. In addition to this the salvage string is send after
 * a long (?) silence in the line.
 *
 * returns : status from PacketFlush()
 *
 *
 * ConConnect() is used to establish a connection.
 *
 * sec : !0 specfies connect timeout on first call
 *        0 on successive calls
 *
 * returns : CON_WAIT   : time remais until timeout, call again
 *           CON_FAILED : connection failed, time out
 *           CON_DONE   : connection established
 *
 *
 * ConDisconnect() is used for clean disconnect.
 *
 * sec : !0 disconnect timeout (first call)
 *        0 on successive calls
 *
 * returns : CON_WAIT   : time remais until timeout, call again
 *           CON_FAILED : disconnect failed, timeout
 *           CON_DONE   : disconnect ok
 */

int ConFlush()
{
    static unsigned long _t;
    unsigned long t;
    int ret;

    ret = PacketFlush(t = mtime());

    if (_salvLen > 0 && !(ret & PACKET_IN) && _t < t)
    {
        _t = t + 5000L;
        LineWrite(_salvString,_salvLen);
    }

    return(ret);
}

/* ConCmp() compares the connect argument strings and uses
 * the first differing characters as local/remote ids for
 * PacketSetId().
 *
 * (internal)
 *
 * returns :  0 : ok
 *           !0 : strings are equal
 */

static int ConCmp(local,remote)
char *local, *remote;
{
    for (; *local && *local == *remote ;local++, remote++);
    PacketSetId((unsigned char) *local,(unsigned char) *remote);
    return(*local != *remote);
}

int ConConnect(sec)
int sec;
{
    static long t;
    static char c, cCh = '\r';
    static char buf[32];
    char *p;

    if (sec)
    {
        t = time(NULL) + sec;
        gethostname(buf,sizeof(buf));
        CtrlSendCmd("CONNECT %s",buf);
        c = ~cCh;
    }

    if (time(NULL) > t)
    {
        MsgDisplay(MSG_USER,"connect : timed out");
        return(CON_FAILED);
    }

    if (c != '\r' && c != '\n') /* AIX mangles \r to \n */
    {
        LineWrite(&cCh,1);
        LineRead(&c,1);
    }
    else
    {
        if (!(ConFlush() & PACKET_QOUT))
        {
            p = CtrlGetConArg();
            if (p != NULL && ConCmp(buf,p))
            {
                MsgDisplay(MSG_USER,"connect : connected to %s",p);
                return(CON_DONE);
            }
        }
    }

    return(CON_WAIT);
}


int ConDisconnect(sec)
int sec;
{
    static long t;
    static int state;
    int ret;

    if (sec)
    {
        state = 0;
        t = time(NULL) + sec;
        ChanClose(0);
        return(CON_WAIT);
    }

    if (time(NULL) > t)
    {
        MsgDisplay(MSG_USER,"disconnect : timed out");
        return(CON_FAILED);
    }

    ret = ConFlush();

    if ((ret & PACKET_IN))
        state = 1;
    else if (state == 1)
    {
        MsgDisplay(MSG_USER,"disconnect : ok");
        return(CON_DONE);
    }


    return(CON_WAIT);
}
