' PROFILER.BAS:     VB program profiler
'
' version:          2-9-95
' compiler:         Visual BASIC 3.0
' uses:             nothing
' module type:      VB module
'

' This module contains a simple program profiler for VB programs.  It
' lets you:
'
'   o Create up to 100 profiler regions ('buckets')
'   o Start and stop timing in them
'   o Display the results of your profiling in the Debug window
'
' The profiler is used during development of a program to find the
' hot spots where it spends most of it's time.
'
' To use this module:
'
'   1. Place "ProfilerInitialize" somewhere in Form_Load
'
'   2. Place an "AddProfile" for each bucket you want.  Pass the
'       tag and level for the region.
'
'   3. Make "StartProfile" and "StopProfile" calls for each of the
'       buckets at the places in your code you want to check.  You
'       should take care make sure the calls are at the same level,
'       and levels correspond roughly to levels of subroutine and
'       function nesting.
'
'   4. Put a "ProfilerPrint" in your program to dump the profile to
'       the debug window.
'
' You'll get error messages if you try to start timing a bucket
' that's already being timed, and stop timing a bucket that's not
' being timed.  This is probably a logic error in your program.

Option Explicit

Const i_TotalBuckets = 100      ' Total number of timer buckets in array

Type ProfileRecord
    tag As String               ' tag for printing
    CurrentStart As Variant     ' current record's start time
    TotalTime As Variant        ' total time recorded
    Count As Integer            ' count of times called
    level As Integer            ' static level #
End Type

Dim Bucket(i_TotalBuckets) As ProfileRecord

Sub AddProfile (iBucket As Integer, sWhat As String, iLevel As Integer)
    ' Add a record to the profiler
    ' Sent bucket #, tag, and static level number
    ' It's up to user to correlate bucket number later on
    Bucket(iBucket).tag = sWhat
    Bucket(iBucket).level = iLevel
End Sub

Function CompareBuckets (x As ProfileRecord, y As ProfileRecord) As Integer
    ' returns: -1 if x < y
    '           0 if x = y
    '           1 if x > y
    ' sorts by time within level
    If x.level <> y.level Then
        CompareBuckets = IIf(x.level > y.level, 1, -1)
    Else
        CompareBuckets = IIf(x.TotalTime > y.TotalTime, 1, IIf(x.TotalTime = y.TotalTime, 0, -1))
    End If
End Function

Sub ProfilerInitialize ()
    ' Initialize the profiler. This means setting all the fields
    ' to zero, and the bucket tags to "N/A" for printing.

    Dim ii As Integer

    For ii = 1 To i_TotalBuckets
        Bucket(ii).tag = "N/A"
        Bucket(ii).CurrentStart = 0
        Bucket(ii).TotalTime = 0
        Bucket(ii).Count = 0
        Bucket(ii).level = 0
    Next ii
End Sub

Sub ProfilerPrint ()
    ' Print the completed profile when the program is 'done'
    ' Prints report to Debug window

    Dim ii As Integer

    Debug.Print : Debug.Print "Profile complete"
    SortBuckets
    Debug.Print "Tag", "Level", "Count", "Total Time (sec.)"

    For ii = 1 To i_TotalBuckets
        If Bucket(ii).tag <> "N/A" Then
            If Len(Bucket(ii).tag) < 14 Then
                Debug.Print Bucket(ii).tag,
            Else
                Debug.Print Left$(Bucket(ii).tag, 10) & "...",
            End If
            Debug.Print Bucket(ii).level, Bucket(ii).Count, Bucket(ii).TotalTime
        End If
    Next ii
End Sub

Sub SortBuckets ()
    Dim l As ProfileRecord ' temporary
    Dim i As Integer, j As Integer ' loop indexes

    ' Sort buckets by level w/insertion sort
    For i = 2 To i_TotalBuckets
        j = i - 1
        l = Bucket(i)
A:      If CompareBuckets(l, Bucket(j)) = -1 Then GoTo B' comparison function
        Bucket(j + 1) = Bucket(j)
        j = j - 1
        If j > 0 Then GoTo A
B:      Bucket(j + 1) = l
    Next i
End Sub

Sub StartProfiling (iBucket As Integer)
    ' Start a profile bucket timer record
    ' Also checks to make sure we're not profiling this bucket already
    
    Dim sErrorMessage As String

    If Bucket(iBucket).CurrentStart = 0 Then
        Bucket(iBucket).Count = Bucket(iBucket).Count + 1
        Bucket(iBucket).CurrentStart = Timer
    Else
        ' Display error message - bucket already in use
        sErrorMessage = "Bucket # " & Str$(iBucket) & " already in use!" & Chr$(13)
        sErrorMessage = sErrorMessage & "Tag: " & Bucket(iBucket).tag
        MsgBox sErrorMessage, 0
    End If
End Sub

Sub StopProfiling (iBucket As Integer)
    ' Stop collecting a profile; add time to bucket total
    ' Also checks to ensure we're profiling this bucket already
    
    Dim sErrorMessage As String
    Dim Finish, Elapsed

    If Bucket(iBucket).CurrentStart > 0 Then
        Finish = Timer
        Elapsed = Finish - Bucket(iBucket).CurrentStart
        Bucket(iBucket).CurrentStart = 0
        Bucket(iBucket).TotalTime = Bucket(iBucket).TotalTime + Elapsed
    Else
        ' Display error message - bucket already in use
        sErrorMessage = "Bucket # " & Str$(iBucket) & " not in use!" & Chr$(13)
        sErrorMessage = sErrorMessage & "Tag: " & Bucket(iBucket).tag
        MsgBox sErrorMessage, 0
    End If
End Sub

