/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "_termint.h"
#include "_termctl.h"
#include "stdmac.h"

#ifndef PB_SDK
	#include <stdio.h>
	#include <ctype.h>
	#include <string.h>
	#include <stdlib.h>
	#include <assert.h>
#else
	#include "pblibc.h"
#endif

/*
 * t h e   a n s i   i n t e r p r e t e r   i m p l e m e n t a t i o n
 * 
 * this is the class that interprets ANSi escape sequences. it implements
 * only the basic standard commands (none of the extended and non-supported)
*/
static int
eval_args(int *argv, char *src)
{
	int   i = 0;
	char *p;

	// special case where the first argument is missing
	// this is because strtok() skips over it blissfully
	if( ';' == src[2] ) argv[i++] = 0;
	for( p = strtok(&src[2], ";"); p && i < 20; ++i )
	{
		argv[i] = atoi(p);
		p = strtok(NULL, ";");
	}
	return i;
}

/*
 * init() serves as an ID for the class. it returns True if the interpreter
 * can handle a sequence which begins with 'aSOS' (also checks if active)
*/
Boolean
ansi_interp::init( uchar aSOS )
{
	Boolean retval = False;

	if( m_active )
	{
		if( aSOS == 0x1b )
		{
			term_interp::init( aSOS );
			retval = True;
		}
	}

	return retval;
}

void
ansi_interp::handle( uchar aChar )
{
	static int save_x = 1;
	static int save_y = 1;

	assert( m_buff );
	assert( m_ctrl );

	// this is the first character after SOS, make sure it's ok
	if( 1 == m_buff->getCount() )
	{
		m_buff->put(aChar);
		if( '[' != aChar ) m_fail = True;
		return; // we're thrugh with the second character
	}
	// we're parsing a sequence here, see if we are accumilating
	if( isdigit(aChar) || ';' == aChar )
	{
		m_buff->put(aChar);
		return;
	}
	// we're at the end of the sequence, attempt to interpret it
	else if( '?' != aChar && '=' != aChar )
	{
		// here we parse the current buffer and evaluate args
		int argv[20];  // should be large enough
		int argc;
		int x, y;

		m_buff->put(EOS);
		argc = eval_args(argv, m_buff->bp());

		switch( aChar )
		{
			case 'H': // fall-through to the synonym command 'f'
			case 'f':
				y = (argc >= 1 && argv[0]) ? argv[0] : 1;
				x = (argc == 2 && argv[1]) ? argv[1] : 1;
				m_ctrl->cursor_xy(x, y);
				break;

			case 'A':
				y = (argc == 1 && argv[0]) ? argv[0] : 1;
				m_ctrl->cursor_up(y);
				break;

			case 'B':
				y = (argc == 1 && argv[0]) ? argv[0] : 1;
				m_ctrl->cursor_down(y);
				break;

			case 'C':
				x = (argc == 1 && argv[0]) ? argv[0] : 1;
				m_ctrl->cursor_right(x);
				break;

			case 'D':
				x = (argc == 1 && argv[0]) ? argv[0] : 1;
				m_ctrl->cursor_left(x);
				break;

			case 'n':
				char buf[10];

				y = (argc >= 1 && argv[0]) ? argv[0] : 1;
				x = (argc == 2 && argv[1]) ? argv[1] : 1;
				sprintf(buf, "\033[%d;%dR", y, x);
				m_ctrl->put(buf, strlen(buf));
				break;

			case 's':
				save_x = m_ctrl->get_x();
				save_y = m_ctrl->get_y();
				break;

			case 'u': m_ctrl->cursor_xy(save_x, save_y); break;
			case 'J': m_ctrl->clr_scr(); break;
			case 'K': m_ctrl->clr_eol(); break;

			case 'h': // the following toggle the wrapping, we
			case 'l': // simply ignore them (silently :-)
			break;

			case 'm': // set graphics rendition
				short ibm2ans[8] = { 0, 4, 2, 6, 1, 5, 3, 7 };
				int   cur_fg;

				for( int i = 0; i < argc; ++i )
				{
					switch( argv[i] )
					{
						case 0: m_ctrl->set_color(7);                 break;
						case 1: m_ctrl->set_fg(m_ctrl->get_fg() | 8); break;
						case 2: m_ctrl->set_fg(m_ctrl->get_fg() & 7); break;
						case 4: m_ctrl->set_fg(1);                    break;
						case 5: m_ctrl->set_bg(m_ctrl->get_bg() | 8); break;
						case 8: m_ctrl->set_fg(m_ctrl->get_bg());     break;
						case 7:
							cur_fg = m_ctrl->get_fg();
							m_ctrl->set_fg(m_ctrl->get_bg());
							m_ctrl->set_bg(cur_fg);
							break;
						default:
							if( argv[i] >= 30 && argv[i] <= 37 )
							{
								uchar attrib = ibm2ans[argv[i] - 30];
								if( m_ctrl->get_fg() & 8 ) attrib |= 8;
								m_ctrl->set_fg(attrib);
							}
							else if( argv[i] >= 40 && argv[i] <= 47 )
							{
								uchar attrib = ibm2ans[argv[i] - 40];
								if( m_ctrl->get_bg() & 8 ) attrib |= 8;
								m_ctrl->set_bg(attrib);
							}
					}
				}
			break;
		}
		// this is the end, my friend
		m_eos = True;
	}
}
