/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "terminal.h"
#include "_termbuf.h"
#include "stdmac.h"

#ifndef PB_SDK
	#include <string.h>
#else
	#include "pblibc.h"
#endif

zTerminal::zTerminal()
	:m_ctrl(0)
	,m_busy(False)
	,m_nCurInterp(0)
	,m_nLastInterp(0)
	,m_nInterp(0)
	,m_buff(new term_buff(128))
{
	RegisterCtrlHandler(m_TermCtrl = new console_ctrl);
}

zTerminal::~zTerminal()
{
	if( m_TermCtrl ) delete m_TermCtrl;
	if( m_buff     ) delete m_buff;
}

void
zTerminal::RegisterHandler(term_interp *handler)
{
	if( m_nInterp < MAX_INTERP )
	{
		m_interp[m_nInterp] = handler;
		m_interp[m_nInterp]->RegisterCtrlHandler(m_ctrl);
		m_interp[m_nInterp]->RegisterBuffer(m_buff);
		m_interp[m_nInterp]->Enable(True);
		m_nInterp++;
		m_nLastInterp = m_nInterp;
	}
}

void
zTerminal::EnableHandler(term_interp *handler, Boolean enable)
{
	for( int i =0; i < m_nInterp; ++i )
	{
		if( m_interp[i] == handler )
		{
			m_interp[i]->Enable(enable);
			break;
		}
	}
}

void
zTerminal::UnloadHandler(term_interp *handler)
{
	for( int i = 0; i < m_nInterp; ++i )
	{
		if( m_interp[i] == handler )
		{
			int bytes = (m_nInterp - i - 1) * sizeof(m_interp[0]);

			if( bytes ) memmove(&m_interp[i], &m_interp[i+1], bytes);
			m_nInterp--;
			break;
		}
	}
}

void
zTerminal::RegisterCtrlHandler(term_ctrl *handler)
{
	m_ctrl = handler;
	m_buff->RegisterTerm(m_ctrl);
	// now update the pointers in all registered handlers
	for( int i = 0; i < m_nInterp; ++i )
	{
		m_interp[i]->RegisterCtrlHandler(m_ctrl);
	}
}


void
zTerminal::handle(uchar aChar)
{
	char    auxBuf[256];
	size_t  auxPos, auxCount;
	state_t status;

	// if not failure, done processing (on end-of-sequence,
	// we have to clear the buffer, just in case)
	if( fail != (status = handleChar(aChar)) )
	{
		if( eos == status ) m_buff->setPos(0);
		return;
	}

	// ok, the handler apparently has a problem now
	// first, save the current handler buffer here
	auxCount = m_buff->getCount();
	auxPos   = 0;
	memcpy(auxBuf, m_buff->bp(), auxCount);

	m_nLastInterp = m_nCurInterp;
	m_buff->setPos(0);

	// now process the local storage buffer here
	while( auxPos < auxCount )
	{
		int count;

		switch( handleChar(auxBuf[auxPos]) )
		{
			case fail: // this handler didn't work
				m_buff->setPos(0);
				m_nLastInterp = m_nCurInterp;
				auxPos = 0;
				break;

			case eos: // we have the end of sequence
				count = auxCount - auxPos - 1;
				if( 0 < count )
				{
					memmove(auxBuf, &auxBuf[auxPos + 1], count);
					auxPos = 0;
					auxCount = count;
					m_nLastInterp = m_nInterp;
				}
				else auxPos = auxCount;
				m_buff->setPos(0);
				break;

			default:
				auxPos++;
				if( !m_busy ) m_nLastInterp = m_nInterp;
		}
	}
}

void
zTerminal::handle(uchar *aString)
{
	if( 0 != aString )
	{
		while( EOS != *aString ) handle(*aString++);
	}
}

void
zTerminal::flush(Boolean display)
{
	// we will only flush if something is active and we have chars!
	if( m_busy && display && m_buff->getCount() )
	{
		m_ctrl->put(m_buff->bp(), m_buff->getCount());
	}
	m_buff->setPos(0);
}

zTerminal::state_t
zTerminal::handleChar( uchar aChar )
{
	if( m_busy )
	{
		m_interp[m_nCurInterp]->handle(aChar);
		if( m_interp[m_nCurInterp]->fail() )
		{
			m_busy = False;
			return fail;
		}
		if( m_interp[m_nCurInterp]->eos() )
		{
			m_busy = False;
			return eos;
		}
		return ok;
	}
	// we are not processing a sequence, see for a SOS
	// note that we are working backwards - this is to
	// give chance to a user-installed handler to react
	// on SOS characters that we might be using ourselves
	for( int i = m_nLastInterp - 1; 0 <= i; --i )
	{
		if( m_interp[i]->init(aChar) )
		{
			m_nCurInterp = i;
			if( m_interp[i]->eos() )
			{
				m_buff->setPos(0);
				return eos;
			}
			m_busy = True;
			return ok;
		}
	}

	// this was not a special sos character, simply write it out
	m_ctrl->put(aChar);
	return eos;
}
