/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "listview.h"
#include "kbcodes.h"

#ifndef PB_SDK
	#include <conio.h>
	#include "compiler.h"
#else
	#include "pblibc.h"
	#include "pblsdk.h"
#endif

zListRecord::zListRecord()
	:m_rows(1)
	,m_tagged(False)
{
}

void
zListRecord::toggleTag()
{
	m_tagged = Boolean( !m_tagged );
}

////////////////////////////////////////////////////////////////////////////
zListViewer::zListViewer(const zRect &bounds, zDoubleList *aList)
	:m_bounds(bounds)
	,m_list(aList)
	,m_focus(*m_list, True)
	,m_top(*m_list, True)
	,m_bottom(*m_list, True)
	,m_clearColor(0)
{
	m_where.x = 0;
	m_where.y = 0;
	getNextLimit(m_bottom, m_top);
}

void
zListViewer::draw()
{
	zListCursor cursor = m_top;
	short       nrows  = m_bounds.Height();
	zPoint      origin;

	origin.x = m_bounds.a.x + m_where.x;
	origin.y = m_bounds.a.y + 0;

	if( 0 != m_list->size() )
	{
		while( 0 < nrows )
		{
			zListRecord *element = (zListRecord *)cursor.get();

			gotoxy(origin.x, origin.y);
			element->draw(origin, m_bounds.Width());
			nrows -= element->m_rows;
			origin.y += element->m_rows;
			if( !cursor.next() ) break;
		}
	}
	// clear the rest of the window lines
	if( 0 < nrows )
	{
		textattr(m_clearColor);
		while( nrows-- )
		{
			gotoxy(origin.x, origin.y++);
			for( int i = m_bounds.Width(); i; --i ) putch(' ');
		}
	}

	if( 0 != m_list->size() ) writeLiteBar(False);
}

void
zListViewer::drawFocus()
{
	zPoint origin;

	m_where.y = rowsToSkip(&m_top, &m_focus);
	origin.x = m_bounds.a.x + m_where.x;
	origin.y = m_bounds.a.y + m_where.y;

	gotoxy(origin.x, origin.y);
	((zListRecord *)m_focus.get())->draw(origin, m_bounds.Width());
	writeLiteBar(False);
}

void
zListViewer::handle(ushort &aCode)
{
	if( 0 == m_list->size() ) return;

	switch( aCode )
	{
		case kbLeft   :
		case '4'      :
			if( !m_top.atStart() )
			{
				zListCursor top = m_top;

				top.prev();
				getPrevLimit(m_top, top);
				getNextLimit(m_bottom, m_top);
				m_focus = m_top;
				m_where.y = 0;
				draw();
			}
			break;

		case kbRight   :
		case '6'       :
			if( !m_bottom.atEnd() )
			{
				zListCursor bottom = m_bottom;

				bottom.next();
				getNextLimit(m_bottom, bottom);
				getPrevLimit(m_top, m_bottom);
				m_focus = m_top;
				m_where.y = 0;
				draw();
			}
			break;

		case kbUp   :
		case '8'    :
			writeLiteBar(True);
			if( m_focus() == m_top() )
			{
				m_focus = m_bottom;
				m_where.y = rowsToSkip(&m_top, &m_bottom);
			}
			else
			{
				m_focus--;
				m_where.y -= rowsToSkip(&m_focus, 0);
			}
			writeLiteBar(False);
			break;

		case kbDown:
		case '2'   :
			writeLiteBar(True);
			if( m_focus() == m_bottom() )
			{
				m_focus = m_top;
				m_where.y = 0;
			}
			else
			{
				m_focus++;
				m_where.y += rowsToSkip(0, &m_focus);
			}
			writeLiteBar(False);
			break;

		case kbHome   :
		case '7'      :
			if( m_focus() != m_top() )
			{
				writeLiteBar(True);
				m_focus = m_top;
				m_where.y = 0;
				writeLiteBar(False);
			}
			break;

		case kbEnd   :
		case '1'     :
			if( m_focus() != m_bottom() )
			{
				writeLiteBar(True);
				m_focus = m_bottom;
				m_where.y = rowsToSkip(&m_top, &m_bottom);
				writeLiteBar(False);
			}
			break;

		default:
			return;
	}
	aCode = 0;
}

void
zListViewer::writeLiteBar(Boolean off)
{
	zPoint       origin;

	origin.x = m_bounds.a.x + m_where.x;
	origin.y = m_bounds.a.y + m_where.y;
	gotoxy(origin.x, origin.y);
	((zListRecord*)m_focus.get())->select(origin, m_bounds.Width(), off);
}

void
zListViewer::newList(zDoubleList *aList)
{
	m_list    = aList;
	m_focus   = zListCursor(*m_list, True);
	m_top     = zListCursor(*m_list, True);
	m_bottom  = zListCursor(*m_list, True);
	m_where.x = m_where.y = 0;
	getNextLimit(m_bottom, m_top);
}

void
zListViewer::getNextLimit(zListCursor &newLimit, zListCursor &whence)
{
	zListCursor cursor = whence;
	short       nrows  = m_bounds.Height();

	nrows -= ((zListRecord *)cursor.get())->m_rows;
	while( 0 < nrows && !cursor.atEnd() )
	{
		cursor++;
		nrows -= ((zListRecord *)cursor.get())->m_rows;
	}
	newLimit = cursor;
}

void
zListViewer::getPrevLimit(zListCursor &newLimit, zListCursor &whence)
{
	zListCursor cursor = whence;
	short       nrows  = m_bounds.Height();

	nrows -= ((zListRecord *)cursor.get())->m_rows;
	while( 0 < nrows && !cursor.atStart() )
	{
		cursor--;
		nrows -= ((zListRecord *)cursor.get())->m_rows;
	}
	newLimit = cursor;
}

/*
 * if any arg == 0, means get to next, i.e. from = 0, get from 'to'
 * to the previous elem; if to = 0, get from 'from' to next elem
*/
short
zListViewer::rowsToSkip(zListCursor *aFrom, zListCursor *aTo)
{
	if( 0 == aFrom && 0 == aTo )
	{
		return 0;
	}
	if( 0 == aFrom && !aTo->atStart() )
	{
		zListCursor to(*aTo);
		to.prev();
		return ((zListRecord *)to.get())->m_rows;
	}
	else if( 0 == aTo && !aFrom->atEnd() )
	{
		zListCursor from(*aFrom);
		from.next();
		return ((zListRecord *)from.get())->m_rows;
	}
	else
	{
		zListCursor from(*aFrom), to(*aTo);
		short num = 0;
		while( !from.atEnd() && from() != to() )
		{
			num += ((zListRecord *)from.get())->m_rows;
			from.next();
		}
		return num;
	}
}

void
zListViewer::setFocus(void *item)
{
	zListCursor cursor(*m_list, True);

	do
	{
		if( cursor.get() == item ) break;
	}while( cursor.next() );

	if( item == cursor.get() )
	{
		getNextLimit(m_bottom, cursor);
		getPrevLimit(m_top, m_bottom);
		m_focus = cursor;
		m_where.y = rowsToSkip(&m_top, &m_focus);
	}
	draw();
}
