/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "febbs.h"
#include "fbase.h"

#ifndef PB_SDK
	#include <sys/stat.h>
	#include <assert.h>
	#include <string.h>
	#include <ctype.h>
#else
	#include "pblibc.h"
#endif

////////////////////////////////////////////////////////////////////////////
// FILES.BBS line interpreter class
////////////////////////////////////////////////////////////////////////////
zFbbsLine::zFbbsLine(date_format_t fmt, char left_dlchar, char right_dlchar):
	m_DlCtLeft(left_dlchar),
	m_DlCtRight(right_dlchar)
{
	m_FindFile = new zFileAreaDir;
	assert(m_FindFile != 0);

	switch( fmt )
	{
		case european: m_DateFormat = "%d/%m/%y"; break;
		case american: m_DateFormat = "%m/%d/%y"; break;
		case japanese: m_DateFormat = "%y/%m/%d"; break;
	}

	ResetVars();
}

zFbbsLine::~zFbbsLine()
{
	delete m_FindFile;
}

void
zFbbsLine::SetPath(const char *path, int areano, Boolean cdrom)
{
	m_FindFile->SetPath(path, areano, cdrom);
}

void
zFbbsLine::ResetVars()
{
	m_FileName[0]  = EOS;
	m_FileDate[0]  = EOS;
	m_FileSize[0]  = EOS;
	m_FileCount[0] = EOS;
	m_FileDesc[0]  = EOS;
	m_Flags = 0;
}

// checks for a date string, stores if necessary
Boolean
zFbbsLine::CheckDate(char *s)
{
	if( ('-' == s[2] || '/' == s[2]) && ('-' == s[5] || '/' == s[5]) )
	{
		m_Flags |= date;
		strcpy(m_FileDate, s);
		return True;
	}
	return False;
}

// checks for a file size string, stores if necessary
Boolean
zFbbsLine::CheckSize(char *s)
{
	for( char *p = s; EOS != *p; ++p ) if( !isdigit(*p) ) return False;
	m_Flags |= size;
	strcpy(m_FileSize, s);
	return True;
}

// checks and copies the download counter definition
Boolean
zFbbsLine::CheckCounter(char *s)
{
	if( m_DlCtLeft == s[0] && m_DlCtRight == s[strlen(s) - 1] )
	{
		strcpy(m_FileCount, s);
		m_Flags |= count;
		return True;
	}
	return False;
}

// get the file date string (use the format from the file area
void
zFbbsLine::GetFileDate(char *fname)
{
	struct stat statbuf;
	if( -1 != stat(fname, &statbuf) )
	{
		strftime(m_FileDate, 9, m_DateFormat, localtime(&statbuf.st_ctime));
		m_Flags |= disk_date;
	}
}

// get the file size string
void
zFbbsLine::GetFileSize(char *fname)
{
	struct stat statbuf;
	if( -1 != stat(fname, &statbuf) )
	{
		sprintf(m_FileSize, "%ld", statbuf.st_size);
		m_Flags |= disk_size;
	}
}

// get the description text
void
zFbbsLine::GetDescription(char *s)
{
	strncpy(m_FileDesc, s, MAX_DESCSZ);
	m_FileDesc[MAX_DESCSZ] = EOS;
	m_Flags |= text;
}

void
zFbbsLine::SetLine(const char *line)
{
	assert(line);
	char *path, *p = (char *)line;
	int   nCount;

	ResetVars();
	if( isascii(*p) && isprint(*p) && !isspace(*p) )
	{
		sscanf(p, "%12s %n", m_FileName, &nCount);
		if( 0 != (path = m_FindFile->Find(m_FileName)) )
		{
			char s[1024];

			m_Flags |= name;
			p += nCount;
			sscanf(p, "%s %n", s, &nCount);
			if( CheckDate(s) || CheckSize(s) ) p += nCount;
			sscanf(p, "%s %n", s, &nCount);
			if( CheckSize(s) || CheckDate(s) ) p += nCount;
			sscanf(p, "%s %n", s, &nCount);
			if( CheckCounter(s) ) p += nCount;
			if( !(m_Flags & date) ) GetFileDate(path);
			if( !(m_Flags & size) ) GetFileSize(path);
			if( !(m_Flags & count) )
			{
				sprintf(m_FileCount, "%c00%c", m_DlCtLeft, m_DlCtRight);
				m_Flags |= fake_count;
			}
		}
	}
	GetDescription(p);
}

char*
zFbbsLine::GetLine()
{
	static char line[1024];

	if( m_Flags & name )
	{
		sprintf(line, "%-12s ", m_FileName);
		if( m_Flags & date ) sprintf(line+strlen(line), "%s ", m_FileDate);
		if( m_Flags & size ) sprintf(line+strlen(line), "%7s ", m_FileSize);
		if( m_Flags & count) sprintf(line+strlen(line), "%s ", m_FileCount);
		if( m_Flags & text ) sprintf(line+strlen(line), "%s", m_FileDesc);
	}
	else
	{
		strcpy(line, m_FileDesc);
	}
	return line;
}
