/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "pblibc.h"

/* this is just to test the module, remove the include above to use it */
#ifdef PB_SDK
	#include "pblsdk.h"
#else
	#include <stdio.h>
	#include <stdarg.h>
	#include <ctype.h>
	#include <assert.h>
	#include <string.h>
#endif

#ifndef EOS
#define EOS '\0'
#endif

#ifndef MAXINT
#define MAXINT 32767
#endif

#define GETCHAR        ( ss.nread++, input(stream)   )
#define UNGETCHAR(c)   ( ss.nread--, unget(c,stream) )
#define ARRAYSZ(a)     ( sizeof(a) / sizeof((a)[0])  )
#define SCANTBL_SZ     32  /* bytes needed to hold 256 bits */
#define BITSET(c)      ( ss.table[(c) >> 3] & (1 << ((c) & 0x07)) )
#define LC_NONE        -2

/* various local flags for the scanner state */
enum{
	_sfNegate   = 0x0001,   /* negate the current value when assigning */
	_sfLong     = 0x0002,   /* current value is a long integer         */
	_sfShort    = 0x0004,   /* current value is a short integer        */
	_sfAssign   = 0x0008,   /* assign the current value if this is set */
	_sfUnsigned = 0x0010,   /* the value should be unsigned            */
	_sfExclude  = 0x0020,   /* the ss.table is an exclusion set        */
	_sfError    = 0x0040    /* an error has occurred while scanning    */
};

/* local scanner state information record */
struct __sstate{
	int     cchar;              /* current character                */
	int     nread;              /* number of characters read so far */
	int     nass;               /* number of assigned fields        */
	int     base;               /* current conversion base          */
	int     flags;              /* current scan state flags         */
	int     width;              /* required width of input for type */
	long    val;                /* value of current scan            */
	char    table[SCANTBL_SZ];  /* table of acceptable characters   */
	va_list ap;                 /* current argument list pointer    */
};

/* last character pushed back by the helper functions (for the PB SDK);  */
/* if this is LC_NONE, then input functions will ignore it and will read */
/* the next available character. If not, this is the one they will read. */
static int __lastChar = LC_NONE;

/* local function prototypes */
static int __scanner(int(*)(void *), int(*)(int, void *),
					 void *, const char *, va_list);
static int __ssGetc(void *);         /* __ss??? - string helpers */
static int __ssUngetc(int, void *);
static int __fsGetc(void *);         /* __fs??? - file helpers   */
static int __fsUngetc(int, void *);
static int __csGetc(void *);         /* __cs?? - console helpers */
static int __csUngetc(int, void *);

/*
 * general-purpose scanner function for formatted input.
 *
 * 'input' is the input function that should return the next available
 *         character from the stream 'fp'.
 * 'unget' is the function that should push back the last character read
 * 'stream' is the input stream
 * 'format' is the format string
 * 'vargp' is the pointer to the variable argument list
*/
	int
__scanner( int (*input)(void *), int (*unget)(int, void *), void *stream,
		   const char *format, va_list vargp )
{
	struct __sstate  ss;  /* current state for the scanner machine    */
		   char      ch;  /* current character from the format string */
		   char     *ptr; /* pointer into the array to store chars    */

	ss.nread = ss.nass = 0;

	/* main loop while there's a format string to match */
	while( EOS != (ch = *format++) ){

		/* get our a character from the stream */
		if( EOF == (ss.cchar = GETCHAR) ) goto _endScan;

		/* see if we need to skip some whitespace */
		if( ' ' == ch || '\t' == ch || '\n' == ch ){
			do{ /* skip additional whitespace in format string */
				ch = *format++;
			}while( ' ' == ch || '\t' == ch || '\n' == ch );
			/* skip any whitespace in the input stream */
			while( isspace(ss.cchar) ) ss.cchar = GETCHAR;
			if( EOS == ch || EOF == ss.cchar ) goto _endScan;
		}

		/* at this point, we have a character in ss.cchar that is not EOF */
		/* and a character in ch that is not whitespace or end of string  */
		/* see if the ch (format character) is the '%' start of type spec */
		if( '%' == ch ){
			/* set width to 0, reset current flags & get next format char */
			ss.width = 0; ss.flags = 0; ch = *format++;

			/* see if we want to suppress the assignment; if not,  */
			/* prepare ss.ap to point to the next storage location */
			if( '*' == ch ) ch = *format++;
			else{
				ss.flags |= _sfAssign;
				ss.ap = va_arg(vargp, char *);
			}

			/* get the optional width, set to BIG if none */
			while( isdigit(ch) ){
				ss.width = 10 * ss.width + (ch - '0');
				ch = *format++;
			}
			if( 0 == ss.width ) ss.width = MAXINT;

			/* get short 'h' or long 'l' type modifiers */
			switch(ch){
				case 'h': ss.flags |= _sfShort; ch = *format++; break;
				case 'l': ss.flags |= _sfLong; ch = *format++; break;
			}

			/* at this point, we have processed all the modifiers, now  */
			/* we move to the actual type character, note that we still */
			/* have the current character from the stream in ss.cchar   */
			switch(ch){
				case '%': goto _exactMatch; /* the 'else' clause */

				case 'U': ss.flags |= _sfUnsigned; /* fall-through */
				case 'D': ss.base = 10; goto __locLong;
				case 'X': ss.base = 16; goto __locLong;
				case 'O': ss.base =  8; goto __locLong;
				case 'B': ss.base =  2; goto __locLong;

				__locLong:
					ss.flags |= _sfLong;
					ss.flags &= ~_sfShort; /* 'h' modifer has no effect */
				goto _intScan;

				case 'u': ss.flags |= _sfUnsigned; /* fall-through */
				case 'd': ss.base = 10; goto _intScan;
				case 'x': ss.base = 16; goto _intScan;
				case 'o': ss.base =  8; goto _intScan;
				case 'b': ss.base =  2; goto _intScan;

				/* this converts decimal, hex, octal and binary. If the    */
				/* input starts with 0x - hex assumed, 0?? - octal assumed */
				/* 0b - binary assumed, otherwise - decimal integer        */
				case 'i':
					if( '0' == ss.cchar ){
						ss.cchar = GETCHAR;
						if( 'x' == ss.cchar || 'X' == ss.cchar ){
							ss.base = 16;
							ss.cchar = GETCHAR;
						}
						else if( 'b' == ss.cchar || 'B' == ss.cchar ){
							ss.base = 2;
							ss.cchar = GETCHAR;
						}
						else{
							ss.base = 8; /* assume octal zero */
						}
					}
					else{ /* assume a decimal */
						ss.base = 10;
					}
				goto _intScan;

				/* we allow any characters for the '%s' specification */
				/* to achieve that, we set all bits in the table to 1 */
				case 's':
					memset( ss.table, 0xff, ARRAYSZ(ss.table) );
				goto _strScan;

				/* scan the character set for allowed characters         */
				case '[':
					/* init the array to zero out all bits */
					memset( ss.table, 0x00, ARRAYSZ(ss.table) );
					if( '^' == (ch = *format++) ){
						ch = *format++;
						ss.flags |= _sfExclude;
					}

					/* we have ch = the start char after '[' here    */
					for( ;; ){
						char schar = ch;

						assert( EOS != ch );  /* unexpected end here */
						ss.table[ch >> 3] |= 1 << (ch & 0x07);
						ch = *format++;
						if( ']' == ch ) break;
						if( '-' == ch && schar < *format && ']' != *format ){
							ch = *format++;
							while( ++schar < ch )
								ss.table[schar >> 3] |= 1 << (schar & 0x07);
						}
					}

				_strScan: /* scan in a string according to table setup */
					ptr = (char *)ss.ap;
					while(1){
						/* did we exhaust the string of valid characters */
						if( !ss.width-- ||        /* width chars reached */
							isspace(ss.cchar) ||  /* whitespace ends it  */
							EOF == ss.cchar ||    /* EOF ends it too     */
							/* see if bit is set & if it's inclusion set */
							(BITSET(ss.cchar) && (ss.flags & _sfExclude)) ||
							(!BITSET(ss.cchar) && !(ss.flags & _sfExclude))
						){
								*ptr = EOS; /* terminate the string       */
								ss.nass++;  /* increase assignment number */
								UNGETCHAR(ss.cchar); /* return offender   */
								break;      /* jump to the 'continue'     */
						}
						/* otherwise, simply store character and get new */
						*ptr++ = ss.cchar;
						ss.cchar = GETCHAR;
					}
				continue; /* to the top of the while loop */

				case 'c': /* read in a character (or an array) */
					if( MAXINT == ss.width ) ss.width = 1;
					ptr = (char *)ss.ap;
					while(ss.width--){
						if( ss.flags & _sfAssign ) *ptr = ss.cchar;
						if( EOF == (ss.cchar = GETCHAR) ) goto _endScan;
					}
					if( ss.flags & _sfAssign ) ss.nass++;
					UNGETCHAR(ss.cchar);
				continue; /* to the top of the while loop */

				case 'n': /* assign number of read characters until %n */
					UNGETCHAR(ss.cchar);
					if( ss.flags & _sfAssign ){
						*(int *)ss.ap = ss.nread;
						ss.nass++;
					}
				continue;  /* to the top of the while loop */

				default :  /* should not happen for well-formed format */
					assert(0);
				#if 0
					ss.flags |= _sfError;
					printf( "\n_scanner: unrecognized char: '%c'\n", ch );
					goto _endScan;
				#endif
			}

			/* we will never reach this point via a fall-through from   */
			/* the switch statement above. The only way to get here is  */
			/* from the 'goto _intScan' statements in the cases there   */
			/* here we have ss.base and ss.width set to relevant values */
			/* and ss.cchar holds the character to process (check sign) */
			_intScan:
			switch( ss.cchar ){
				case '-': ss.flags |= _sfNegate; /* fall-through */
				case '+': ss.cchar = GETCHAR; ss.width--;
				default : ;   /* not needed, for clarity only */
				/* if no sign, just skip it and go convert it */
			}

			ss.val = 0L;
			while( ss.width-- ){  /* read width number of characters */
				int i;

				if( !isxdigit(ss.cchar) ) break; /* invalid digit    */
				if( '9' >= ss.cchar ) i = ss.cchar - 48;
				else if( 'a' <= ss.cchar ) i = ss.cchar - 87;
				else if( 'A' <= ss.cchar ) i = ss.cchar - 55;
				if( 0 > i || i >= ss.base ) break; /* invalid digit */

				/* add current value  */
				ss.val = l_mul(ss.val, (long)ss.base) + i;
				ss.cchar = GETCHAR; /* get next input character */
			}

			/* push back the offending character or EOF */
			UNGETCHAR(ss.cchar);

			/* now do the actual assignment if necessary; note that the   */
			/* only thing that gets assigned here is ss.val from _intScan */
			if( ss.flags & _sfAssign ){
				if( ss.flags & _sfNegate && !(ss.flags & _sfUnsigned))
					ss.val = -ss.val;

				if( ss.flags & _sfShort ) *(short *)ss.ap = (short)ss.val;
				else if( ss.flags & _sfLong ) *(long *)ss.ap = ss.val;
				else *(int *)ss.ap = (int)ss.val;
				ss.nass++;
			}

			/* loop back to get next format string character         */
			continue; /* this is not really needed, it's for clarity */
		}
		else{
	_exactMatch: /* we can get here from the case(ch) of '%' too */
			/* if not special conversion, must match literally   */
			/* we do have non-whitespace ch and non-EOF ss.cchar */
			if( ch != ss.cchar ){
				ss.flags |= _sfError;
				goto _endScan;
			}
			/* loop back to the start of the input while(...) */
		}
	}
_endScan:
	return ss.nass ? ss.nass : EOF;
}

/*
 
  Various helper functions to get and unget characters to and from streams.
 
*/

/* helper functions to getc and ungetc to string (do not use __lastChar) */
	int
__ssGetc( void *sp )
{
	return (*(*(char **)sp)) ? *(*(char **)sp)++ : EOF;
}

/* we don't allow modification of the string */
	int
__ssUngetc( int ch, void *sp )
{
	return (EOF == ch) ? EOF : *(--(*(char **)sp));
}

/* helper functions to getc and ungetc to file */
	int
__fsGetc( void *fp )
{
	int ch = __lastChar;

	if( LC_NONE == __lastChar ) return fgetc((FILE *)fp);
	__lastChar = LC_NONE;
	return ch;
}

/* we push back only one character (this is all __scanner needs */
	int
__fsUngetc( int ch, void *fp )
{
	fp = fp; /* shut off compiler, will be optimized out */
	return __lastChar = ch;
}

/* helper functions to getc and ungetc from the console */
	int
__csGetc( void *cp )
{
	int ch = __lastChar;

	cp = cp; /* shut off compiler, will be optimized out */
	if( LC_NONE == __lastChar ) return getchar();
	__lastChar = LC_NONE;
	return ch;
}

/* we can only push back one character, but this is all __scanner needs */
	int
__csUngetc( int ch, void *cp )
{
	cp = cp; /* shut off compiler, will be optimized out */
	return __lastChar = ch;
}

/*
 
   These are the callable functions (simply wrappers around __scanner)
 
*/

/* scanf wrapper */
	int
scanf( const char *fmt, ... )
{
	va_list ap;

	assert(fmt);
	va_start(ap, fmt);
	return __scanner( __csGetc, __csUngetc, NULL, fmt, ap );
}

/* sscanf wrapper */
	int
sscanf( const char *buf, const char *fmt, ... )
{
	va_list ap;

	assert( buf && fmt );
	va_start(ap, fmt);
	return __scanner( __ssGetc, __ssUngetc, &buf, fmt, ap );
}

/* fscanf wrapper */
	int
fscanf( FILE *fp, const char *fmt, ... )
{
	va_list ap;

	assert( fp && fmt );
	va_start(ap, fmt);
	return __scanner( __fsGetc, __fsUngetc, fp, fmt, ap );
}

/* vscanf wrapper */
	int
vscanf( const char *fmt, va_list ap )
{
	assert(fmt);
	return __scanner( __csGetc, __csUngetc, NULL, fmt, ap );
}

/* vsscanf wrapper */
	int
vsscanf( const char *buf, const char *fmt, va_list ap )
{
	assert(fmt && buf);
	return __scanner( __ssGetc, __ssUngetc, &buf, fmt, ap);
}

/* vfscanf wrapper */
	int
vfscanf( FILE *fp, const char *fmt, va_list ap )
{
	assert(fp && fmt);
	return __scanner( __fsGetc, __fsUngetc, fp, fmt, ap );
}

#undef GETCHAR
#undef UNGETCHAR
#undef ARRAYSZ
#undef SCANTBL_SZ
#undef BITSET
#undef LC_NONE
