/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "bintree.h"

zBinTree::zBinTree(fpCompFunc cmp, Boolean purge)
{
	compare = cmp;
	purging = purge;
	cleanup = 0;
	current = 0;
	parent  = 0;
	root    = 0;
	count   = 0;
}

zBinTree::~zBinTree()
{
	destroy(root);
}

void
zBinTree::destroy(zBTNode *node)
{
	if( node )
	{
		destroy(node->left);
		destroy(node->right);
		if( purging )
		{
			if( cleanup ) cleanup(node->data);
			else delete node->data;
		}
		delete node;
	}
}

void
zBinTree::purge()
{
	destroy(root);
	root = current = 0;
	parent = 0;
	count = 0;
}

Boolean
zBinTree::insert(void *data)
{
	zBTNode *node;

	if( !root )
	{
		root = new zBTNode(data);
		current = root;
		parent  = 0;
		count   = 1;
		return root ? True : False;
	}

	for( node = root; ; )
	{
		int cmp = compare(data, node->data);

		if( 0 == cmp ) return False;  // duplicate

		if( cmp < 0 )
		{
			if( !node->left )
			{
				node->left = new zBTNode(data);
				current = node->left;
				break;
			}
			node = node->left;
		}
		else if( cmp > 0 )
		{
			if( !node->right )
			{
				node->right = new zBTNode(data);
				current = node->right;
				break;
			}
			node = node->right;
		}
	}

	// current points to the inserted node
	if( current )
	{
		parent = node;
		count++;
		return True;
	}

	parent = 0;
	return False;
}

void*
zBinTree::find(void *data)
{
	for( parent = 0, current = root; current; )
	{
		int cmp = compare(data, current->data);

		if( 0 == cmp ) return current->data;

		parent = current;

		if( cmp < 0 ) current = current->left;
		else if( cmp > 0 ) current = current->right;
	}

	return current;
}

void
zBinTree::update(void *data)
{
	if( current )
	{
		if( purging )
		{
			if( cleanup ) cleanup(current->data);
			else delete current->data;
		}
		current->data = data;
	}
}

void
zBinTree::remove(void *data)
{
	if( find(data) ) remove();
}

// 'parent' points to the parent of 'current' (if any, 0 otherwise)
void
zBinTree::remove()
{
	zBTNode *old_node = current;
	zBTNode **parent_link = 0;

	if( parent )
	{
		if( parent->left == current ) parent_link = &parent->left;
		else parent_link = &parent->right;
	}

	if( current )
	{
		// no right child (also handles no children at all)
		if( !current->right ) current = current->left;
		// right child without a left child of its own
		else if( !current->right->left )
		{
			current = current->right;
			current->left = old_node->left;
		}
		// find the leftmost node in the right child subtree
		// this is a node that has no left child link
		else
		{
			zBTNode *small = current->right;

			while( small->left->left ) small = small->left;
			current = small->left;
			small->left = current->right;
			current->left = old_node->left;
			current->right = old_node->right;
		}

		if( parent ) *parent_link = current;
		else root = current;

		if( purging )
		{
			if( cleanup ) cleanup(old_node->data);
			else delete old_node->data;
		}
		delete old_node;
		count--;
	}
}

void
zBinTree::forEach(fpAppFunc af, void *arg, tree_order order)
{
	action     = af;
	action_arg = arg;

	switch( order )
	{
		case in  : in_order(root);   break;
		case pre : pre_order(root);  break;
		case rev : rev_order(root);  break;
		case post: post_order(root); break;
	}
}

void
zBinTree::in_order(zBTNode *node)
{
	if( node )
	{
		in_order(node->left);
		action(node->data, action_arg);
		in_order(node->right);
	}
}

void
zBinTree::pre_order(zBTNode *node)
{
	if( node )
	{
		action(node->data, action_arg);
		pre_order(node->left);
		pre_order(node->right);
	}
}

void
zBinTree::post_order(zBTNode *node)
{
	if( node )
	{
		post_order(node->left);
		post_order(node->right);
		action(node->data, action_arg);
	}
}

void
zBinTree::rev_order(zBTNode *node)
{
	if( node )
	{
		rev_order(node->right);
		action(node->data, action_arg);
		rev_order(node->left);
	}
}
