/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#ifndef INCLUDED_PBTFRAME_H
#define INCLUDED_PBTFRAME_H

/*
 * t i m e f r a m e   s t r u c t u r e   m a n i p u l a t i o n
 * 
 * advanced inlined implementation of the timeframe structure access class
*/
#include "typedef.h"
#include "comdef.h"

#ifndef PB_SDK
	#include <time.h>
#else
	#include "pblibc.h"
#endif

/*
 * TimeFrame for certain day, hours are in 6 bytes, 48 bits for
 * half-hour segments. For example, from 0:00 to 0:30, bit 1 of byte 1,
 * from 0:30 to 1:00, bit 2 of byte 1, etc. Days start from Monday = 0
 * and go to Sunday = 6.
 *
 * NOTE: for all functions, the tm structure needs to be filled by the
 * caller with the desired time and day of week information in
 * tm-compatible format, i.e., the day of week starts at Sunday = 0.
 * The only fields used by the function are tm_wday, tm_hour and tm_min.
 *
 * An easy way to calculate the day of the week is to fill the rest of
 * the structure and then call mktime() which will do it for you!
*/
class zTimeFrame
{
public:
	static Boolean Status(const TimeFrame frame, const struct tm *timer);
	static void    Enable(TimeFrame frame, const struct tm *timer);
	static void    Disable(TimeFrame frame, const struct tm *timer);
	static void    Toggle(TimeFrame frame, const struct tm *timer);

private:
	static char bitno(int hour, int min);
	static void adjust(int &day);
};

/*
 * This routine adjusts the weekday to a TimeFrame-compatible value. The
 * standard library uses Sunday = 0, we need Monday = 0 and Sunday = 6.
*/
inline void
zTimeFrame::adjust( int &day )
{
	if( 0 == day ) day = 6;
	else day--;
}

/*
 * This calculates the bit number for a given hour and minute of the day.
 * The half-hour byte is in hour / 4, so the bit we need to calculate is
 * in hour modulus 4 (ranging from 0 to 3) multiplied by 2, which gets us
 * the range of halves, next we need to add one if it's past the 30th minute.
*/
inline char
zTimeFrame::bitno( int hour, int mins )
{
	return ( ((hour % 4) << 1) + ((mins > 30) ? 1 : 0) );
}

inline Boolean
zTimeFrame::Status(const TimeFrame frame, const struct tm *t)
{
	int day = t->tm_wday;

	adjust(day);
	return Boolean(
		iTestBit(frame[day][t->tm_hour >> 2], bitno(t->tm_hour, t->tm_min))
	);
}

inline void
zTimeFrame::Enable(TimeFrame frame, const struct tm *t)
{
	int day = t->tm_wday;

	adjust(day);
	iSetBit(frame[day][t->tm_hour >> 2], bitno(t->tm_hour, t->tm_min));
}

inline void
zTimeFrame::Disable(TimeFrame frame, const struct tm *t)
{
	int day = t->tm_wday;

	adjust(day);
	iClearBit(frame[day][t->tm_hour >> 2], bitno(t->tm_hour, t->tm_min));
}

inline void
zTimeFrame::Toggle(TimeFrame frame, const struct tm *t)
{
	int day = t->tm_wday;

	adjust(day);
	iToggleBit(frame[day][t->tm_hour >> 2], bitno(t->tm_hour, t->tm_min));
}

#endif /* INCLUDED_PBTFRAME_H */
