/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#ifndef INCLUDED_PBSTRUCT_H
#define INCLUDED_PBSTRUCT_H
#include "typedef.h"

/*
 * p r o b o a r d   f i l e   s t r u c t u r e s
 * 
 * these are all proboard file structures with some notes on file processing.
*/

/*
 
  LIMITS.PRO - stores the definitions for the various levels. the file has
  an unknown number of the LIMIT records defined below (also see Limits)
 
*/
typedef struct{
	ushort level;            //....................................user level
	short  timelimit;        //....................................time limit
	short  daily_klimit;     //..........................daily download limit
	short  pre_download;     //.......................pre-download time limit
	char   id[6];            //..................................usergroup id
	ushort free;             //.............................free upload in kb
	uchar  factor;           //....................percentage upload required
	ushort max_download;     //...................max download for this level
	ushort fallto;           //.............fall to level x when max. reached
	short  msgfactor;        //..........# kbytes granted per message written
	char   extra[5];
} LIMIT;

/*
 
  FILECFG.PRO - stores the file area configuration. The file has an unknown
  (but no more than 10,000) records of the FILECFG structure. When processing
  this file you need to note that the filepath field can contain a FA_n.CTL
  file instead of a directory, where 'n' is the current number of the area
  (1-based). If this is the case, the program should read this text file for
  a list of the directories for the file area (one per line, full specifica-
  tion D:\DIR\SUBDIR) for the location of the files. The listpath field will
  contain the combined listing for all files in all directories listed in the
  control file. Note that while the full path to the .CTL file will be in the
  filepath field, the file will ALWAYS reside in the PB system directory
  (required by ProBoard). Also note that some areas can be empty (in which
  case the name will be an empty string ""). Your should account for these.
 
*/
#define DATE_MMDDYY 0   //...........use MM-DD-YY date format in the listings
#define DATE_DDMMYY 1   //......................use DD-MM-YY (European-style)
#define DATE_YYMMDD 2   //......................use YY-MM-DD (Japanese-style)

typedef struct{
	char   name[80];        //..............................name of file area
	char   listpath[80];    //.............................path for file-list
	char   filepath[80];    //.................................path for files
	ushort level;           //...............level needed to access file area
	ulong  flags;           //...............flags needed to access file area
	BOOL   copyLocal;       //.................true = cdrom file listing type
	short  maxfiles;        //..............max files per day in this area dl
	short  maxkb;           //................max kbytes per day in this area
	BOOL   notops;          //.................set to true if not topfiles.a*
	BOOL   free;            //.....................if true, this area is free
	uchar  groups[4];       //............................groups belonging to
	BOOL   allGroups;       //..........................belongs to all groups
	uchar  minAge;          //....................................minimum age
	ulong  flagsNot;        //...................................access flags
	uchar  dateFormat;      //......................date format, see date_???
	char   extra[2];
} FILECFG;
typedef FILECFG FILEAREA;   //...........create an alias (some people use it)

/*
 
  AKA - the AKA.PRO file contains an unknown number of those records. These
  are the standard FidoNet definitions. Note that some implementations treat
  the fields as unsigned shorts, which is not correct according to the
  FTSC-002 standard. The definitions below follow the standard.
 
*/
typedef struct{
	short zone;    //............................zone number (always present)
	short net;     //..............................................net number
	short node;    //.............................................node number
	short point;   //...................if not a point system, this will be 0
} FIDO_NODE, AKA;
//typedef FIDO_NODE AKA;     //.....................create an alias for the SDK

/*
 
  BINLOG.PB - the binary log file. Records are stored for each call to the
  BBS, with the most recent calls appearing at the end of the log file
  (inconvenient). The number of records is determined by the configuration
  entry in CONFIG.binlogdays. Also, local calls my not be logged. This is
  determined by binloglocal in the CONFIG record too.
 
*/
typedef struct{
	DateType date;          //...............................date of the call
	TimeType timeIn;        //..............................time call started
	TimeType timeOut;       //...........................time user logged out
	char     name[36];      //......................................user name
	char     city[26];      //....................................user's city
	char     country[26];   //.................................user's country
	long     baud;          //..........................baud rate of the call
	ushort   node;          //.............node number that accepted the call
	long     kbDown;        //..............kbytes downloaded during the call
	long     kbUp;          //................kbytes uploaded during the call
	ushort   yells;         //.................times the user paged the sysop
	ushort   level;         //.......................access level of the user
	ulong    uflags;        //...................................user's flags
	char     alias[36];     //...................................user's alias
	uchar    extra[45];
} BINLOG;

/*
 
  TAGGED_FILE - this is the structure used internally by ProBoard for the
  tagged file lists, but it is also the structure used in the TAGLIST.PB
  file (defined below), that's why it is shown here as a file structure.
 
*/
typedef struct{
	char   name[13];    //...................................name of the file
	ushort area;        //..............# of area where the file is (1-based)
} TAGGED_FILE;

/*
 
  TAGLIST.PB - lists the files tagged by users which were not downloaded.
  Each entry has a TAGLIST header, followed by numTagged number of records
  of TAGGED_FILE structure. Thus, each user will have her/his own TAGLIST
  header and TAGGED_FILE entries for each file in the tagged list.
 
*/
typedef struct{
	char  name[36];    //...............user's name (as in the user's record)
	short numTagged;   //......number of TAGGED_FILE entries following header
} TAGLIST;

/*
 
  USERS.BBS - RemoteAccess-compatible userbase file. Note that the strings
  in this file are Pascal, not C (i.e. you must use the string conversion
  routines from PB-Lib before the strings are usable. Also note that the time
  is stored as a Pascal string (HH:MM) and the date - as a Pascal string too
  (MM-DD-YY). There are two attribute bytes, and the constants are split into
  two accordingly (RA_UFLAG1_xxx) and (RA_UFLAG2_xxx). If 'n/a' appears in
  the description comment, the flag is not used by ProBoard.
 
*/
#define RA_UFLAG1_DELETED    0x01    //.......................user is deleted
#define RA_UFLAG1_CLEAR      0x02    //..........................clear screen
#define RA_UFLAG1_PAUSE      0x04    //...........................more prompt
#define RA_UFLAG1_ANSI       0x08    //..................................ansi
#define RA_UFLAG1_NOKILL     0x10    //...............................no kill
#define RA_UFLAG1_IGNORE     0x20    //.....................transfer priority
#define RA_UFLAG1_FSED       0x40    //....................full-screen editor
#define RA_UFLAG1_QUIET      0x80    //............................quiet mode
#define RA_UFLAG2_HOTKEYS    0x01    //..............................hot-keys
#define RA_UFLAG2_AVATAR     0x02    //..............................avatar/0
#define RA_UFLAG2_FSVIEW     0x04    //......full-screen message viewer (n/a)
#define RA_UFLAG2_HIDDEN     0x08    //.........................hide activity
#define RA_UFLAG2_PAGEPRI    0x10    //.........................page priority
#define RA_UFLAG2_NOECHOMAIL 0x20    //.......no echo mail when checked (n/a)
#define RA_UFLAG2_GUEST      0x40    //.........................guest account
#define RA_UFLAG2_POSTBILL   0x80    //.......................post bill (n/a)

typedef struct{
	char       name[36];            //............................user's name
	char       city[26];            //.......................user's home city
	char       company[51];         //.........................user's company
	char       address1[51];        //....................address, first line
	char       address2[51];        //...................address, second line
	char       address3[51];        //....................address, third line
	char       alias[36];           //...........................alias/handle
	char       comment[81];         //................................comment
	ulong      passWordCRC;         //.............32-bit crc of the password
	char       dataPhone[16];       //.............................data phone
	char       voicePhone[16];      //............................voice phone
	TimeRaType lastTime;            //................last login time (hh:mm)
	DateRaType lastDate;            //.............last login date (mm-dd-yy)
	uchar      attribute1;          //....................flags: see RAF1_???
	uchar      attribute2;          //....................flags: see RAF2_???
	uchar      aFlags[4];           //..............user flags (not a dword!)
	ulong      credit;              //.........................netmail credit
	ulong      pending;             //.........................pending credit
	ushort     msgsPosted;          //..............number of posted messages
	ushort     level;               //..................user's security level
	ulong      highMsgRead;         //......................last message read
	ulong      timesCalled;         //..................total number of calls
	ulong      numUploads;          //................total number of uploads
	ulong      numDownloads;        //..............total number of downloads
	ulong      kbUploaded;          //..................total kbytes uploaded
	ulong      kbDownloaded;        //................total kbytes downloaded
	ulong      kbToday;             //................kbytes downloaded today
	ushort     timeUsed;            //..............time used today (minutes)
	short      screenLength;        //......user's screen length (# of lines)
	uchar      lastPwdChange;       //...............??? last password change
	ushort     group;               //....................................???
	short      combinedBoards[200]; //....................??? combined boards
	DateRaType firstDate;           //.........date of first login (mm-dd-yy)
	DateRaType birthDate;           //............user's birthdate (mm-dd-yy)
	DateRaType subDate;             //....................??? additional date
	uchar      screenWidth;         //.....user's screen width (# of columns)
	uchar      language;            //...........................??? language
	uchar      dateFormat;          //............................date format
	char       forwardTo[36];       //....................forward messages to
	ushort     msgArea;             //.................last message area used
	ushort     fileArea;            //....................last file area used
	uchar      defaultProtocol;     //..............default transfer protocol
	short      fileGroup;           //...................last file group used
	uchar      lastDOBCheck;        //...................??? last birth check
	uchar      sex;                 //.........0: unknown, 1: male, 2: female
	ulong      USERSXIrec;          //........number of record in usersxi.bbs
	short      msgGroup;            //................last message group used
	char       extra[48];
} USERS;

/*
 
  USERSPB.BBS - the auxiliary userbase. This file contains ProBoard-specific
  data which complements the information found in USERS.BBS and USERSXI.BBS
  files. It is tied to those two files. The records are stored in the same
  order as USERS.BBS. Since the user name is duplicated in this file, it is
  possible to fix the userbase after using a utility which does not know
  about this file. Refer to the manual for WriteUser() for info. The
  PB_UFLAG_???? constants are for the uFlag field of the record. They should
  be ANDed, XORer or ORed with the field (i.e. they are bitmap masks).
 
*/
#define PB_UFLAG_NOIBM      0x00000001L  //.............no IBM extended ASCii
#define PB_UFLAG_NOTOPS     0x00000002L  //.............don't include in tops
#define PB_UFLAG_AVTPLUS    0x00000004L  //...........use Avatar/0+ emulation
#define PB_UFLAG_ATTEN      0x00000008L  //..........sound attention for user
#define PB_UFLAG_NORIP      0x00000010L  //..............disable RIP for user
#define PB_UFLAG_MULTILOGIN 0x00000020L  //.............allow multiple logins
#define PB_UFLAG_FREECHAT   0x00000040L  //..........free time while chatting
#define PB_UFLAG_LOCALONLY  0x00000080L  //.........only local logins allowed

#define LOG_FRIEND		0  //..............log some info (log always in Log()
#define LOG_NORMAL      1  //.............................normal: doors, etc.
#define LOG_SUSPICIOUS	2  //...............extensive logs: menu change, etc.
#define LOG_DANGEROUS	3  //.................log everything: each key typed!
#define LOG_IGNORE      4  //............ignore when logging (nothing logged)

typedef struct{
	char     name[36];				  //.............user's name (mixed case)
	char     country[26];			  //.......................user's country
	char     state[26];				  //.........................user's state
	char     faxPhone[16];			  //................user's fax/data phone
	char     passWord[16];			  //......................user's password
	char     language[9];			  //......................user's language
	DateType lastPwdChange;			  //..........date of last password check
	DateType lastNewFilesCheck;		  //.........date of last new files check
	short    logLevel;				  //...............log level: see LOG_xxx
	short    tbTimeBalance;			  //........time bank: time balance (min)
	short    tbKbBalance;			  //............time bank: kbytes balance
	short    tbTimeWithdrawn;		  //......time bank: time withdrawn (min)
	short    tbKbWithdrawn;			  //..........time bank: kbytes withdrawn
	ushort   tbTimeDeposited;		  //......time bank: time deposited (min)
	ushort   tbKbDeposited;			  //..........time bank: kbytes deposited
	ushort   tbTimeLoaned;			  //.........time bank: time loaned (min)
	ushort   tbKbLoaned;			  //.............time bank: kbytes loaned
	DateType tbTimePayback;			  //........time bank: payback time (min)
	DateType tbKbPayback;			  //............time bank: payback kbytes
	DateType tbLastUsed;			  //............time bank: date last used
	ushort   expLevel;				  //.......level to set when acct expires
	ulong    expFlagsOn;			  //........flags to turn on when expires
	ulong    expFlagsOff;			  //.......flags to turn off when expires
	ulong    uFlags;				  //........user flags (see PB_UFLAG_xxx)
	combinedboards mailCheckBoards;   //.......areas to scan for mail (1,000)
	ulong    totalTimeUsed;			  //........total time used on the system
	ushort   qwkMaxMsgsPerArea;		  //...........qwk: max messages per area
	ushort   qwkMaxMsgs;			  //..............qwk: max total messages
	uchar    qwkArchiver;			  //.................qwk: archiver to use
	uchar    ripFont;				  //................small or big RIP font
	uchar    checkMail;				  //.............should we check for mail
	uchar    checkNewFiles;			  //........should we check for new files
	uchar    extra[398];
} USERSPB;

/*
 
  USERSIDX.BBS - the userbase flat index file. This file contains the 32-bit
  CRC values for the user name and alias (both upper-case). Note that the
  regular 32-bit CRC generation routines will produce a value which needs to
  be complemented (with the '~' operator) before it can be used on the index.
  Once you locate the record by name or alias CRC, the relative number of the
  record from the beginning of the file will be the number of the record in
  USERS.BBS and USERSPB.BBS files.
 
*/
typedef struct{
	ulong nameCRC;  //..............................32-bit CRC of user's name
	ulong aliasCRC; //.............................32-bit CRC of user's alias
} USERSIDX;

/*
 
  USERSXI.BBS - this file is tied to the USERS.BBS, USERSIDX and USERSPB
  files. It contains one record per user and this record is curerntly unused
 
*/
typedef struct{
	char unused[200];     //..........currently unused empty space (set to 0)
} USERSXI;

/*
 
  CONFIG.PRO - this is the main configuration file. Note that it is also
  the stucture for the *CurUser->config and *Config pointers. Note that
  Philippe has three different versions of this structure (the one in the
  PB_SDK.H, the one in the file PB_STRUCT.215 and the one I got as a beta-
  tester). The one below is a combination of the three and as far as I can
  tell, it is the correct one... duh! I wish there was consistency.
 
*/
typedef struct{
	char      shellmsg[81];         //..........message to show when shelling
	char      sysopname[36];        //..........................name of sysop
	char      txtpath[61];          //.....................path for textfiles
	char      mnupath[61];          //....................path for menu-files
	char      msgpath[61];          //..................path for message base
	char      uploadpath[61];       //............................upload path
	char      editorname[61];       //................name of external editor
	ushort    newuserlevel;         //.....................level for new user
	ushort    newuserloglevel;      //..................loglevel for new user
	ulong     newuserflags;         //.........................new user flags
	short     max_passinput;        //........max attempts for password entry
	short     min_passlength;       //................minimum password length
	short     inactivity_time;      //..............inactivity time-out limit
	short     max_sysop_pages;      //.......maximum times sysop can be paged
	short     pagebell_length;      //.............length of page-bell (secs)
	short     mailcheck;            //...............check for mail at logon?
	short     europe;               //..................european date format?
	short     numnodes;             //................................# nodes
	BOOL      allowansi;            //............................allow ansi?
	BOOL      allowavatar;          //...............allow avatar/0 or avt/0+
	short     discrete;             //...................hide sysop activity?
	short     askphone;             //..................ask for phone number?
	short     allowoneword;         //...................allow one-word names
	ushort    crashlevel;           //.............level needed for crashmail
	ulong     crashflags;           //.............flags needed for crashmail
	ushort    attachlevel;          //...........level needed for file attach
	ulong     attachflags;          //...........flags needed for file attach
	short     allowmsgupload;       //..................allow message uploads
	short     allowstacking;        //.................allow command stacking
    uchar     spare1[6];
	short     handshaking;          //........................i/o handshaking
	short     allowalias;           //..................allow alias for login
	short     loglocal;             //........................log local calls
	short     doswap;               //.........................allow swapping
	char      originline[61];       //............................origin line
	char      nodelistdir[61];      //.....................nodelist directory
	char      sysopkeys[10][60];    //..........................sysop hotkeys
	uchar     spare2[6];
	short     uploadspace;          //...............space needed for uploads
	char      pvtuploadpath[61];    //..........directory for private uploads
	char      quotestring[6];       //................string used for quoting
	short     fastmode;             //......use fast mode (needs more memory)
    BOOL      killsent;             //................kill netmail after sent
    BOOL      multiline;            //.......use message base sharing/locking
	BOOL      egamode;              //....................use 43/50 line mode
	BOOL      showuserinfo;         //......show user info while in ega mode?
	char      pexpath[61];          //................directory for pex files
	short     allowquicklogin;      //...............allow quick sysop login?
	short     securityboard;        //.........msgboard for security messages
	short     pwdmessages;          //...............write security-messages?
	char      bbsname[36];          //........................name of the bbs
	char      pwdchar;              //.....................password character
	short     tb_maxtimedeposit;    //........time bank: max time deposit/day
	short     tb_maxkbdeposit;      //......time bank: max kbytes deposit/day
	short     tb_maxtimewithdrawal; //.....time bank: max time withdrawal/day
	short     tb_maxkbwithdrawal;   //....time bank: max kbyte withdrawal/day
	short     usage_days;           //..............days to keep usage graphs
	char      systempwd[16];        //........................system password
	BOOL      usesystempwd;         //...................use system password?
	BOOL      askbirthdate;         //........................ask birth date?
	short     binlogdays;           //.............# days to log in binlog.pb
	BOOL      binloglocal;          //....log local calls to binlog.pb yes/no
	short     pageArea;             //..........area number for page messages
	BOOL      indexfiles;           //...........use file indexing (obsolete)
	BOOL      checkdupes;           //........................check for dupes
	BOOL      killdupes;            //...................kill duplicate files
	BOOL      ignore_ext;           //......no file extensions for dupe check
	char      RIPpath[61];          //...................path for rip scripts
	char      iconspath[61];        //.....................path for rip icons
	char      location[36];         //....................bbs location (city)
	char      phone[26];            //............................bbs phone #
	char      QWKid[9];             //.............................bbs qwk id
	ushort    IObuffersize;         //...............i/o buffer size in bytes
	TimeFrame pagingHours;          //...........................paging hours
	char      defaultLanguage[9];   //.......................default language
	BOOL      addUploaderName;      //.......add uploader's name to files.bbs
	TimeFrame downloadHours;        //.........................download hours
	BOOL      askdataphone;         //.......................ask data phone #
	BOOL      askfaxphone;          //........................ask fax phone #
	BOOL      askaddress;           //....................ask mailing address
	BOOL      asksex;               //................................ask sex
	BOOL      askdateformat;        //........................ask date format
	BOOL      askstate;             //..............................ask state
	BOOL      askcountry;           //............................ask country
	ushort    fuzzyRate;            //................fuzzy search percentage
	BOOL      hidePassword;         //...........hide password in user editor
	BOOL      valConfirm;           //................confirm user validation
	uchar     spare3[17];
	char      extChat[81];          //..................external chat program
	char      virScanCommand[61];   //...........external upload scan command
	uchar     virScanType;          //.....external upload scan command: type
	char      virScanSemaphore[13]; //....ext. upload scan command: semaphore
	uchar     listColorTag;         //.............file list color (tag char)
	uchar     listColorFileName;    //............file list color (file name)
	uchar     listColorSize;        //............file list color (file size)
	uchar     listColorDate;        //............file list color (file date)
	uchar     listColorCounter;     //.........file list color (file counter)
	uchar     listColorDesc;        //..........file list color (description)
	uchar     listColorSeperator;   //.....file list color (description sep.)
	uchar     listColorMissing;     //....file list color (missing indicator)
	BOOL      hideFileCounter;      //....file list setup - hide file counter
	BOOL      disableRIP;           //..........true = disable rip completely
	uchar     spare4[81];
	uchar     virScanFailedAction;  //.......ext. upload scan command: action
	ushort    virScanFailedArea;    //..........ext. upload scan: failed area
	uchar     promptColor;          //..............prompt color (background)
	BOOL      internalfsed;         //......set to 1 if internal fsed enabled
	uchar     spare5[3];
} CONFIG;

/*
 
  TIMELOG.PRO - This file has only one record of the type below. It records
  the total system statistics, but most of the fields are now considered
  obsolete (mainly because they can be calculated from the data in the
  BINLOG file, which is what _GRAPH does)
 
*/
typedef struct{
	short    numdays;      //...........................number of days active
	DateType lastdate;     //.....................................last update
	long     hours[24];    //.............# minutes per hour usage (obsolete)
	long     days[7];      //.............# minutes per day usage  (obsolete)
	long     weeks[53];    //.............# minutes per week usage (obsolete)
	long     totalcalls;   //...........................total calls to system
} TIMELOG;

/*
 
  MESSAGES.PB - This is the file with the message area definitions. Note
  that the format has changed since v2.01 and so has the name of the file.
  As with the file areas, some entries can have a NUL string ("") in the
  name field. This means that this slot is not a valid area and should not
  be used. Programs must take that into consideration. Also, the Hudson
  message board number is apparently not used at all. The areaNum field
  also appears not to be used (but it is guaranteed to be the same as the
  relative number of the record in the file - the number that the area uses
  for identification. For *.MSG areas, the 'path' field will have a directory
  specification. For Squish and JAM, it will have a directory plus the root
  file name (without the extension). For Hudson, this field should be ignored
  and the CONFIG.msgpath should be used instead. The 'replyBoard' field will
  be 0 (zero) is replies go to the same area. The 'aka' field is the 0-based
  number of the record in AKA.PRO file (which lists the FidoNet-compatible
  addresses). Warning: do NOT modify the hudson board #, it MUST be 0 for now
 
*/
#define MSGKIND_LOCAL    0  //..........................................local
#define MSGKIND_NET      1  //........................................netmail
#define MSGKIND_ECHO     2  //.......................................echomail
#define MSGKIND_PVTECHO  3  //...................................pvt echomail

#define MSGTYPE_BOTH     0  //.................................private/public
#define MSGTYPE_PVT      1  //...................................private only
#define MSGTYPE_PUBLIC   2  //....................................public only
#define MSGTYPE_TOALL    3  //.........................................to all

#define MSGBASE_HUDSON   0  //.........................................hudson
#define MSGBASE_SQUISH   1  //.........................................squish
#define MSGBASE_SDM      2  //.....................................fido *.msg
#define MSGBASE_JAM      3  //............................................jam

#define MSGFLAG_NAME     0  //................................real names only
#define MSGFLAG_ALIAS    1  //............................free alias (choose)
#define MSGFLAG_BOTH     2  //.......................fixed alias or real name
#define MSGFLAG_ENFORCED 3  //...........................fixed alias enforced

typedef struct{
	ushort areaNum;         //...................# of message area (1-10,000)
	ushort hudsonBase;      //..................number of hudson message base
	char   name[81];        //...........................name of message area
	uchar  msgType;         //...............kind of message area MSGTYPE_???
	uchar  msgKind;         //....................type of message MSGKIND_???
	uchar  msgBaseType;     //...............type of message base MSGBASE_???
	char   path[81];        //........................path to squish or *.MSG
	uchar  flags;           //.......................posting type MSGFLAG_???
	ushort readLevel;       //..............minimum level needed to read msgs
	ulong  readFlags;       //......................flags needed to read msgs
	ulong  readFlagsNot;    //...................flags non-grata to read msgs
	ushort writeLevel;      //.............minimum level needed to write msgs
	ulong  writeFlags;      //.....................flags needed to write msgs
	ulong  writeFlagsNot;   //...................flags non-grata to read msgs
	ushort sysopLevel;      //............minimum level needed to change msgs
	ulong  sysopFlags;      //....................flags needed to change msgs
	ulong  sysopFlagsNot;   //...................flags non-grata to read msgs
	char   origin[62];      //....................................origin line
	short  aka;             //..............................aka record number
	ushort rcvKillDays;     //....................kill received after xx days
	ushort msgKillDays;     //.............................kill after xx days
	ushort maxMsgs;         //.....................................max # msgs
	char   sysop[36];       //.....................................area sysop
	short  replyBoard;      //............area number where replies should go
	char   echoTag[61];     //..............................echomail tag name
	char   qwkTag[13];      //..................................qwk area name
	uchar  groups[4];       //............................groups belonging to
	BOOL   allGroups;       //..........................belongs to all groups
	uchar  minAge;          //......................minimum age for this area
	uchar  extra[112];
} MESSAGES;
typedef MESSAGES MSGAREA;   //...........................alias for easy usage

/*
 
  ONLINE.PRO - Information about the user(s) online. This file will contain
  a record for each node, with the record position == node number minus 1
 
*/
typedef struct{
	char   name[36];  //..................................name of user online
	char   city[16];  //..................................city of user online
	ushort baud;      //....................................baud rate of user
	short  status;    //...........0 -> online , <0 -> not online/unavailable
	char   extra[8];
} ONLINE;

/*
 
  TOPS.PB - The top user statistics file. It contains the 'Best 20 Users'
  statistics. This file is divided into 7 sections (for the 7 different
  stats) and each section has 20 records of the format described below. The
  numbering of the sections is as follows:

            1. Callers by total number of calls (stats = # of calls)
            2. Downloaders by KBytes (stats = # of KBytes)
            3. Downloaders by # of files (stats = # of files)
            4. Uploaders by KBytes (stats = # of KBytes)
            5. Uploaders by # of files (stats = # of files)
            6. Message writers (stats = # messages written)
            7. Callers by time online (stats = minutes online)

  Note that the file name has been changed from TOPS.PRO in v2.0x to TOPS.PB
  in versions 2.10 and later. The format appears to be the same, though.
 
*/
typedef struct{
	char  name[36];  //..........................................name of user
	ulong n;         //......................data (# kb/minutes/files/etc...)
} TOPS;

/*
 
  PROTOCOL.PRO - the protocol definition file. Each installed protocol has
  a record of the type described below.
 
*/
#define PROT_BATCH   1    //...................................batch protocol
#define PROT_ENABLED 2    //..........................................enabled
#define PROT_BOTH    4    //.............................full-duplex protocol
#define PROT_BIM     8    //............................bimodem-type ctl-file
#define PROT_LOCAL  16    //.......................................local only

typedef struct{
	char  name[50];      //..................................name of protocol
	char  key;           //...............................hotkey for protocol
	char  flags;         //......................protocol behavior (PROT_???)
	char  logfile[60];   //.............................logfile from protocol
	char  ctlfile[60];   //...................control file (redirection file)
	char  dlcom[128];    //.............................download command line
	char  ulcom[128];    //...............................upload command line
	char  dlstring[80];  //..........string to place in download control file
	char  ulstring[80];  //............string to place in upload control file
	char  dlkeyword[40]; //.........download keyword to search for in logfile
	char  ulkeyword[40]; //...........upload keyword to search for in logfile
	short wordnr;        //.............file name word # after keyword (1-..)
} PROTOCOL;

/*
 
  MODEM.PB - the modem definition file (hardware setup). Note that ProBoard
  allows node-specific modem setups that override the definitions from this
  file. The node-specific data will be stored in files names MODEMnnn.PB,
  where 'nnn' is zero-padded node number, i.e. for node 3, the file would be
  names MODEM003.PB. Those files will have the same structure as MODEM.PB and
  they don't have to exist. That is, you can have custom configs for some of
  the nodes and not for others (which will use the defaults below).
 
*/
#define MODEM_LOCKED        (0x00000001L)
#define MODEM_MANUAL_ANSWER (0x00000002L)

typedef struct{
	long  maxBps;             //............................maximum baud rate
	long  lockedBps;          //.................locked baud rate  (not used)
	ulong flags;              //.......................attributes (see below)
	short port;               //...............................com port (1-8)
	short commandDelay;       //.......delay in 1/10s between characters sent
	short answerDelay;        //.....delay in 1/10s before answer string sent
	short blankTime;          //...........time in seconds for screen blanker
	char  msgCon300[ 80];     //...........................connect 300 string
	char  msgCon1200 [80];    //..........................connect 1200 string
	char  msgCon1275 [80];    //..........................connect 1275 string
	char  msgCon2400 [80];    //..........................connect 2400 string
	char  msgCon4800 [80];    //..........................connect 4800 string
	char  msgCon7200 [80];    //..........................connect 7200 string
	char  msgCon9600 [80];    //..........................connect 9600 string
	char  msgCon12000 [80];   //.........................connect 12000 string
	char  msgCon14400 [80];   //.........................connect 14400 string
	char  msgCon16800 [80];   //.........................connect 16800 string
	char  msgCon19200 [80];   //.........................connect 19200 string
	char  msgCon21600 [80];   //.........................connect 21600 string
	char  msgCon24000 [80];   //.........................connect 24000 string
	char  msgCon26400 [80];   //.........................connect 26400 string
	char  msgCon28800 [80];   //.........................connect 28800 string
	char  msgCon38400 [80];   //.........................connect 38400 string
	char  msgCon57600 [80];   //.........................connect 57600 string
	char  msgCon64000 [80];   //.........................connect 64000 string
	char  msgCon115200 [80];  //........................connect 115200 string
	char  msgConExternal[80]; //..................external/fax connect string
	char  msgRing[80];        //..................................ring string
	char  msgOk[80];          //.................................."ok" string
	char  cmdInit1[80];       //........................init command string 1
	char  cmdInit2[80];       //........................init command string 2
	char  cmdInit3[80];       //........................init command string 3
	char  cmdAnswer[80];      //........................answer commnad string
	char  cmdOffHook[80];     //......................off hook command string
	char  cmdDown[80];        //......................bbs down command string
	char  cmdAttention[80];   //.....attention command string (usually "at|")
	short externalErrorLevel; //..........errorlevel for external/fax connect
	char  msgCon31200 [80];   //.........................connect 31200 string
	char  msgCon33600 [80];   //.........................connect 33600 string
	char  msgCon36000 [80];   //.........................connect 36000 string
	char  msgConUser1[80];    //.......................user connect string #1
	long  userConBps1;        //..........................user connect bps #1
	char  msgConUser2[80];    //.......................user connect string #2
	long  userConBps2;        //..........................user connect bps #2
	char  msgConUser3[80];    //.......................user connect string #3
	long  userConBps3;        //..........................user connect bps #3
	char  msgConUser4[80];    //.......................user connect string #4
	long  userConBps4;        //..........................user connect bps #4
	char  msgConUser5[80];    //.......................user connect string #5
	long  userConBps5;        //..........................user connect bps #5
	char  msgConUser6[80];    //.......................user connect string #6
	long  userConBps6;        //..........................user connect bps #6
	uchar extra[920];
} MODEM;

/*
 
  FILESIDX.PB -	the filebase index. This file consists of all files in the
  filebase, with their area numbers. When building the index, ProBoard reads
  the FILES.BBS for each area and puts all the names that it can find in this
  file. Note that files are not actually checked for existence and files not
  in the list do not appear in the index. Some entries may have NUL ("") in
  the filename field. Those should be skipped. Area numbers start from 1
  (i.e. the record number in FILECFG + 1).
 
*/
typedef struct{
	char   filename[13];  //.................name of the file, with extension
	ushort area;          //...........file area number where file is located
} FILESIDX;

/*
 
  FGROUPS.PB / MGROUPS.PB - The file/message group definitions. Those files
  have the same structure, with each group allocated one record in the
  relevant file. Note that some groups may have NUL ("") strings in the name
  field and should be ignored.
 
*/
typedef struct{
	char   name[80];   //.................group name (NUL string if no group)
	ushort level;      //................................minimum access level
	ulong  flags;      //...........................flags required for access
	ulong  flagsNot;   //...............flags-non grata (must not be present)
	uchar  extra[10];
} GROUP_PB;

typedef GROUP_PB FILEGROUP;  //..........alias for the file group definitions
typedef GROUP_PB MESGGROUP;  //.......alias for the message group definitions

/*
 
  PVTFILES.PB -	private files to users. Each file has a separate recorord in
  the datafile. When a file is sent, the record is deleted and the file can
  have 0-byte length (or not exist at all).
 
*/
#define PVTFILE_KEEP (0x00000001L)  //.........keep file after it is received

typedef struct{
	DateType date;       //..............................date file was posted
	char     fname[80];  //..........................path to the file to send
	char     to[36];     //...................................receiver's name
	char     from[36];   //.....................................sender's name
	char     desc[80];   //..........................description for the file
	ulong    attr;       //........................attribute, see PVTFILE_???
	uchar    extra[61];
} PVTFILE;

/*
 
  FLAGDESC.PB - the flag descriptions (appear in the flag editor). Each flag
  has its own entry in the file. The name field is an ASCIIZ string. The
  flags go from A to 6, with A's description in record# 0 in the file and
  6's - in record# 31. This file has all records present, even if there's no
  description for the flag, in which case the name field has a NUL string.
 
*/
typedef struct{
	char name[40]; //................................description of this flag
} FLAGDESC;

/*
 
  TEMPLATE.PB - the user-validation templates. This file has an unknown # of
  records, one per template. The valMsgFile field contains the filename for
  the message to post to the user when validated. This field is currently
  only used by TheUser.
 
*/
#define TEMPL_HIDE_NO      0 //..........................hidden flag: visible
#define TEMPL_HIDE_YES	   1 //.......................hide user (set flag ON)
#define TEMPL_HIDE_IGNORE  2 //............................ignore the setting

typedef struct{
	char   name[61];         //.........................name of this template
	char   comment[81];      //...............comment to place in user record
	char   valMsgFile[9];    //...................validation message filename
	ushort newLevel;         //.............assign this as the new user level
	ushort subDays;          //..................days the new level will last
	ushort expLevel;         //.............assign this level when it expires
	ushort tbTime;           //............bank: deposit time to user account
	ushort tbKBytes;         //..............bank: deposit KB to user account
	ushort netCredit;        //...................netmail credit user can use
	short  logLevel;         //.............log level, see the LOG_??? consts
	uchar  hidden;           //............set HIDDEN (TEMPL_HIDE_??? consts)
	uchar  noTops;           //............set the NOTOPS flag in user record
	uchar  attention;        //.............set ATTENTION flag in user record
	uchar  noKill;           //..............set NOKILL flag in user's record
	uchar  ignoreDL;         //..............set IGNOREDL flag in user record
	BOOL   deleted;          //..................template marked for deletion
	char   extra[99];
	ulong  newFlagsOn;       //...............flags to turn on when validated
	ulong  newFlagsOff;      //..............flags to turn off when validated
	ulong  expFlagsOn;       //.................flags to turn on when expired
	ulong  expFlagsOff;      //................flags to turn off when expired
} TEMPLATE;

/*
 
  EVENTS.PRO - the event definition file. Each even has one record of the
  type defined below. To see if event is enabled for a day, use the EVT_???
  masks (EVENT.daysEnabled & EVT_???) will be 0 if not enalbed and != 0 if
  enabled.
 
*/
#define EVT_ERRORLEVEL 0     //.................event is exit with errorlevel
#define EVT_COMMAND    1     //..........................event is run command

#define EVT_MONDAY     0x01  //...................event is enabled for Monday
#define EVT_TUESDAY    0x02	 //..................event is enabled for Tuesday
#define EVT_WEDNESDAY  0x04	 //................event is enabled for Wednesday
#define EVT_THURSDAY   0x08	 //.................event is enabled for Thursday
#define EVT_FRIDAY     0x10	 //...................event is enabled for Friday
#define EVT_SATURDAY   0x20	 //.................event is enabled for Saturday
#define EVT_SUNDAY     0x40	 //...................event is enabled for Sunday
#define EVT_ANYDAY     0xFF  //.................event is enabled for all days

typedef struct{
	short    type;           //.......................event type, see EVT_???
	short    errorLevel;     //...........errorlevel for EVT_ERRORLEVEL event
	char     command[80];    //.............command string EVT_COMMAND events
	TimeType startTime;      //......................start time for the event
	short    duration;       //............duration of the event (in minutes)
	ushort   daysEnabled;    //............days event is enabled, see EVT_???
	uchar    nodeNum;        //.................node number this event is for
	BOOL     enabled;        //............true if event is enabled,false=not
} EVENT;

/*
 
  *.PBM - the menu files. Each menu has its own menu file. The file consists
  of a header of type MENU_HEADER and an undefined number of records of type
  MENU_ITEM following it. Note that Philippe did not provide the new format
  of the file (with the node setup, etc.) so this continues to be a hack (as
  it as before the structure was defined in the PB_STRUCT file after the
  first PB-Lib was released :-) Note that the 'nodeDisable' field is here
  restricted to 255 nodes. This is not the case in ProCFG. If you define 999
  nodes there, the menu editor will let you edit all of them (and you will
  overwrite part of the next menu item - BUG!). Since the docs say that the
  max. number of supported nodes is 255, I decided to restrict the field.
 
*/
#define MNU_HEAD_RIP  0x00000001L  //............this the prompt has RIP file

typedef struct{
	long  flags;               //........flags for the menu, see MNU_HEAD_???
	char  prompt[200];         //...............................prompt string
	uchar color;               //......................normal color attribute
	uchar highColor;           //...................highlight color attribute
	char  ripName[9];          //.............filename of the RIP prompt file
	char  extra[100];
} MENU_HEADER;

#define MNU_SHOWREMOTE  1      //.............................rip show remote
#define MNU_NOLOCAL     2      //..................rip suppress local display
#define MNU_NORESET     4      //.............................rip don't reset
#define MNU_PASSWORD    8      //................the menu item has a password

#define SEX_IGNORE      0      //........................don't care about sex
#define SEX_MALE        1      //...............only male users can access it
#define SEX_FEMALE      2      //................only female users can access

typedef struct{
	ulong     flags;           //...................see the MNU_xxx constants
	char      text[160];       //..............................menu line text
	char      data[160];       //...........................data for function
	uchar     color;           //..........................color for the line
	uchar     hotkey;          //............hotkey to activate this function
	ushort    function;        //..........................pb function number
	char      password[16];    //..................password for the menu item

	ushort    minLevel;        //............min. level for access (0=ignore)
	ushort    maxLevel;        //............max. level for access (0=ignore)
	ulong     aFlags;          //...................flags required for access
	ulong     flagsNot;        //..................flags non-grata for access
	uchar     minAge;          //..............min. age for access (0=ignore)
	uchar     maxAge;          //..............max. age for access (0=ignore)
	uchar     sex;             //.....................sex: see SEX_??? consts
	ushort    timeLeft;        //.........min. time left online to get access
	ushort    timeOnline;      //........min. time spent online before access
	TimeFrame timeFrame;       //...................time menu item is enabled
	uchar     terminals;       //....................0xff - ignore (obsolete)
	ulong     minSpeed;        //............min. speed for access (0=ignore)
	ulong     maxSpeed;        //............max. speed for access (0=ignore)
    uchar     nodeDisable[32]; //..............bit set for each node disabled
	uchar     extra[18];
	uchar     extra2[50];
} MENU_ITEM;

/*
 
  *.PBL - language files. Each file has a header PBL_HEADER type, which
  is followed by PBL_ITEM records. The number of those records varies by
  ProBoard version and can be obtained by the undocumented field in the
  header (numItems). All strings in the header are regular ASCIIZ C strings.
  This is not the case with the language items. Each item has PBL_ITEM
  header, which is followed by the item data: 'len' characters for the
  strings and numHotkeys characters for the hotkeys. Neither the strings nor
  the hotkeys are NUL-terminated. There can be a maximum of 8 strings and
  they will be separated by 0x0A only. The maximum length of each string is
  200 characters, including control codes. The hotkeys can be no more than 80
  (these are the limits of the language editor).
 
*/
typedef struct{
	char   name[41];         //....................title of the language file
	BOOL   avail;            //............TRUE if available, FALSE otherwise
	ushort level;            //...............security level needed to use it
	ulong  aFlags;           //........................flags needed to use it
	char   menuPath[61];     //..............directory for the language menus
	char   textPath[61];     //...................directory for the textfiles
	char   questPath[61];    //..............directory for the questionnaires
	char   copyright[81];    //.................copyright string for the file
	uchar  attrib;           //......................................not used
	uchar  extra[500];
	ushort numItems;         //...............undocumented: number of prompts
} PBL_HEADER;

#define PBL_PROMPT   1       //.................the language item is a prompt
#define PBL_NOCOLOR  2       //.............no color (what the heck is that?)

typedef struct{
	short  len;         //.........................length of string, see NOTE
	short  numHotkeys;  //..........................number of defined hotkeys
	uchar  color;       //....................color attribute for the strings
	uchar  highCol;     //..........................highlight color attribute
	uchar  promptCol;   //............................foreground prompt color
	uchar  flags;       //..................................PBL_??? constants
} PBL_ITEM;

/*
 
  NODE_IDX.PRO - nodelist index (built by PBUTIL NC). This is a complex file
  which uses a strange encoding scheme (why?). After each item is read, it
  needs to be XOR'ed with the values in the XOR array (shown below). Then,
  if 'type' is > 10, the record is a nodelist filename (12 characters plus
  NUL at the end). This file is in the CONFIG.nodelistdir directory. The
  first record is always a filename. Here's the XOR array

           234, 123, 76, 245, 87, 12, 43, 56, 23, 12, 98, 55, 34

  Use the NODEIDX_??? masks to determine the entry type.
 
*/
#define NODE_ZONE   1    //................................record 'type' Zone
#define NODE_REGION 2    //..............................record 'type' Region
#define NODE_HOST   3    //................................record 'type' Host
#define NODE_HUB    4    //.................................record 'type' Hub
#define NODE_PVT    5    //.............................record 'type' Private
#define NODE_DOWN   6    //................................record 'type' Down
#define NODE_HOLD   7    //................................record 'type' Hold
#define NODE_NODE   8    //................................record 'type' Node

typedef union{
    struct{
		char type;       //........see the NODE_??? constants, >10 = filename
		long offset;     //............offset in the nodelist list for record
		union{
			short zone;  //......................if type == zone, zone number
			short net;   //.......if type is not zone, this is the net number
		};
		short cost;      //............cost for node (compiler uses COST.PRO)
	}    node;           //.............access the structure within the union
	char name[13];       //......used for XORing and for filename when needed
} NODEIDX;

#endif /* INCLUDED_PBSTRUCT_H */
