/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#ifndef INCLUDED_BUCKET_H
#define INCLUDED_BUCKET_H
#include "typedef.h"
#include "comdef.h"
#include "bitvect.h"

#ifndef PB_SDK
	#include <stdio.h>
	#include <stdlib.h>
#else
	#include "pblibc.h"
#endif

/*
 * This class is for internal use only. It provides the mechanism for the
 * context-sensitive operator[] in the zBucket class. This class should not
 * be used directly by callers. They should refer to the zBucket instead.
*/
template <class Type>
class bucket_array_reference
{
private:
	class zBucket<Type> &m_bucket;
	long  m_index;

public:
	bucket_array_reference(zBucket<Type> &bucket, long index);
	bucket_array_reference& operator=(const Type &item);
	operator Type();
};

/*
 * Bucket class, tied to a file. This provides an array-like access to the
 * file 'fileName' which has records of type 'Type'. The size of the buffer
 * is controlled by 'nItemsToBuffer' and this specifies the number of items
 * which will be buffered in memory by the bucket. You can have a read-only
 * bucket. In that case, all attempts to update elements in it will cause
 * the error() function to be called (see bCanUpdate parameter to the CSTOR)
*/
template <class Type>
class zBucket
{
	friend class bucket_array_reference<Type>;

public:
	// CONSTRUCTORS AND DESTRUCTOR
	zBucket(const char *fileName, size_t nItemsToBuffer, Boolean bCanUpdate);
	virtual ~zBucket();
	// ACCESSORS
			operator void *() const;
	Boolean operator      !() const;
	long    nelems()          const;
	// MANIPULATORS
	bucket_array_reference<Type> operator[](long index);

protected:
	enum error_types
	{
		ERROR_FILE = 100,
		ERROR_MEMORY,
		ERROR_UNDERFLOW,
		ERROR_OVERFLOW,
		ERROR_INVREQ,
	};

	virtual void error(int code);

private:
	// PREVENT FROM ASSIGNING ZBUCKET OBJECTS
	zBucket(const zBucket&);
	zBucket& operator=(const zBucket&);

	// MANIPULATORS
	void    insert(long index, const Type &item);
	Type    at(long index);
	Boolean load(long index, Boolean force);
	void    flush();

	// INTERNAL DATA MEMBERS
	Type        m_dummy;   // dummy return value in case of error
	FILE       *m_fp;      // pointer to the bucket file in use
	Type       *m_buf;     // buffer with three pages of Type records
	Boolean     m_update;  // can we update the records in the file
	Boolean     m_dirty;   // must update file when flushing records
	long        m_first;   // index of the first item in the buffer
	long        m_last;    // index of the last item in the file
	size_t      m_items;   // number of items to hold in the buffer
	zBitVector *m_map;     // map of the "dirty" items in buffer
};

/*
 ****************************************************************************
 * templatized implementation of the bucket_array_reference class
 ****************************************************************************
*/
template<class Type>
bucket_array_reference<Type>::bucket_array_reference(zBucket<Type> &bucket,
													 long index)
	:m_bucket(bucket)
	,m_index(index)
{
}

template<class Type>
bucket_array_reference<Type>&
bucket_array_reference<Type>::operator=(const Type &item)
{
	m_bucket.insert(m_index, item);
	return *this;
}

template<class Type>
bucket_array_reference<Type>::operator Type ()
{
	return m_bucket.at(m_index);
}

/*
 ****************************************************************************
 * templatized implementation of the zBucket class
 ****************************************************************************
*/
template<class Type>
zBucket<Type>::zBucket(const char *fileName, size_t nelem, Boolean update)
	:m_update(update)
	,m_dirty(False)
	,m_first(0L)
	,m_items(nelem)
{
	m_fp = fopen(fileName, update ? "r+b" : "rb");

	if( !m_fp )
	{
		error(ERROR_FILE);
		goto FunctionExit;
	}
	else
	{
		// get the last index available
		fseek(m_fp, 0L, SEEK_END);
		m_last = ftell(m_fp) / sizeof(Type);
		rewind(m_fp);

		if( !(m_map = new zBitVector(nelem)) )
		{
			fclose(m_fp);
			error(ERROR_MEMORY);
			goto FunctionExit;
		}
		if( !(m_buf = new Type [nelem]) )
		{
			delete m_map;
			fclose(m_fp);
			error(ERROR_MEMORY);
			goto FunctionExit;
		}
		m_map->clearAll();
		load(0L, True);
	}
FunctionExit:
	return;
}

template<class Type>
zBucket<Type>::~zBucket()
{
	if( m_dirty ) flush();
	if( m_map   ) delete   m_map;
	if( m_buf   ) delete[] m_buf;
	if( m_fp    ) fclose(m_fp);
}

template<class Type>
long
zBucket<Type>::nelems() const
{
	return m_last;
}

template<class Type>
zBucket<Type>::operator void *() const
{
	if( !m_fp || !m_buf || !m_map ) return 0;
	return (void *)this;
}

template<class Type>
Boolean
zBucket<Type>::operator !()const
{
	return Boolean(0 != operator void *());
}

template<class Type>
bucket_array_reference<Type>
zBucket<Type>::operator[](long index)
{
	return bucket_array_reference<Type>(*this, index);
}

template<class Type>
Type
zBucket<Type>::at(long index)
{
	if( index < 0L )
	{ // ivalid index request
		error(ERROR_UNDERFLOW);
	}
	else if( index >= m_first && index < m_first + m_items )
	{ // the item is in the buffer already
		return m_buf[size_t(index - m_first)];
	}
	else
	{ // must re-load items in buffer
		if( load(index, False) ) return m_buf[size_t(index - m_first)];
	}
	// this is not good, return the dummy element
	error(ERROR_INVREQ);
	return m_dummy;
}

template<class Type>
void
zBucket<Type>::error(int code)
{
	exit(255 - code);
}

template<class Type>
void
zBucket<Type>::insert(long index, const Type &item)
{
	if( !m_update || index < 0L )
	{ // can't update or invalid index
		error(ERROR_INVREQ);
	}
	else
	{
		if( index < m_first || index >= m_first + m_items )
		{ // element to update is not in the buffer
			load(index, False);
		}
		// element to update is in the buffer (maybe just loaded)
		m_dirty = True;
		m_map->set(size_t(index - m_first));
		m_buf[size_t(index - m_first)] = item;
	}
}

template<class Type>
Boolean
zBucket<Type>::load(long index, Boolean force)
{
	Boolean retval = False;

	if( m_dirty ) flush(); // flush first if needed

	if( force || index < m_first || index >= m_first + m_items )
	{ // only load if requested index is not in the buffer already
		if( index < 0L )
		{
			error(ERROR_UNDERFLOW);
			goto FunctionExit;
		}
		if( index >= m_last )
		{ // do we need to extend the file?
			if( !m_update )
			{ // we can't do it
				error(ERROR_OVERFLOW);
				goto FunctionExit;
			}
			// extend the file
			fseek(m_fp, 0L, SEEK_END);
			while( m_last <= index )
			{
				fwrite(&m_dummy, sizeof(Type), 1, m_fp);
				m_last++;
			}
		}
		// adjust the bounds to load
		m_first = max(0L, index - m_items / 2);
		fseek(m_fp, m_first * sizeof(Type), SEEK_SET);
		fread(m_buf, sizeof(Type), min(m_items, size_t(m_last-m_first)), m_fp);
		retval = True;
	}

FunctionExit:
	return retval;
}

template<class Type>
void
zBucket<Type>::flush()
{
	if( m_dirty && m_update )
	{ // we flush only if needed and allowed
		for( int i = 0; i < m_items; ++i )
		{
			if( m_map->has(i) )
			{
				fseek(m_fp, (m_first + i) * sizeof(Type), SEEK_SET);
				fwrite(&m_buf[i], sizeof(Type), 1, m_fp);
			}
		}
	}
	m_dirty = False;
	m_map->clearAll();
}

#endif /* INCLUDED_BUCKET_H */
