/*
 * PB-Lib C/C++ Library Version 0.021
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, Inc.
 *
 * See the file "copying.pbl" for licensing information.
*/

#ifndef __PROUTIL_H
#include <proutil.h>
#endif

#ifndef __MEMDBG_H
#include <memdbg.h>
#endif

#ifndef PB_SDK
#include <assert.h>
#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <stdlib.h>
#else
#define textattr SetFullColor
#define putch    putchar
#endif

/*
 * 8 lines, max of 200 characters each + 2 (NUL and starting normal color)
 * we play it safe and reserve a lot more (4096 bytes to be exact)
*/
#define PBLS_LEN 0x1000

static char __xlatBuf[PBLS_LEN];

/*
 * reads a language string from language. If none found, will try to use
 * the defaut language file. Language is the name of the language (up to
 * eight characters), if NULL, the default language will be used. rec
 * will hold information about the language string, s will contain the
 * string itself and id is the number of string to load. You must pass
 * this function a valid id (1-394 for v2.10) or else the results are
 * unpredictable (and wrong).
 *
 * Warning: prompt numbering starts at 1!
*/
	int
xlatLoad( int id, xlat_t *rec, const char *language )
{
	char  path[MAXPATH];
	int   retval = -1;
	FILE *fp;

	assert( NULL != rec );
	assert( id > 0 && id <= PBLANG_DEFS );

	if( language ){
		strcat( strcpy(path, language), fnLANGEXT );
		if( NULL == (fp = pbfopen(path, "rb")) ) goto _defLoad;
	}
	else{
_defLoad:
		fp = pbfopen( fnDEFLANG, "rb" );
	}

	if( NULL == fp ) return -1;
	fseek( fp, (long)sizeof(LANGHDR), SEEK_SET );
	while( --id ){
		LANGREC dummy;
		fread( &dummy, sizeof(LANGREC), 1, fp );
		if( feof(fp) ) goto _closeExit;
		fseek( fp, (long)(dummy.len + dummy.numHotKeys), SEEK_CUR );
	}

	if( 1 != fread(&rec->def, sizeof(LANGREC), 1, fp) )
		goto _closeExit;
	if( rec->def.len ){
		if( NULL == (rec->val = (char *)mem_malloc(rec->def.len + 1)) )
			goto _closeExit;
		fread( rec->val, sizeof(char), rec->def.len, fp );
		rec->val[rec->def.len] = EOS;
	}
	else rec->val = NULL;
	if( rec->def.numHotKeys ){
		if( NULL == (rec->keys = (char *)mem_malloc(rec->def.numHotKeys + 1)) )
			goto _closeExit;
		fread( rec->keys, sizeof(char), rec->def.numHotKeys, fp );
		rec->keys[rec->def.numHotKeys] = EOS;
	}
	else rec->keys = NULL;

	retval = 0;
_closeExit:
	fclose(fp);
	return retval;
}


/*
 * deallocates the internal xlat_t structure
*/
	void
xlatFree( xlat_t *rec )
{
	assert( NULL != rec );
	if( rec->val ) mem_free( rec->val );
	if( rec->keys ) mem_free( rec->keys );
}

/*
 * compiles the ProBoard language strings to an internal format. Colors
 * will be defined as '\num' where 'num' is an unsigned byte encoding
 * that uses the same format as CONIO's textattr() function. If there's
 * a backslash '\' in the original string (generated by two '\\'), then
 * we encode it as 0x80 (blinking black on black). This function does not
 * interpret the v2.10+ @a|pFILENAME@ strings. Also, all 0x0A's in the
 * source string will be replaced by '\r' (carriage returns) to force
 * new lines.
 *
 * The expected input is a pointer to a LANGREC structure which has to
 * be preloaded by the caller with info from the appropriate language. The
 * s parameter is the actual prompt string (not NUL-terminated).
 *
 * The resulting buffer is static and is overwritten by each call. It is
 * responsibility of the caller to save it if required for further use.
*/
	xlts_t
xlatCompile( const xlat_t *rec )
{
	char *p       = __xlatBuf;
	const char *s = rec->val;
	bool  highCol = TRUE;
	int   i;

	assert( NULL != rec );
	*p++ = '\\'; *p++ = rec->def.normCol;
	for( i = rec->def.len; 0 < i; --i, ++s ){

		/* see if somebody passed incorrect strings here */
		assert( PBLS_LEN > (p - __xlatBuf) );

		switch( *s ){
			case '\n':
				*p++ = *s; *p++ = '\r';
			break;

			case '^':
				*p++ = '\\';
				*p++ = highCol ? rec->def.highCol : rec->def.normCol;
				highCol = !highCol;
			break;

			case '\\':
				if( '\\' == s[1] ){
					s++; i--;                  /* go pass the second '\'  */
					if( p > __xlatBuf ) p--;   /* do the backspace xlat   */
				}
				else if( '^' == s[1] ){
					s++; i--;            /* go past the special character */
					*p++ = '^';          /* generate a caret character    */
				}
				else{
					*p++ = '\\';
					/* translate the normal hex using next two characters */
					if( 'H' != s[1] && 'L' != s[1] ){
						int color;
						sscanf( (char *)&s[1], "%2X", &color );
						i -= 2; s += 2;
						*p++ = (byte)(color & 0x00FF);
					}
					/* these are the short-cuts */
					else{
						byte color;
						if( 'B' == s[2] ) color = 0x01;
						else if( 'G' == s[2] ) color = 0x02;
						else if( 'C' == s[2] ) color = 0x03;
						else if( 'R' == s[2] ) color = 0x04;
						else if( 'P' == s[2] ) color = 0x05;
						else if( 'Y' == s[2] ) color = 0x06;
						else if( 'W' == s[2] ) color = 0x07;

						if( 'H' == s[1] ) *p++ = color + 8;
						else *p++ = color;
						i -= 2; s += 2;
					}
				}
			break;

			default: *p++ = *s;
		}
	}

	*p = EOS;
	return __xlatBuf;
}


/*
 * displays a xlat-compiled string in color (if ansiOn is TRUE)
*/
	void
xlatDisp( const xlts_t str, bool ansiOn )
{
	char *s;

	assert( NULL != str );
	for( s = (char *)str; EOS != *s; ++s ){
		if( '\\' == *s ){
			++s;
			if( ansiOn ) textattr( (int)*s );
		}
		else putch( *s );
	}
}

#undef PBLS_LEN
#ifdef PB_SDK
#undef textattr
#undef putch
#endif
