/*
 * PB-Lib C/C++ Library Version 0.021
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, Inc.
 *
 * See the file "copying.pbl" for licensing information.
*/

#ifndef __WHATIS_H
#include <whatis.h>
#endif

#ifndef PB_SDK
#include <stdio.h>
#include <string.h>
#else
#include <pblib.h>
#endif

/* local helper function */
static int __whichARC( const char *header );

/* table with IDs, offsets, lengths of signatures and strings */
static struct{
	int  type;    /* enumeration type of the archive */
	long seekofs; /* seek offset into the file */
	int  ofs;     /* offset into the header */
	int  slen;    /* length of signature to check */
	char sig[10]; /* more than enough */
} _ttbl[__totalID] =
{
	{ ftARJ, 0L, 0, 2, "\x60\xEA" },
	{ ftPKZIP, 0L, 0, 4, "\x50\x4B\x03\x04" },
	{ ftLHA, 0L, 2, 3, "-lh" },
	{ ftARC, 0L, 0, 1, "\x1A" },
	{ ftRAR, 0L, 0, 4, "Rar!" },
	{ ftZOO, 0L, 0, 4, "ZOO " },
	{ ftUC2, 0L, 0, 4, "UC2\x1A" },
	{ ftHAP, 0L, 0, 4, "\x91\x33\x48\x46" },
	{ ftSQZ, 0L, 0, 5, "HLSQZ" },
	{ ftAIN, 0L, 0, 2, "\x21\x12" },
	{ ftPKLITE, 0L, 30, 6, "PKLITE" },
	{ ftAINEXE, 0L, 32, 4, "AIN2" },
	{ ftLZEXE, 0L, 28, 4, "LZ91" },
	{ ftEXE, 0L, 0, 2, "MZ" },
	{ ftJPEG, 0L, 6, 4, "JFIF" },
	{ ftGIF87, 0L, 0, 5, "GIF87" },
	{ ftGIF89, 0L, 0, 5, "GIF89" },
	{ ftError, 0L, 0, 0, "" }, /* we don't parse past this point, __numID */
	{ ftARJSX, 0L, 28, 4, "RJSX" },
	{ ftLHASX, 0L, 36, 3, "LHA" },
	{ ftRARSX, 0L, 90, 7, "RAR SFX" },
	{ ftSQZSX, 0x526BL, 0, 3, "SQZ" },
	{ ftPKZIPSX, 0x3D9AL, 0, 4, "\x50\x4B\x03\x04" },
	{ ftPKZIPSA, 0x3BF5L, 0, 4, "\x50\x4B\x03\x04" },
	{ ftSEZSX, 0x8C6L, 0, 4, "Sez " },
	{ ftARCSX, 0x1E24L, 0, 2, "\x1A\x08" },
	{ ftPAKSX, 0x1AD3L, 0, 2, "\x1A" },  /* need additional checks */
	{ ftDWC, 0L, 0, 3, "DWC" },
};

/* macro check signature in buffer against _ttbl element */
#define CHECKSIG(buf,id) \
	( 0 == memcmp(&buf[_ttbl[id].ofs], _ttbl[id].sig, _ttbl[id].slen) )
/* macro to seek to a position within the file and read bytes there */
#define READOFS(fp,buf,id) {                       \
	fseek(fp, _ttbl[id].seekofs, SEEK_SET );       \
	fread(buf, sizeof(char), _ttbl[id].slen, fp ); }

/* identify different ARC-type archives */
	int
__whichARC( const char *header )
{
	int retval = ftARC; /* assume this, we know it's true when calling */

	if( 0x0A == header[1] || 0x0B == header[1] ) retval = ftPAK;
	else if( 0x14 <= header[1] ) retval = ftARC7;
	return retval;
}

/*
 * simple table-driven matching with special cases for some self-extractors
*/
	int
getFileType( const char *path )
{
	char buf[128], tmpBuf[10];
	int  i, retval = -1;

	FILE *fp = fopen( path, "rb" );
	if( NULL == fp ) return -1;
	fread( buf, sizeof(char), 128, fp );

	/* ok, here we have a buffer with the first 256 bytes of the file */
	for( retval = -1, i = 0; i < __NumID && -1 == retval; ++i ){
		if( CHECKSIG(buf, i) ) retval = _ttbl[i].type;
	}

	/* see for PAK and ARC7 files (same as ARC + second byte) */
	if( ftARC == retval ){ retval = __whichARC(buf); goto bye; }

	/* check for DWC archives (signature at the end of the file) */
	fseek( fp, -3L, SEEK_END );
	fread( tmpBuf, sizeof(char), 3, fp );
	if( CHECKSIG(tmpBuf, ftDWC) ){
		retval = (ftEXE == retval) ? ftDWCSX : ftDWC; goto bye;
	}

	/* see if it's an ARJ, LHA, ARC, ZOO or PAK self-extractor */
	if( ftEXE == retval ){
		if( CHECKSIG(buf, ftARJSX) ){ retval = ftARJSX; goto bye; }
		if( CHECKSIG(buf, ftLHASX) ){ retval = ftLHASX; goto bye; }
		if( CHECKSIG(buf, ftRARSX) ){ retval = ftRARSX; goto bye; }
		READOFS(fp, tmpBuf, ftSEZSX);
		if( CHECKSIG(tmpBuf, ftSEZSX) ){ retval = ftSEZSX; goto bye; }
		READOFS(fp, tmpBuf, ftARCSX);
		if( CHECKSIG(tmpBuf, ftARCSX) ){ retval = ftARCSX; goto bye; }
		READOFS(fp, tmpBuf, ftPAKSX);
		if( 0x1A == *tmpBuf ){
			switch( __whichARC(tmpBuf) ){
				case ftARC : retval = ftARCSX;  break;
				case ftARC7: retval = ftARC7SX; break;
				case ftPAK : retval = ftPAKSX;  break;
			}
		}
		goto bye; /* ok, then it's just an EXE file */
	}

	/* see if it's PKLITE, it may be SQZ or PKZIP self-extracting archive */
	if( ftPKLITE == retval ){
		READOFS(fp, tmpBuf, ftPKZIPSX);
		if( CHECKSIG(tmpBuf, ftPKZIPSX) ){ retval = ftPKZIPSX; goto bye; }
		READOFS(fp, tmpBuf, ftPKZIPSA);
		if( CHECKSIG(tmpBuf, ftPKZIPSA) ){ retval = ftPKZIPSX; goto bye; }
		READOFS(fp, tmpBuf, ftSQZSX);
		if( CHECKSIG(tmpBuf, ftSQZSX) ){ retval = ftSQZSX; goto bye; }
	}

bye:
	fclose(fp);
	return retval;
}

#undef CHECKSIG
#undef READOFS
