/*
 * PB-Lib C/C++ Library Version 0.021
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, Inc.
 *
 * See the file "copying.pbl" for licensing information.
*/

/* Memory management package.

   Copyright (C) 1989-1995 by Walter Bright.

   Modified and reformatted by Branislav L. Slantchev for inclusion in PBLib
   Added some comments and removed non-essential stuff. The original file is
   available in the FidoNet C Echo Snippets collection, courtesy of Bob Stout
*/

#ifndef __MEMDBG_H
#include <memdbg.h>
#endif

#ifndef PB_SDK
	#include <assert.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <io.h>
#else
	#define  fflush(x)
#endif

int mem_inited = 0;                       /* != 0 if initialized           */

static int mem_behavior = MEM_ABORTMSG;  /* default mem behaviour          */
static int (*fp)( void ) = NULL;         /* out-of-memory handler          */
static int mem_count;             /* # of allocs that haven't been free'd  */
static int mem_scount;            /* # of sallocs that haven't been free'd */

static int near mem_exception( void );

/*
 * set the out of memory handler
*/
	void
mem_setexception( enum MEM_E flag, int(*handler_fp)(void) )
{
	mem_behavior = flag;
	fp = (mem_behavior == MEM_CALLFP) ? handler_fp : 0;
#ifdef MEM_DEBUG
    assert(0 <= flag && flag <= MEM_RETRY);
#endif
}


/*
 * called when out of memory condition encountered
*/
	static int near
mem_exception( void )
{
	int behavior;

	behavior = mem_behavior;
	while( 1 ){
		switch (behavior){
			case MEM_ABORTMSG:
				{
					static char msg[] = "Fatal error: out of memory\r\n";
					write(1,msg,sizeof(msg) - 1);
				}
			/* fall-through */

			case MEM_ABORT:
				exit( EXIT_FAILURE );
			/* not reached */

			case MEM_CALLFP:
				assert(fp);
				behavior = (*fp)();
			break;

			case MEM_RETNULL: return 0;
			case MEM_RETRY: return 1;
			default: assert(0); return 0;
		}
	}
}


/*
 * mem_strdup() definition
*/
#ifdef MEM_DEBUG

#undef mem_strdup
static void near mem_printdl(struct mem_debug *dl);
static void near mem_fillin(char *fil,int lin);

	char*
mem_strdup( const char *s )
{
	return mem_strdup_debug(s,__FILE__,__LINE__);
}

	char*
mem_strdup_debug( const char *s, char *file, int line )
{
	char *p;

	p = s
		? (char *) mem_malloc_debug((unsigned) strlen(s) + 1,file,line)
		: NULL;
	return p ? strcpy(p,s) : p;
}

#else /* MEM_DEBUG is not defined */

	char*
mem_strdup(const char *s)
{
	char *p;

	p = s ? (char *) mem_malloc((unsigned) strlen(s) + 1) : NULL;
	return p ? strcpy(p,s) : p;
}

#endif /* #ifdef MEM_DEBUG */

#ifdef MEM_DEBUG

#define BADVAL		0xFF          /* pointer loaded with this should crash */
#define MALLOCVAL	0xEE          /* set all malloc'd data to this value   */
#define BEFOREVAL	0x12345678L   /* value to detect underrun              */
#define AFTERVAL	0x87654321L   /* value to detect overrun               */
#undef	mem_malloc
#undef	mem_calloc
#undef	mem_realloc
#undef	mem_free

static long mem_maxalloc;         /* max # of bytes allocated              */
static long mem_numalloc;         /* current # of bytes allocated          */

/*
 * list of all memory pointers with info about their origin
*/
struct mh{
	struct mem_debug *Mnext;  /* next in list                          */
	struct mem_debug *Mprev;  /* previous value in list                */
	char *Mfile;              /* filename of where allocated           */
	int Mline;                /* line number of where allocated        */
	unsigned Mnbytes;         /* size of the allocation                */
	long Mbeforeval;          /* detect underrun of data               */
};

static struct mem_debug{
	struct mh m;
	char data[1];                 /* the data actually allocated           */
} mem_alloclist =
{
	{
		(struct mem_debug *) NULL,
		(struct mem_debug *) NULL,
		"noname",
		11111,
		0,
		BEFOREVAL
	},
	{ AFTERVAL }
};

/*
 * Convert from void* to mem_debug and from mem_debug to mem_ptr
*/
#define mem_ptrtodl(p)  ((struct mem_debug *) ((char *)p - sizeof(struct mh)))
#define mem_dltoptr(dl) ((void *) &((dl)->data[0]))
#define next            m.Mnext
#define prev            m.Mprev
#define file            m.Mfile
#define line            m.Mline
#define nbytes          m.Mnbytes
#define beforeval       m.Mbeforeval


/*
 * set the new value of file and line
*/
	void
mem_setnewfileline( void *ptr, char *fil, int lin )
{
	struct mem_debug *dl;

    dl = mem_ptrtodl(ptr);
    dl->file = fil;
	dl->line = lin;
}

/*
 * print out structure mem_debug
*/
	static void near
mem_printdl( struct mem_debug *dl )
{
	printf( "alloc'd from file '%s' line %d nbytes %d ptr x%lx\n",
		dl->file, dl->line, dl->nbytes, (long)mem_dltoptr(dl) );
}


/*
 * print out file and line number
*/
	static void near
mem_fillin( char *fil, int lin )
{
	printf( "File '%s' line %d\n", fil, lin );
	fflush( stdout );
}


/*
 * if MEM_DEBUG is not on, these routines are called
*/
	void*
mem_calloc( unsigned u )
{
	return mem_calloc_debug(u,__FILE__,__LINE__);
}

	void*
mem_malloc( unsigned u )
{
	return mem_malloc_debug(u,__FILE__,__LINE__);
}

	void*
mem_realloc( void *p, unsigned u )
{
	return mem_realloc_debug(p,u,__FILE__,__LINE__);
}

	void
mem_free( void *p )
{
	mem_free_debug(p,__FILE__,__LINE__);
}

	void
mem_freefp( void *p )
{
	mem_free(p);
}

/*
 * Debug versions called when MEM_DEBUG is defined.
*/
	void*
mem_malloc_debug( unsigned n, char *fil, int lin )
{
	void *p;

	p = mem_calloc_debug(n,fil,lin);
	if( p ) memset(p,MALLOCVAL,n);
	return p;
}

	void*
mem_calloc_debug( unsigned n, char *fil, int lin )
{
	struct mem_debug *dl;

	do{
		dl = (struct mem_debug *)
			calloc(sizeof(*dl) + n + sizeof(AFTERVAL) - 1,1);
	}while (dl == NULL && mem_exception());
	if( dl == NULL ) return NULL;
	dl->file = fil;
	dl->line = lin;
	dl->nbytes = n;
	dl->beforeval = BEFOREVAL;
	*(long *) &(dl->data[n]) = AFTERVAL;

	/* Add dl to start of allocation list */
	dl->next = mem_alloclist.next;
	dl->prev = &mem_alloclist;
	mem_alloclist.next = dl;
	if( dl->next != NULL ) dl->next->prev = dl;

	mem_count++;
	mem_numalloc += n;
	if( mem_numalloc > mem_maxalloc ) mem_maxalloc = mem_numalloc;
    return mem_dltoptr(dl);
}

	void
mem_free_debug( void *ptr, char *fil, int lin )
{
	struct mem_debug *dl;

	if( ptr == NULL ) return;
	if( mem_count <= 0 ){
		printf( "More frees than allocs at " );
		goto err;
	}
	dl = mem_ptrtodl(ptr);
	if( dl->beforeval != BEFOREVAL ){
		printf( "Pointer x%lx underrun\n", (long)ptr );
		goto err2;
	}
	if( *(long *) &dl->data[dl->nbytes] != AFTERVAL ){
		printf( "Pointer x%lx overrun\n",(long)ptr );
		goto err2;
	}
	mem_numalloc -= dl->nbytes;
	if( mem_numalloc < 0 ){
		printf( "error: mem_numalloc = %ld, dl->nbytes = %d\n",
			mem_numalloc, dl->nbytes );
		goto err2;
	}

	/* Remove dl from linked list */
	if( dl->prev ) dl->prev->next = dl->next;
	if( dl->next ) dl->next->prev = dl->prev;

	/* Stomp on the freed storage to help detect references */
	/* after the storage was freed.                         */
	memset((void *) dl,BADVAL,sizeof(*dl) + dl->nbytes);
	mem_count--;

	free((void *) dl);
	return;

err2:
	mem_printdl(dl);
err:
	printf( "free'd from " );
	mem_fillin(fil,lin);
	assert(0);
}

	void*
mem_realloc_debug( void *oldp, unsigned n, char *fil, int lin )
{
	void *p;
	struct mem_debug *dl;

	if( n == 0 ){
		mem_free_debug(oldp,fil,lin);
		p = NULL;
	}
	else if( oldp == NULL ) p = mem_malloc_debug(n,fil,lin);
	else{
		p = mem_malloc_debug(n,fil,lin);
		if( p != NULL ){
			dl = mem_ptrtodl(oldp);
			if( dl->nbytes < n ) n = dl->nbytes;
			memcpy(p,oldp,n);
			mem_free_debug(oldp,fil,lin);
		}
    }
    return p;
}

	void
mem_check( void )
{
	register struct mem_debug *dl;

	for( dl = mem_alloclist.next; dl != NULL; dl = dl->next )
		mem_checkptr(mem_dltoptr(dl));
}

	void
mem_checkptr(void *p)
{
	register struct mem_debug *dl;

	for( dl = mem_alloclist.next; dl != NULL; dl = dl->next ){
		if( p >= (void *) &(dl->data[0]) &&
			p < (void *)((char *)dl + sizeof(struct mem_debug)-1 + dl->nbytes))
		goto L1;
	}
	assert(0);

L1:
	dl = mem_ptrtodl(p);
	if( dl->beforeval != BEFOREVAL ){
		printf( "Pointer x%lx underrun\n", (long)p );
	    goto err2;
    }
	if( *(long *) &dl->data[dl->nbytes] != AFTERVAL ){
		printf( "Pointer x%lx overrun\n", (long)p );
		goto err2;
	}
	return;

err2:
	mem_printdl(dl);
	assert(0);
}

#else /* MEM_DEBUG is not defined */

	void*
mem_malloc( unsigned numbytes )
{
	void *p;

	if( numbytes == 0 ) return NULL;
	while(1){
		p = malloc(numbytes);
		if( p == NULL ){
			if( mem_exception() ) continue;
		}
		else mem_count++;
		break;
	}
	return p;
}

	void*
mem_calloc( unsigned numbytes )
{
	void *p;

	if( numbytes == 0 ) return NULL;
	while(1){
		p = calloc(numbytes,1);
		if( p == NULL ){
			if( mem_exception() ) continue;
		}
		else mem_count++;
		break;
	}
	return p;
}

	void*
mem_realloc( void *oldmem_ptr, unsigned newnumbytes )
{
	void *p;

	if( oldmem_ptr == NULL ) p = mem_malloc(newnumbytes);
	else if( newnumbytes == 0 ){
		mem_free(oldmem_ptr);
		p = NULL;
	}
	else{
		do{
			p = realloc(oldmem_ptr,newnumbytes);
		}while( p == NULL && mem_exception() );
    }
    return p;
}

	void
mem_free( void *ptr )
{
	if( ptr != NULL ){
		assert(mem_count != 0);
		mem_count--;
		free(ptr);
    }
}

/* scalloc() and scfree() for Zortech C compilers */
#if __ZTC__

#define MINBLKSIZE (sizeof(size_t) + sizeof(void*))  /* min size of block  */
#define ALIGNSIZE  (sizeof(size_t))                  /* boundary of allocs */

	void*
mem_scalloc( size_t numbytes )
{
	size_t *p;

	if( numbytes == 0 ) return NULL;
	if( numbytes < MINBLKSIZE ) numbytes = MINBLKSIZE;
	else numbytes = (numbytes + (ALIGNSIZE - 1)) & ~(ALIGNSIZE - 1);
	p = (size_t *) mem_calloc(numbytes - sizeof(size_t));
	if( p ){
		p--;
		*p = 0;
		mem_count--;
		mem_scount++;
    }
    return p;
}

	void
mem_sfree( void *ptr, size_t numbytes )
{
	if( ptr != NULL ){
		assert(mem_scount > 0);
		mem_scount--;
		if( numbytes < MINBLKSIZE ) numbytes = MINBLKSIZE;
		else numbytes = (numbytes + (ALIGNSIZE - 1)) & ~(ALIGNSIZE - 1);
		*((size_t *)ptr)++ = numbytes;	/* store size of free block	*/
		free(ptr);
    }
}

#endif /* Specific to Zortech C compilers */
#endif /* #ifdef MEM_DEBUG sections */


/*
 * General initialization and closedown routines
*/
	void
mem_init( void )
{
	if( mem_inited == 0 ){
		mem_count = 0;
#ifdef MEM_DEBUG
		mem_numalloc = 0;
		mem_maxalloc = 0;
		mem_alloclist.next = NULL;
#endif
#if __ZTC__
		/* Necessary if mem_sfree calls free before any calls to malloc */
		free(malloc(1));                /* initialize storage allocator */
#endif
		mem_inited++;
	}
}

	void
mem_term( void )
{

	if( mem_inited ){
#ifdef MEM_DEBUG
		register struct mem_debug *dl;

		for( dl = mem_alloclist.next; dl; dl = dl->next){
			printf( "Unfreed pointer: " );
			mem_printdl(dl);
		}
#else
		if( mem_count ) printf( "%d unfreed items\n", mem_count );
		if( mem_scount ) printf( "%d unfreed s items\n", mem_scount );
#endif /* MEM_DEBUG */
		assert(mem_count == 0 && mem_scount == 0);
		mem_inited = 0;
	}
}


/*
 * Undefine macros used in this source
*/
#undef next
#undef prev
#undef file
#undef line
#undef nbytes
#undef beforeval

