/*
 * PB-Lib C/C++ Library Version 0.021
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, Inc.
 *
 * See the file "copying.pbl" for licensing information.
*/

#ifndef __GREP_H
#include <grep.h>
#endif

#ifndef PB_SDK
#include <string.h>
#endif

#define GREP_ANY      1   /* matches any character                         */
#define GREP_STAR     2   /* matches zero or more occurences of pattern    */
#define GREP_PLUS     3   /* matches one or more occurences of pattern     */
#define GREP_SET      4   /* matches character in specified set            */
#define GREP_NSET     5   /* matches character not in specified set        */
#define GREP_EOS      6   /* marks the end of the current set              */
#define GREP_EOL      7   /* matches at end of line                        */
#define GREP_BOL      8   /* matches at beginning of line                  */
#define GREP_RANGE    9   /* matches characters in specified range         */
#define GREP_EOP     10   /* end of pattern marker                         */

#define GREP_WILDCARD 1   /* pmatch() should operate in wildcard mode      */
#define GREP_GREP     2   /* pmatch() should operate in grep mode          */

/* sets grepError and returns error -1 or NULL */
#define reterr(c,t)                                 \
	do{                                             \
		grepError = (c);                            \
		return (t);                                 \
	}while(0)

#define error(c)     reterr(c,-1)   /* called by grepCompile() */
#define rerror(c)    reterr(c,NULL) /* called by rbuild()      */

/* wrappers for sstore() called by grepCompile() and rbuild()  */
#define store(c)  if( -1 == sstore(c) )  error( GREP_OVRFLOW )
#define rstore(c) if( -1 == sstore(c) ) rerror( GREP_OVRFLOW )

static char _patBuf[MAXPAT];        /* compiled pattern                    */
static int  _patNdx   = 0;          /* index to put next compiled info     */
static int  _mmode    = GREP_GREP;  /* pmatch() mode of operation          */
static char *_sLine   = NULL;       /* points to the beginning of line     */
	   int  grepError = GREP_OK;    /* holds the error number              */

/*
 * local function prototypes
*/
static int   sstore( int c );
static char *rbuild( const char *s );
static const char *pmatch( const char *line, const char *pat );


/*
 * compile pattern into static buffer _patBuf
*/
	int
grep_compile( const char *pat )
{
	const char *p;
		   int  pStart = 0, cp;

	/* init global index to 0 and pointer p to start of pattern */
	for( _patNdx = 0, p = pat; EOS != *p; ++p ){
		/* star and plus require characters before them */
		if( '*' == *p || '+' == *p ){
			if( 0 == _patNdx || GREP_EOL == (cp = _patBuf[_patNdx-1]) ||
				GREP_BOL == cp || GREP_EOP == cp )
					error( GREP_BADPAT );
			/* store the special character in front of last pattern */
			store( GREP_EOP ); store( GREP_EOP );
			memmove(&_patBuf[pStart+1], &_patBuf[pStart], _patNdx - pStart);
			_patBuf[pStart] = ('*' == *p ) ? GREP_STAR : GREP_PLUS;
		}
		/* all other characters */
		else{
			/* save the start so we can adjust the star and the plus */
			pStart = _patNdx;
			switch( *p ){
				case '.' : store( GREP_ANY ); break;
				case '$' : store( GREP_EOL ); break;
				case '^' : store( GREP_BOL ); break;
				case '[' :
					/* grepError is set by rbuild() itself */
					if( NULL == (p = rbuild( ++p )) ) return -1;
				break;
				case '\\': if( EOS == *++p ) error( GREP_BADESC );
				default  :
					store( *p );
			}
		}
	}
	store( GREP_EOP );
	store( EOS );
	return 0;
}


/*
 * filename parsing compile
*/
	int
glob_compile( const char *pat )
{
	const char *p;

	/* init global index to 0 and pointer p to start of pattern */
	for( _patNdx = 0, p = pat; EOS != *p; ++p ){
		switch( *p ){
			case '*' :
				if( GREP_STAR == _patBuf[_patNdx-1] ) error( GREP_BADSTAR );
				store( GREP_STAR );
			break;
			case '?' : store( GREP_ANY ); break;
			case '[' :
				/* grepError is set by rbuild() itself */
				if( NULL == (p = rbuild( ++p )) ) return -1;
			break;
			case '\\': if( EOS == *++p ) error( GREP_BADESC );
			default  :
				store( *p );
		}
	}
	store( GREP_EOP );
	store( EOS );
	return 0;
}


/*
 * compile character or code in buffer
*/
	static int
sstore( int c )
{
	if( _patNdx >= MAXPAT ) return -1;

	_patBuf[_patNdx++] = c;
	return 0;
}

/*
 * builds a range of characters, returns pointer to end of NULL on error
 * '-', if followed by ']' or if first in set, loses its special meaning
*/
	static char*
rbuild( const char *pPat )
{
	const char *p = pPat;
	/* see if we want an exclusion or inclusion set */
	if( '^' == *p ){
		rstore( GREP_NSET );
		p++; pPat++;
	}
	else rstore( GREP_SET );

	for( ; EOS != *p && ']' != *p; ++p ){

		/* get escaped character literal */
		if( '\\' == *p ){
			if( EOS == *++p ) rerror( GREP_BADESC );
			else rstore( *p );
		}
		else{
			/* check for start of range */
			if( '-' == *p && ']' != p[1] && EOS != p[1] && p - pPat >= 1 ){
				/* save old character: it's the start of range */
				int rangeStart = _patBuf[--_patNdx];
				rstore( GREP_RANGE );
				rstore( rangeStart );
				/* store the end of range, check for escaped literal */
				if( '\\' == *++p && EOS == *++p ) rerror( GREP_BADESC );
				rstore( *p );
			}
			/* not a range, store the character */
			else rstore( *p );
		}
	}

	/* see if we ended without the ']' marker or with an empty set */
	if( EOS == *p || p == pPat ) rerror( GREP_BADSET );
	else rstore( GREP_EOS );

	/* return pointer to ']' */
	return (char *)p;
}

/*
 * the actual matching routine
*/
	static const char*
pmatch( const char *line, const char *pat )
{
	for( ; EOS != *line && GREP_EOP != *pat; ++line, ++pat ){
		char p = (char)*pat;
		const char *eln, *stln;
		int found = 0;

		switch( p ){        /* we got to have at least one character here */
			case GREP_ANY:
				if( EOS == *line ) return NULL;
			break;

			case GREP_PLUS:     /* GREP_GREP mode, we got to have a match */
				if( NULL == (line = pmatch(line, ++pat)) )
					return NULL;
						   /* PLUS falls through to STAR in case of match */
			case GREP_STAR:
							  /* stars are different in GREP and WILDCARD */
				if( GREP_GREP == _mmode ){
					stln = line; if( GREP_PLUS != p ) ++pat;
					   /* try finding longest match from current position */
					while( *line && NULL != (eln = pmatch(line,pat)) )
						line = eln;
						   /* skip to the end of the STAR or PLUS pattern */
					while( GREP_EOP != *pat++ ) ;
										 /* try matching rest of the line */
					while( line >= stln ){
						if( NULL != (eln = pmatch(line,pat)) )
							return eln;
						--line;
					}
					return NULL;           /* nothing, nada, zilch, sorry */
				}
				else{                                    /* WILDCARD mode */
									   /* nothing else in the pattern, Ok */
					if( GREP_EOP == *++pat ) return line;
						 /* try finding longest match, from end of string */
					for( eln = &line[strlen(line)-1]; eln >= line; --eln ){
						if( NULL != pmatch(eln, pat) ) return eln;
					}
					/* nope, no match */
					return NULL;
				}
			/* break; */

			case GREP_SET:
			case GREP_NSET:
				while( GREP_EOS != *++pat ){
					/* if it's a range, special treatment here */
					if( GREP_RANGE == *pat ){
						if( *line >= *++pat && *line <= *++pat ){
							if( GREP_SET == p ){
								found = 1;
								break;
							}
							else return NULL;
						}
					}
					else if( *line == *pat ){
							if( GREP_SET == p ){
								found = 1;
								break;
							}
							else return NULL;
					}
				}
				if( !found && GREP_SET == p ) return NULL;
				for( ; GREP_EOS != *pat; ++pat ) ;  /* skip to end of set */
			break;

			case GREP_EOL:  /* only in GREP_GREP mode */
				if( EOS != *line ) return NULL;
			break;
			case GREP_BOL:  /* only in GREP_GREP mode */
				if( line != _sLine ) return NULL;
				--line;
			break;
			default:       /* match single character  */
				if( *line != *pat ) return NULL;
		}
	}

	/* GREP mode, EOL in pattern and end of line, we got a match */
	if( GREP_GREP == _mmode && GREP_EOL == *pat && EOS == *line )
		return line;

	/* end of pattern, GREP mode success, WILDCARD mode: Ok only on EOS */
	if( GREP_EOP == *pat ){
		if( GREP_GREP == _mmode ) return line;  /* in grep mode, success  */
		if( EOS == *line ) return line; /* glob mode: only if string ends */
	}
	/* last char in pattern STAR, in WILDCARD, we have a match */
	else if( GREP_WILDCARD == _mmode )
		if( GREP_STAR == *pat && GREP_EOP == pat[1] ) return line;

	/* anything else is not a match */
	return NULL;
}

/*
 * performs grep matching against line
*/
	char*
grep_match( const char *line )
{
	_mmode = GREP_GREP;
	_sLine = (char *)line;
	for( ; EOS != *line; ++line )
		if( NULL != pmatch(line, _patBuf) ) return (char *)line;
	return NULL;
}

/*
 * performs wild-card matching against string
*/
	int
glob_match( const char *s )
{
	_mmode = GREP_WILDCARD;
	return (NULL == pmatch(s, _patBuf) ? -1 : 0 );
}


/*
 * performs compilation and matching in one step
*/
	char*
grep( const char *pat, const char *line )
{
	if( -1 == grep_compile(pat) ) return NULL;
	return grep_match( line );
}


/*
 * performs compilation and wildcard matching in one step
*/
	int
glob( const char *pat, const char *s )
{
	if( -1 == glob_compile(pat) ) return -1;
	return glob_match( s );
}

#undef GREP_ANY
#undef GREP_STAR
#undef GREP_PLUS
#undef GREP_SET
#undef GREP_NSET
#undef GREP_EOS
#undef GREP_EOL
#undef GREP_BOL
#undef GREP_RANGE
#undef GREP_EOP
#undef GREP_WILDCARD
#undef GREP_GREP
#undef reterr
#undef error
#undef rerror
#undef store
#undef rstore

