
{*******************************************************}
{                                                       }
{      Borland Pascal 7.0 WinG Rotating Cube Demo       }
{                                                       }
{      Translated from C++ version by                   }
{      Mike Scott, CIS 100140,2420, 24th July 1994      }
{                                                       }
{*******************************************************}

program Cube ;

{$R cube.res}

uses WinTypes, WinProcs, Objects, OWindows, Odialogs, WinG, Dumb3D ;

const
  AmbientLight : real = 0.2 ;

{ cube edges - ordered indices into the vertex array }
const
  CubeFaces : array[ 0..5, 0..3 ] of integer =
              ( ( 0, 1, 2, 3 ),
                ( 2, 1, 6, 5 ),
                ( 3, 2, 5, 4 ),
                ( 0, 3, 4, 7 ),
                ( 1, 0, 7, 6 ),
                ( 4, 5, 6, 7 ) ) ;

{ Cube colours - one RGB colour per surface }
const
  CubeColors : array[ 0..5, 0..2 ] of byte =
               ( ( 240, 20, 20 ),
                 ( 20, 240, 20 ),
                 ( 20, 20, 240 ),
                 ( 128, 64, 0 ),
                 ( 240, 20, 240 ),
                 ( 240, 240, 20 ) ) ;

var i : integer ;

{ TApp Declaration }

type
  PApp = ^TApp ;
  TApp = object(  TApplication )
    procedure InitMainWindow ; virtual ;
    function  IdleAction : boolean ; virtual ;
  end ;

{ TMainWindow }

const
  cm_About        = 1 ;
  cm_Spin         = 3 ;
  cm_Dispersed8x8 = 4 ;
  cm_Speed1       = 30 ;
  nSpeeds         = 5 ;

type
  PMainWindow = ^TMainWindow ;
  TMainWindow = object( TWindow )
    constructor Init( AParent : PWindowsObject ;
                      AName   : PChar ) ;
    destructor Done ; virtual ;
    function  GetClassName : PChar ; virtual ;
    procedure GetWindowClass( var AWndClass : TWndClass ) ; virtual ;
    procedure SetupWindow ; virtual ;
    procedure InitCubeTransform ;
    procedure wmPaletteChanged( var Msg : TMessage ) ; virtual wm_First + wm_PaletteChanged ;
    procedure wmQueryNewPalette( var Msg : TMessage ) ; virtual wm_First + wm_QueryNewPalette ;
    procedure wmActivateApp( var Msg : TMessage ) ; virtual wm_First + wm_ActivateApp ;
    procedure wmSize( var Msg : TMessage ) ; virtual wm_First + wm_Size ;
    procedure cmSpin( var Msg : TMessage ) ; virtual cm_First + cm_Spin ;
    procedure DefCommandProc( var Msg : TMessage ) ; virtual ;
    procedure ProjectAndDrawCube( DC : HDC ;
                                  XOffset, YOffset : integer ) ;
    procedure TransformCube( const Transform : TMatrix4x4 ) ;
    procedure PaintCube( DC : HDC ) ;
    procedure Paint( DC : HDC ;
                     var PaintInfo : TPaintStruct ) ; virtual ;
    procedure wmLButtonDown( var Msg : TMessage ) ; virtual wm_First + wm_LButtonDown ;
    procedure wmMouseMove( var Msg : TMessage ) ; virtual wm_First + wm_MouseMove ;
    procedure IdleAction ;
  private
    { flags }
    AppActive : boolean ;
    SpinFlag  : boolean ;

    { application's palette }
    hPalApp   : HPalette ;

    { WinG device context plus original and current bitmaps }
    hdcWinG   : HDC ;
    OldBitmap : HBitmap ;
    WinGBitmap : HBitmap ;

    { bitmap info header and color table }
    HeaderAndPalette : record
      Header : TBitmapInfoHeader ;
      aColorTable : array[ 0..255 ] of TRGBQuad ;
    end ;

    { width and height of the WinGBitmap surface }
    DIBWidth, DIBHeight : integer ;

    { vertices of the cube }
    CubeVertices : array[ 0..7 ] of TPoint4 ;

    { normals, shades and transform matrix }
    CubeSurfaceNormals : array[ 0..5 ] of TVector4 ;
    CubeSurfaceShades  : array[ 0..5 ] of real ;
    CubeTransform      : TMatrix4x4 ;
    Speed              : 1..nSpeeds ;

    { lighting vector }
    LightSourceDirection : TVector4 ;

    { viewing and perspective }
    ViewPerspective : TMatrix4x4 ;
    ViewPoint       : TPoint4 ;
    Up              : TVector4 ;
    Origin          : TPoint4 ;

    { current WinG dither type }
    DitherType   : integer ;

    { mouse control etc. }
    LastX, LastY : integer ;
    XMove, YMove : real ;
    OldPen : HPen ;
  end ;


{ TMainWindow }

constructor TMainWindow.Init( AParent : PWindowsObject ;
                              AName   : PChar ) ;

begin
  inherited Init( AParent, AName ) ;
  with Attr do begin
    Style := ws_Overlapped or ws_Visible or ws_SysMenu or ws_MinimizeBox ;
    w := 350 ;
    h := 350 ;
    Menu := LoadMenu( HInstance, 'AppMenu' ) ;
  end ;

  { reset the app's palette, normals and vertices for now }
  hPalApp := 0 ;
  for i := 0 to 5 do CubeSurfaceNormals[ i ].InitZero ;
  for i := 0 to 7 do CubeVertices[ i ].Init( 0, 0, 0 ) ;

  { initialise header and palette as C++ declaration does }
  with HeaderAndPalette.Header do begin
    biSize := sizeof( TBitmapInfoHeader ) ;
    biWidth := 50 ;
    biHeight := 50 ;
    biPlanes := 1 ;
    biBitCount := 8 ;
    biCompression := BI_RGB ;
    biSizeImage := 0 ;
    biXPelsPerMeter := 0 ;
    biYPelsPerMeter := 0 ;
    biClrUsed := 0 ;
    biClrImportant := 0 ;
  end ;

  { initialise CubeVertices as C++ declaration }
  CubeVertices[ 0 ].Init( -10,  10, -10 ) ;
  CubeVertices[ 1 ].Init( -10,  10,  10 ) ;
  CubeVertices[ 2 ].Init(  10,  10,  10 ) ;
  CubeVertices[ 3 ].Init(  10,  10, -10 ) ;
  CubeVertices[ 4 ].Init(  10, -10, -10 ) ;
  CubeVertices[ 5 ].Init(  10, -10,  10 ) ;
  CubeVertices[ 6 ].Init( -10, -10,  10 ) ;
  CubeVertices[ 7 ].Init( -10, -10, -10 ) ;

  { initialise the view system }
  ViewPoint.Init( 60, 60, 60 ) ;
  Up.Init( 0, 1, 0 ) ;
  Origin.Init( 0, 0, 0 ) ;

  { initialise various fields }
  SpinFlag := true ;
  AppActive := true ;
  DitherType := 0 ;
  Xmove := 0 ;
  YMove := 0 ;

  { set and normalize the light source }
  LightSourceDirection.Init( 50, 30, -15 ) ;
  LightSourceDirection.Normalize ;

  { Distance to view plane }
  ViewPerspective.Init ;
  ViewPerspective.Elements[ 3, 2 ] := 1 / 300.0 ;
  ViewPerspective.Elements[ 3, 3 ] := 0.0 ;

  { viewport scaling - some arbitrary number like 3.5 will do }
  ViewPerspective.Elements[ 0, 0 ] := 3.5 ;
  ViewPerspective.Elements[ 1, 1 ] := 3.5 ;

  { set up the cube transform }
  Speed := 3 ;
  InitCubeTransform ;

  { get the hdcWinG }
  hdcWinG := WinGCreateDC ;
  WinGBitmap := 0 ;
  if hdcWinG <> 0 then
    OldPen := SelectObject( hdcWinG, GetStockObject( NULL_PEN ) ) else
    OldPen := 0 ;
end ;


destructor TMainWindow.Done ;

begin
  inherited Done ;
  DeleteObject( hPalApp ) ;
  if hdcWinG <> 0 then begin
    SelectObject( hdcWinG, OldPen ) ;
    SelectObject( hdcWinG, OldBitmap ) ;
    DeleteObject( WinGBitmap ) ;
    DeleteDC( hdcWinG ) ;
  end ;
end ;


function  TMainWindow.GetClassName : PChar ;

begin
  GetClassName := 'PascalCube' ;
end ;


procedure TMainWindow.GetWindowClass( var AWndClass : TWndClass ) ;

begin
  inherited GetWindowClass( AWndClass ) ;
  with AWndClass do begin
    Style := CS_BYTEALIGNCLIENT or CS_VREDRAW or CS_HREDRAW ;
    hIcon := 0 ;
  end ;
end ;


procedure TMainWindow.SetupWindow ;

var Temp : byte ;

begin
  inherited SetupWindow ;

  { use the halftone palette to make things easy }
  hPalApp := WinGCreateHalftonePalette ;
  GetPaletteEntries( hPalApp, 0, 256, HeaderAndPalette.aColorTable ) ;

  { palette entries and RGBQuads are backwards }
  with HeaderAndPalette do for i := 0 to 255 do with aColorTable[ i ] do begin
    Temp := rgbBlue ;
    rgbBlue := rgbRed ;
    rgbRed := Temp ;
  end ;

  { set the default spin and fill option }
  with Attr do begin
    CheckMenuItem( Menu, cm_Spin, MF_CHECKED ) ;
    CheckMenuItem( Menu, cm_Dispersed8x8, MF_CHECKED ) ;
    CheckMenuItem( Menu, cm_Speed1 + Speed - 1, MF_CHECKED ) ;
  end ;
end ;


procedure TMainWindow.InitCubeTransform ;

const Angles : array[ 1..nSpeeds ] of array[ 0..2 ] of real =
               ( ( 0.08, 0.2, 0.15 ),
                 ( 0.1, -0.8, 0.3 ),
                 ( -1.2, 0.7, 2.0 ),
                 ( 6.0, 3.5, 2.0 ),
                 ( 5.0, -7.0, 12.0 ) ) ;
begin
  CubeTransform.Init ;
  TransformCube( CubeTransform ) ;

  { Then generate an interesting rotation for the spin }
  with CubeTransform do begin
    ConcatenateYRotation( Angles[ Speed ][ 0 ] ) ;
    ConcatenateXRotation( Angles[ Speed ][ 1 ] ) ;
    ConcatenateZRotation( Angles[ Speed ][ 2 ] ) ;
  end ;
end ;


procedure TMainWindow.wmPaletteChanged( var Msg : TMessage ) ;

begin
  if Msg.wParam <> HWindow then wmQueryNewPalette( Msg ) ;
end ;


procedure TMainWindow.wmQueryNewPalette( var Msg : TMessage ) ;

var DC         : HDC ;
    OldPalette : HPalette ;
    i          : integer ;

begin
  DC := GetDC( HWindow ) ;
  OldPalette := SelectPalette( DC, hPalApp, false ) ;
  i := RealizePalette( DC ) ;
  SelectPalette( DC, OldPalette, true ) ;
  ReleaseDC( HWindow, DC ) ;
  if i <> 0 then InvalidateRect( HWindow, nil, false ) ;
  Msg.Result := i ;
end ;


procedure TMainWindow.wmActivateApp( var Msg : TMessage ) ;

begin
  AppActive := Msg.wParam <> 0 ;
end ;


procedure TMainWindow.wmSize( var Msg : TMessage ) ;

var ARect : TRect ;

begin
  inherited wmSize( Msg ) ;
  if not IsIconic( HWindow ) then begin
    { create a WinGBitmap for the buffer that fills the client area}
    if WinGBitmap <> 0 then begin
      SelectObject( hdcWinG, OldBitmap ) ;
      DeleteObject( WinGBitmap ) ;
    end ;

    GetClientRect( HWindow, ARect ) ;
    with HeaderAndPalette.Header do begin
      biWidth := ARect.Right ;
      biHeight := ARect.Bottom ;
    end ;

    WinGBitmap := WinGCreateBitmap( hdcWinG,
                                    PBitmapInfo( @HeaderAndPalette ),
                                    nil ) ;
    OldBitmap := SelectObject( hdcWinG, WinGBitmap ) ;

    DIBWidth := ARect.Right ;
    DIBHeight := ARect.Bottom ;
  end ;
end ;


procedure TMainWindow.cmSpin( var Msg : TMessage ) ;

begin
  if SpinFlag then
    CheckMenuItem( Attr.Menu, cm_Spin, MF_UNCHECKED ) else
    CheckMenuItem( Attr.Menu, cm_Spin, MF_CHECKED ) ;
  SpinFlag := not SpinFlag ;
end ;


procedure TMainWindow.DefCommandProc( var Msg : TMessage ) ;

begin
  Msg.Result := 1 ;
  with Msg do begin
    if ( cm_Dispersed8x8 <= wParam ) and
       ( wParam <= cm_Dispersed8x8 + 2 ) then
    begin
      CheckMenuItem( Attr.Menu, cm_Dispersed8x8 + DitherType, MF_UNCHECKED ) ;
      DitherType := Msg.wParam - cm_Dispersed8x8 ;
      CheckMenuItem( Attr.Menu, cm_Dispersed8x8 + DitherType, MF_CHECKED ) ;
      InvalidateRect( HWindow, nil, false ) ;
      UpdateWindow( HWindow ) ;
    end else
    if ( cm_Speed1 <= wParam ) and ( wParam < cm_Speed1 + nSpeeds ) then begin
      CheckMenuItem( Attr.Menu, cm_Speed1 + Speed - 1, MF_UNCHECKED ) ;
      Speed := wParam - cm_Speed1 + 1 ;
      CheckMenuItem( Attr.Menu, cm_Speed1 + Speed - 1, MF_CHECKED ) ;
      InitCubeTransform ;
    end else
    if wParam = cm_About then
      Application^.ExecDialog( new( PDialog, Init( @Self, 'AppAbout' ) ) ) ;
  end ;
end ;


procedure TMainWindow.ProjectAndDrawCube( DC : HDC ;
                                          XOffset, YOffset : integer ) ;

const DitherTypeList : array[ 0..2 ] of WING_DITHER_TYPE =
                       ( WING_DISPERSED_8x8, WING_DISPERSED_4x4,
                         WING_CLUSTERED_4x4 ) ;

var ViewDirection : TVector4 ;
    View          : TViewTransform ;
    i, j          : integer ;
    aScreenVertices : array[ 0..7 ] of TPoint ;
    cr              : TColorRef ;
    hbr             : HBrush ;
    aQuadVertices   : array[ 0..3 ] of TPoint ;
    Temp            : TPoint4 ;
    v1, w1, v2, w2  : real ;
    ADitherType     : WING_DITHER_TYPE ;

begin
  { Create a viewing transform for the current eye position }
  with ViewDirection do begin
    InitCopy( Origin ) ;
    Subtract( ViewPoint ) ;
    Normalize ;
  end ;
  View.Init( ViewPoint, ViewDirection, Up ) ;

  { transform and project the vertices into screen space }
  with Temp do for i := 0 to 7 do begin
    InitCopy( CubeVertices[ i ] ) ;
    MultiplyByMatrix( View ) ;
    MultiplyByMatrix( ViewPerspective ) ;
    Homogenize ;

    aScreenVertices[ i ].x := Trunc( Temp.Elements.x ) + XOffset ;
    aScreenVertices[ i ].y := Trunc( Temp.Elements.y ) + YOffset ;
  end ;

  for i := 0 to 5 do begin
    { standard culling operation }
    v1 := aScreenVertices[ CubeFaces[ i, 2 ] ].x -
          aScreenVertices[ CubeFaces[ i, 1 ] ].x ;
    w1 := aScreenVertices[ CubeFaces[ i, 0 ] ].x -
          aScreenVertices[ CubeFaces[ i, 1 ] ].x ;
    v2 := aScreenVertices[ CubeFaces[ i, 2 ] ].y -
          aScreenVertices[ CubeFaces[ i, 1 ] ].y ;
    w2 := aScreenVertices[ CubeFaces[ i, 0 ] ].y -
          aScreenVertices[ CubeFaces[ i, 1 ] ].y ;
    if v1 * w2 - v2 * w1 <= 0 then continue ;

    { get a brush for the shaded face color using the selected dither }
    cr := RGB( Trunc( CubeColors[ i, 0 ] * CubeSurfaceShades[ i ] ),
               Trunc( CubeColors[ i, 1 ] * CubeSurfaceShades[ i ] ),
               Trunc( CubeColors[ i, 2 ] * CubeSurfaceShades[ i ] ) ) ;
    hbr := WinGCreateHalftoneBrush( hdcWinG, cr, DitherTypeList[ DitherType ] ) ;

    { collect the correct points in an array }
    for j := 0 to 3 do
      aQuadVertices[ j ] := aScreenVertices[ CubeFaces[ i, j ] ] ;

    { use GDI to draw the face }
    hbr := SelectObject( DC, hbr ) ;
    Polygon( DC, aQuadVertices, 4 ) ;
    hbr := SelectObject( DC, hbr ) ;
    DeleteObject( hbr ) ;
  end ;
end;


procedure TMainWindow.TransformCube( const Transform : TMatrix4x4 ) ;

var i : integer ;
    Edge1, Edge2 : TVector4 ;
    Shade : real ;

begin
  { Transform the cube by the matrix }
  for i := 0 to 7 do
    CubeVertices[ i ].MultiplyByMatrix( Transform ) ;

  { Recalculate normals and shades }
  for i := 0 to 5 do begin
    { Normals are perpendicular to two edges of the cube }
    Edge1.InitCopy( CubeVertices[ CubeFaces[ i, 1 ] ] ) ;
    Edge1.Subtract( CubeVertices[ CubeFaces[ i, 0 ] ] ) ;
    Edge2.InitCopy( CubeVertices[ CubeFaces[ i, 3 ] ] ) ;
    Edge2.Subtract( CubeVertices[ CubeFaces[ i, 0 ] ] ) ;
    with CubeSurfaceNormals[ i ] do begin
      InitCopy( Edge1 ) ;
      CrossProduct( Edge2 ) ;
      Normalize ;

      { cosine shading based on the surface normal clamped to [0, 1] }
      Shade := DotProduct( LightSourceDirection ) ;
      Shade := Shade + AmbientLight ;
      if Shade < 0.0 then Shade := 0.0 else
      if Shade > 1.0 then Shade := 1.0 ;
      CubeSurfaceShades[ i ] := Shade ;
    end ;
  end ;
end ;


procedure TMainWindow.PaintCube( DC : HDC ) ;

var Movement : TMatrix4x4 ;
    ARect    : TRect ;

begin
  { clear the DIB buffer to white }
  PatBlt( hdcWinG, 0, 0, DibWidth, DibHeight, WHITENESS ) ;

  { move the viewport according to the mouse movement
   // rotate it around the x & y axes }

  if ( YMove <> 0 ) or ( XMove <> 0 ) then begin
    Movement.Init ;
    Movement.ConcatenateYRotation( -YMove ) ;
    Movement.ConcatenateXRotation( XMove ) ;
    XMove := 0 ;
    YMove := 0 ;
    TransformCube( Movement ) ;
  end ;

  { and GO! }
  ProjectAndDrawCube( hdcWinG, DibWidth div 2, DibHeight div 2 ) ;

  GetClientRect( HWindow, ARect ) ;
  if IsIconic( HWindow ) then
    WinGStretchBlt( DC, 0, 0, ARect.Right, ARect.Bottom,
                    hdcWinG, 0, 0, DIBWidth, DIBHeight )
  else
    WinGBitBlt( DC, 0, 0, ARect.Right, ARect.Bottom,
                hdcWinG, 0, 0 ) ;
end ;


procedure TMainWindow.Paint( DC : HDC ;
                             var PaintInfo : TPaintStruct ) ;

var OldPalette : HPalette ;

begin
  if hPalApp <> 0 then begin
    OldPalette := SelectPalette( DC, hPalApp, false ) ;
    RealizePalette( DC ) ;
  end ;
  PaintCube( DC ) ;
  if hPalApp <> 0 then SelectPalette( DC, OldPalette, true ) ;
end ;


procedure TMainWindow.wmLButtonDown( var Msg : TMessage ) ;

begin
  inherited wmLButtonDown( Msg ) ;
  LastX := Msg.LParamLo ;
  LastY := Msg.LParamHi ;
end ;


procedure TMainWindow.wmMouseMove( var Msg : TMessage ) ;

var x, y : integer ;

begin
  if GetKeyState( VK_LBUTTON ) < 0 then with Msg do begin
    x := LParamLo ;
    y := LParamHi ;
    YMove := x - LastX ;
    XMove := y - LastY ;

    LastX := x ;
    LastY := y ;

    InvalidateRect( HWindow, nil, false ) ;
    UpdateWindow( HWindow ) ;
  end ;
end ;


procedure TMainWindow.IdleAction ;

var DC : HDC ;
    OldPalette : HPalette ;

begin
  { spin while the app is active, lbutton is up, spinning is on.
    spin while the app is iconized}
  if ( SpinFlag and AppActive and ( GetKeyState( VK_LBUTTON ) >= 0 ) ) or
     IsIconic( HWindow ) then
  begin
    TransformCube( CubeTransform ) ;
    DC := GetDC( HWindow ) ;
    if hPalApp <> 0 then begin
      OldPalette := SelectPalette( DC, hPalApp, false ) ;
      RealizePalette( DC ) ;
    end ;
    PaintCube( DC ) ;
    if hPalApp <> 0 then SelectPalette( DC, OldPalette, true ) ;
    ReleaseDC( HWindow, DC ) ;
  end ;
end ;


{ TApp }

procedure TApp.InitMainWindow ;

begin
  MainWindow := new( PMainWindow, Init( nil, 'Spinning Cube in Pascal' ) ) ;
end ;


function  TApp.IdleAction : boolean ;

begin
  if MainWindow <> nil then PMainWindow( MainWindow )^.IdleAction ;
  IdleAction := true ;
end ;


var App : TApp ;

begin
  with App do begin
    Init( 'Cube' ) ;
    Run ;
    Done ;
  end ;
end.
