{******************************************************}
{*                   DOORIO.PAS 2.01                  *}
{*      Copyright (c) TurboPower Software 1993.       *}
{*                All rights reserved.                *}
{******************************************************}

{$I APDEFINE.INC}

{$IFNDEF UseUart}
  !! Error!  The settings in APDEFINE.INC are incompatible with this unit
{$ENDIF}

{$IFNDEF UseOOP}
  !! Error - You must use the object-oriented interface for this program
{$ENDIF}

{$S-,R-,V-,I-,B-,F+,O+,A-}

unit DoorIO;
  {-Unit for door program screen I/O }

interface

uses
  {$IFDEF UseOpro}
  OpCrt,
  OpString,
  {$ENDIF}
  {$IFDEF UseTpro}
  TpCrt,
  TpString,
  {$ENDIF}
  {$IFDEF Standalone}
  Crt,
  {$ENDIF}
  Dos,
  ApMisc,
  ApPort,
  OoCom,
  ApAnsi,
  ApTimer;


const
  OutCharTics : Word = 9;
    {-Number of tics to wait for a character to leave the output buffer }

  WaitCharTics : Word = 3276;
    {-Wait 3 minutes for user input }

  {$IFNDEF UseFossil}
  UseFossil   : Boolean = False;
    {-True if the DOORIO unit should use a fossil driver instead of UART }
  {$ELSE}
    {$IFDEF UseUart}
    UseFossil : Boolean = False;
    {$ELSE}
    UseFossil : Boolean = True;
    {$ENDIF}
  {$ENDIF}

  OutOfTimeStr =
    #27'[2J'#27'[0mYour time has expired for the day.  Returning you to the BBS...'#13#10#13#10;
    {-Message displayed if the user runs out of time }

  InactivityStr =
    #27'[2J'#27'[0mYou haven''t typed anything in a while.  I''ll let the BBS deal with you...'#13#10#13#10;

type
  DoorKeyPressedFunc = function : Boolean;
  DoorGetKeyFunc     = function : Char;

  DoorConfigRec =
    record
      LocalMode        : Boolean;
      ComPort          : AbstractPortPtr;
      ScreenDisplay    : Boolean;
      FullName         : String[80];
      SecondsRemaining : LongInt;
      PageLength       : Word;
    end;

var
  DoorConfig     : DoorConfigRec;
  DoorKeyPressed : DoorKeyPressedFunc;
  DoorGetKey     : DoorGetKeyFunc;
  DoorIn         : Text;
  DoorOut        : Text;
  TimeRemaining  : EventTimer;

const
  ecAnsiRequired = 9999;

function ReadDoorSys(DoorSysName : PathStr) : Boolean;
  {-Read a DOOR.SYS file into the DOORIO configuration record }

procedure DoorShutDown;
  {-Shut down the DOORIO unit }

procedure AssignAnsiDev(var F : Text);
  {-Assign an ANSI text file device driver to F }

procedure DoorClrScr;
  {-Clear the local and remote screens }

function DoorGetChar : Char;
  {-Get a keystroke from local or remote }

procedure DoorGotoXY(X, Y : Byte);
  {-Goto the X,Y screen coordinates on local and remote }

procedure DoorNormVideo;
  {-Turn on normal video }

procedure DoorBoldVideo;
  {-Turn on bold video }

procedure DoorSetFG(Color : Byte);
  {-Change foreground attribute }

procedure DoorSetAttr(FG, BG : Byte);
  {-Change foreground and background attributes }

procedure DoorEraseEOL;
  {-Erase to the end of the current line }

function DisplayTextFile(FName : PathStr; Color : Boolean) : Boolean;
  {-Write a text file to the screen }

implementation

  {$IFDEF Standalone}
  function Long2Str(L : LongInt) : string;
    {-Convert a long/word/integer/byte/shortint to a string}
  var
    S : string;
  begin
    Str(L, S);
    Long2Str := S;
  end;

  function HasExtension(Name : string; var DotPos : Word) : Boolean;
    {-Return whether and position of extension separator dot in a pathname}
  var
    I : Word;
  begin
    DotPos := 0;
    for I := Length(Name) downto 1 do
      if (Name[I] = '.') and (DotPos = 0) then
        DotPos := I;
    HasExtension := (DotPos > 0) and (Pos('\', Copy(Name, Succ(DotPos), 64)) = 0);
  end;

  function ForceExtension(Name : string; Ext : ExtStr) : string;
    {-Return a pathname with the specified extension attached}
  var
    DotPos : Word;
  begin
    if HasExtension(Name, DotPos) then
      ForceExtension := Copy(Name, 1, DotPos)+Ext
    else if Name = '' then
      ForceExtension := ''
    else
      ForceExtension := Name+'.'+Ext;
  end;
  {$ENDIF}

  procedure FlushPortPrim;
  var
    Tics : LongInt;

  begin
    with DoorConfig.ComPort^ do begin
      { add 100 character times for safety }
      Tics := ((PR^.OutBuffCount + 100) * LongInt(100)) div (PR^.CurBaud div 10);
      TIcs := (Tics * 182) div 1000;

      DrainOutBuffer(Tics);
    end;
  end;

  procedure WriteCharBoth(Ch : Char);
  begin
    WriteCharAnsi(Ch);
    if (DoorConfig.ComPort <> nil) then
      DoorConfig.ComPort^.PutCharTimeOut(Ch, OutCharTics);
  end;

  procedure WriteBothPrim(St : String);
  var
    I : Byte;

  begin
    for I := 1 to Length(St) do
      WriteCharBoth(St[I]);
  end;

  function PortAborted : Boolean;
  var
    I : Byte;

  begin
    with DoorConfig do begin
      if (ComPort <> nil) then begin
        if not ComPort^.CheckDCD then begin
          DoorShutDown;
          Halt;
        end else if TimerExpired(TimeRemaining) then begin
          WriteBothPrim(OutOfTimeStr);
          DoorShutDown;
          Halt;
        end else
          PortAborted := ComPort^.PR^.UserAbort;
      end else
        PortAborted := False;
    end;
  end;

  procedure WriteStrBoth(St : String);
  begin
    WriteStringAnsi(St);
    if (DoorConfig.ComPort <> nil) then begin
      DoorConfig.ComPort^.PutStringTimeOut(St, OutCharTics * Length(St));
      if not PortAborted then
        FlushPortPrim;
    end;
  end;

  function AnsiFlush(var F : TextRec) : Integer;
    {-Flush an ANSI TFDD }
  var
    Tics : LongInt;

  begin
    if (DoorConfig.ComPort <> nil) then
      FlushPortPrim;
    AnsiFlush := 0;
  end;

  function AnsiOut(var F : TextRec) : Integer;
    {-Move output characters to screen and port }
  var
    St : String;

  label
    EndingPoint;

  begin
    with F do begin
      if (BufPos = 0) then
        goto EndingPoint;

      St := '';
      Move(BufPtr^[0], St[1], BufPos);
      St[0] := Char(BufPos);
      WriteStringAnsi(St);

      if (DoorConfig.ComPort <> nil) then begin
        DoorConfig.ComPort^.PutStringTimeOut(St, Length(St) * OutCharTics);
        if not PortAborted then
          FlushPortPrim;
      end;
    end;

EndingPoint:
    AnsiOut  := 0;
    F.BufPos := 0;
  end;

  function AnsiIn(var F : TextRec) : Integer;
  var
    Cnt      : Word;
    Ch       : Char;
    Finished : Boolean;
    BufMax   : Word;
    KeyTimer : EventTimer;

    function CharAvail : Boolean;
    begin
      CharAvail := True;

      if (DoorConfig.ComPort <> nil) then begin
        with DoorConfig.ComPort^ do
          if DoorKeyPressed then
            Ch := DoorGetKey
          else if CharReady then begin
            GetChar(Ch);
            if (AsyncStatus <> ecOK) then
              InOutRes := AsyncStatus;
          end else
            CharAvail := False;
      end else
        if DoorKeyPressed then
          Ch := DoorGetKey
        else
          CharAvail := False;
    end;

  begin
    with F do begin
      Cnt    := 0;
      BufMax := BufSize - 2;
      AnsiIn := 0;

      repeat
        { wait for a key press }
        NewTimer(KeyTimer, WaitCharTics);
        while not CharAvail and not TimerExpired(KeyTimer) do
          if (AsyncStatus <> ecOK) or PortAborted then
            Exit;

        if TimerExpired(KeyTimer) then begin
          WriteBothPrim(InactivityStr);
          DoorShutDown;
          Halt;
        end;

        if (Ch = cBS) then begin
          if (Cnt <> 0) then begin
            Dec(Cnt);
            WriteCharBoth(#8);
            WriteCharBoth(#32);
            WriteCharBoth(#8);
          end
        end else if (Ch = cCR) or (Ch = cEsc) or (Ch >= #32) then begin
          BufPtr^[Cnt] := Ch;
          if (Cnt < BufMax) then
            Inc(Cnt);

          Finished := True;

          if (Ch = cCr) then begin
            if (Cnt = BufMax) then
              Inc(Cnt);
            BufPtr^[Cnt] := cLF;
            Inc(Cnt);
            WriteCharBoth(#13);
            WriteCharBoth(#10);
          end else begin
            WriteCharAnsi(Ch);
            if (DoorConfig.ComPort <> nil) then
              DoorConfig.ComPort^.PutCharTimeOut(Ch, OutCharTics);
            Finished := False;
          end;
        end else
          Finished := False;

      until Finished;

      BufPos := 0;
      BufEnd := Cnt;
    end;
  end;

  function AnsiNA(var F : TextRec) : Integer;
  begin
    AnsiNA := 0;
  end;

  function AnsiOpen(var F : TextRec) : Integer;
    {-Open the ANSI TFDD for input or output }
  begin
    AnsiOpen := 0;
    with TextRec(F) do begin
      if (Mode = fmOutput) then begin
        Mode := fmOutput;
        InOutFunc := @AnsiOut;
        FlushFunc := @AnsiFlush;
        BufSize   := SizeOf(Char);
      end else begin
        Mode := fmInput;
        InOutFunc := @AnsiIn;
        FlushFunc := @AnsiNA;
      end;
    end;
  end;

  procedure AssignAnsiDev(var F : Text);
    {-Assign an ANSI text file device driver to F }
  const
    AnsiName : String[5] = 'ANSI'#0;

  begin
    with TextRec(F) do begin
      Handle    := $FFFF;
      Mode      := fmClosed;
      BufSize   := SizeOf(Buffer);
      BufPtr    := @Buffer;
      OpenFunc  := @AnsiOpen;
      CloseFunc := @AnsiNA;
      Move(AnsiName[1], Name, Length(AnsiName));
    end;
  end;

  function DoorGetChar : Char;
  var
    Ch       : Char;
    KeyTimer : EventTimer;

  begin
    DoorGetChar := #0;

    NewTimer(KeyTimer, WaitCharTics);
    repeat
    until TimerExpired(KeyTimer) or
          DoorKeyPressed or
          PortAborted or
          ((DoorConfig.ComPort <> nil) and DoorConfig.ComPort^.CharReady);

    if (TimerExpired(KeyTimer)) then begin
      WriteBothPrim(InactivityStr);
      DoorShutDown;
      Halt;
    end;

    if (AsyncStatus <> ecOK) then
      Exit;

    if DoorKeyPressed then
      Ch := DoorGetKey
    else
      DoorConfig.ComPort^.GetChar(Ch);

    DoorGetChar := Ch;
  end;

  procedure DoorClrScr;
    {-Clear the local and remote screens }
  begin
    WriteStrBoth(#27'[2J');
  end;

  procedure DoorGotoXY(X, Y : Byte);
    {-Goto the X,Y screen coordinates on local and remote }
  begin
    WriteStrBoth(#27'[' + Long2Str(Y) + ';' + Long2Str(X) + 'H');
  end;

  procedure DoorNormVideo;
    {-Turn on normal video }
  begin
    WriteStrBoth(#27'[0m');
  end;

  procedure DoorBoldVideo;
    {-Turn on bold video }
  begin
    WriteStrBoth(#27'[1m');
  end;

  procedure DoorSetFG(Color : Byte);
    {-Change foreground attribute }
  begin
    WriteStrBoth(#27'[' + Long2Str(Color) + 'm');
  end;

  procedure DoorSetAttr(FG, BG : Byte);
    {-Change foreground and background attributes }
  begin
    WriteStrBoth(#27'[' + Long2Str(FG) + ';' + Long2Str(BG) + 'm');
  end;

  procedure DoorEraseEOL;
    {-Erase to the end of the current line }
  begin
    WriteStrBoth(#27'[0K');
  end;

  function DisplayTextFile(FName : PathStr; Color : Boolean) : Boolean;
    {-Write a text file to the screen }
  var
    F : Text;
    S : String;

  begin
    DisplayTextFile := False;
    if Color then
      FName := ForceExtension(FName, 'ANS')
    else
      FName := ForceExtension(FName, 'TXT');

    Assign(F, FName);
    Reset(F);
    if (IoResult <> ecOK) then
      Exit;
    while not Eof(F) do begin
      ReadLn(F, S);
      if (IoResult <> ecOK) then begin
        Close(F); if (IoResult = 0) then ;
        Exit;
      end;
      WriteLn(DoorOut, S);
    end;

    Close(F);
    DisplayTextFile := (IoResult = ecOK);
  end;

  function ReadDoorSys(DoorSysName : PathStr) : Boolean;
    {-Read a DOOR.SYS file into the DOORIO configuration record }
  var
    DoorSys  : Text;
    Line     : String;
    GraphStr : String;
    LineLen  : Byte absolute Line;
    Temp     : LongInt;
    Err      : Integer;

    { port parameters }
    ComName  : ComNameType;
    Local    : Boolean;
    Baud     : LongInt;
    Parity   : ParityType;
    DataBits : DataBitType;

  label
    ExitPoint;

    function ReadAndCheck(UpCaseIt : Boolean) : Boolean;
    begin
      ReadLn(DoorSys, Line);
      if UpCaseIt then
        Line := StUpCase(Line);
      AsyncStatus := IoResult;
      ReadAndCheck := (AsyncStatus = ecOK);
    end;

    function ReadNumber : Boolean;
    begin
      ReadLn(DoorSys, Temp);
      AsyncStatus := IoResult;
      ReadNumber := (AsyncStatus = ecOK);
    end;

    function SkipLines(Num : Word) : Boolean;
    var
      I : Word;

    begin
      SkipLines := False;
      for I := 1 to Num do begin
        ReadLn(DoorSys, Line);
        AsyncStatus := IoResult;
        if (AsyncStatus <> ecOK) then
          Exit;
      end;
      SkipLines := True;
    end;

  begin
    ReadDoorSys := False;

    FillChar(DoorConfig, SizeOf(DoorConfig), 0);

    Assign(DoorSys, DoorSysName);
    Reset(DoorSys);
    AsyncStatus := IoResult;
    if (AsyncStatus <> ecOK) then
      Exit;

    { read and assign the ComPort }
    if not ReadAndCheck(True) then
      goto ExitPoint;

    if (LineLen > 6) or (Line[LineLen] <> ':') or (Copy(Line, 1, 3) <> 'COM') then
      goto ExitPoint;
    Dec(LineLen);
    Delete(Line, 1, 3);
    Local := (Line = '0');
    if not Local then begin
      Val(Line, Temp, Err);
      if (Err <> ecOK) then
        goto ExitPoint;

      ComName := ComNameType(Pred(Temp));
    end;

    if not Local then begin
      { read baud rate }
      if not ReadNumber then
        goto ExitPoint;

      { read data bits }
      if not ReadNumber or (Temp < 5) or (Temp > 8) then
        goto ExitPoint;
      if (Temp = 8) then
        Parity := NoParity
      else
        Parity := EvenParity;
      DataBits := Temp;

      { skip the node number }
      if not ReadNumber then
        goto ExitPoint;

      { read actual BPS rate }
      if not ReadNumber then
        goto ExitPoint;
      Baud := Temp;
    end else
      if not SkipLines(4) then
        goto ExitPoint;

    { determine whether or not to display DOOR stuff on the screen }
    if not ReadAndCheck(True) then
      goto ExitPoint;
    if (LineLen = 0) then
      Line := 'Y';
    if (Line = 'Y') then
      DoorConfig.ScreenDisplay := True
    else if (Line = 'N') then
      DoorConfig.ScreenDisplay := False
    else
      goto ExitPoint;

    if not SkipLines(3) then
      goto ExitPoint;

    { get the user's full name }
    if not ReadAndCheck(False) then
      goto ExitPoint;
    DoorConfig.FullName := Line;

    if not SkipLines(7) then
      goto ExitPoint;

    if not ReadNumber then
      goto ExitPoint;
    DoorConfig.SecondsRemaining := Temp;

    if not SkipLines(1) then
      goto ExitPoint;

    if not ReadAndCheck(True) then
      goto ExitPoint;
    GraphStr := Line;

    if not ReadNumber then
      goto ExitPoint;
    DoorConfig.PageLength := Temp;

    { initialize the com port }
    if not Local then
      with DoorConfig do begin
        {$IFDEF UseFossil}
        if UseFossil then
          ComPort := New(FossilPortPtr, InitCustom(
            ComName, Baud, Parity, DataBits, 1, 4096, 4096,
            DefPortOptions and not ptDropModemOnClose and not ptRaiseModemOnOpen))
        else begin
          ComPort := New(UartPortPtr, InitCustom(
            ComName, Baud, Parity, DataBits, 1, 4096, 4096,
            DefPortOptions and not ptDropModemOnClose));
        end;
        {$ELSE}
        ComPort := New(UartPortPtr, InitCustom(
          ComName, Baud, Parity, DataBits, 1, 4096, 4096,
          DefPortOptions and not ptDropModemOnClose));
        {$ENDIF}

        if (ComPort = nil) then
          goto ExitPoint;

        {$IFDEF UseHWFlow}
        ComPort^.HWFlowEnable(3800, 200, hfUseRTS or hfRequireCTS);
        {$ENDIF}
      end
    else
      DoorConfig.ComPort := nil;

    AssignAnsiDev(DoorIn);
    Reset(DoorIn);
    AsyncStatus := IoResult;
    if (AsyncStatus <> ecOK) then begin
      if (DoorConfig.ComPort <> nil) then
        Dispose(DoorConfig.ComPort, Done);
      goto ExitPoint;
    end;

    AssignAnsiDev(DoorOut);
    Rewrite(DoorOut);
    if (AsyncStatus <> ecOK) then begin
      if (DoorConfig.ComPort <> nil) then
        Dispose(DoorConfig.ComPort, Done);
      Close(DoorIn); if (IoResult = 0) then ;
      goto ExitPoint;
    end;

    if (GraphStr <> 'GR') then begin
      WriteLn(DoorOut, 'Sorry, ANSI graphics are required for play');
      WriteLn(DoorOut);
      WriteLn(DoorOut);
      DoorShutDown;
      AsyncStatus := ecAnsiRequired;
      goto ExitPoint;
    end;

    NewTimer(TimeRemaining, Secs2Tics(DoorConfig.SecondsRemaining));

    ReadDoorSys := True;

ExitPoint:
    Close(DoorSys); if (IoResult = 0) then ;
  end;

  procedure DoorShutDown;
    {-Shut down the DOORIO unit }
  begin
    Close(DoorIn); if (IoResult = 0) then ;
    Close(DoorOut); if (IoResult = 0) then ;
    if (DoorConfig.ComPort <> nil) then
      Dispose(DoorConfig.ComPort, Done);
  end;

begin
  DoorKeyPressed := KeyPressed;
  DoorGetKey     := ReadKey;
  DoorConfig.ComPort := nil;
end.
