#include <io.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <memory.h>
#include <sys\types.h>
#include <sys\stat.h>

#include "types.h"
#include "2obj.h"
#include "errs.h"
#include "errors.h"

		/* GIF line decoder, from DECODER.C */
extern int decoder(int);

char    Palette[768];              /* Global Palette variables */
int     PaletteSize;
int     PaletteLevel;


void ByteSwapDword(unsigned long far *DoubleWord);
void ByteSwapWord(unsigned short far *SwapWord);

void CopyLines(int hInfile, int hOutFile, int Width, int Height);


int     get_byte(void);
void    InitialiseReadBuffer(int handle);

char ReadBuffer[4096];
int ReadIndex;
int ReadLimit;
int hBufferedFile;


void InitialiseReadBuffer(int handle)
{
	hBufferedFile = handle;
	ReadIndex = -1;
}


int get_byte()
{
	unsigned char Buffer;

	if(ReadIndex == -1) {
		ReadIndex = 0;

		if( (ReadLimit = read(hBufferedFile,ReadBuffer, 4096)) == -1 )
			Error(E_FATAL, ES_BAD_FILE_READ, NULL, 1);
	}


	Buffer = ReadBuffer[ReadIndex];

	ReadIndex++;

	if(ReadIndex == ReadLimit)
		ReadIndex = -1;

	return Buffer;
}


void ByteSwapDword(unsigned long far *SwapDWord)
{
	_asm {
	les     di,SwapDWord
	mov     ax,word ptr es:[di]
	mov     dx,word ptr es:[di+2]
	xchg    ah,al
	xchg    dh,dl
	mov     word ptr es:[di],dx
	mov     word ptr es:[di+2],ax
	}
}


void ByteSwapWord(unsigned short far *SwapWord)
{
	_asm {
	les     di,SwapWord
	mov     ax,es:[di]
	xchg    ah,al
	mov     es:[di],ax
	}
}


/**** BIN file routines ****/


int IsBIN(int hFile)
{
	return 1;
}


unsigned long SizeBIN(int hFile, int HeaderFlag)
{
	return( filelength(hFile) );
}


void EndBIN()
{
}


/**** CEL file routines ****/



struct  CELHeader
	{
	short   CELYear;
	short   CELWidth;
	short   CELHeight;
	char    CELUnknown[26];
	};


struct CELHeader *pCELHeader;
unsigned long CELFileSize;


int IsCEL(int hFile)
{
	unsigned long CELSize;


		/* The only checks one a CEL that are known at this     */
		/* moment are: a check on the CELYear at the start of   */
		/* the file, and a check that the CEL's (width*height)  */
		/* + 768 palette bytes + 32 bytes header size are equal */
		/* to the size of the file.                             */

	CELFileSize = filelength(hFile);
	pCELHeader = AllocateMemory( sizeof (struct CELHeader) );

	read(hFile, (char *)pCELHeader, sizeof (struct CELHeader) );

	CELSize = sizeof (struct CELHeader) + 768L + (long)pCELHeader->CELWidth * (long)pCELHeader->CELHeight;

	ByteSwapWord(&(pCELHeader->CELYear));

	if( ( pCELHeader->CELYear > 0x1900) && (CELFileSize == CELSize) ) {
		PaletteSize = 768;
		read(hFile, Palette, PaletteSize);
		PaletteLevel = 0;
		return 1;
	}
	else
		return 0;
}


int DecodeCEL(int hInFile, int hOutFile)
{
	lseek(hInFile, sizeof(struct CELHeader) + 768L, SEEK_SET);

	CopyLines(hInFile, hOutFile, pCELHeader->CELWidth, pCELHeader->CELHeight);

	return 0;
}


void    HeaderCEL()
{
	printf("CEL Version    : %4x\n", pCELHeader->CELYear);
	printf("Dimensions     : %ux%ux%u\n", pCELHeader->CELWidth,
					      pCELHeader->CELHeight,
					      256);
}


unsigned long SizeCEL(int hFile, int HeaderFlag)
{
	if(HeaderFlag)
		return( CELFileSize );
	else
		return( (long)pCELHeader->CELWidth * (long)pCELHeader->CELHeight );
}


void EndCEL()
{
	free (pCELHeader);
}


/**** PCX file routines ****/


struct  PCXHeader
	{
	char    PCXManufacturer;
	char    PCXVersion;
	char    PCXEncoding;
	char    PCXBitsPerPixel;
	short   PCX_XMin;
	short   PCX_YMin;
	short   PCX_XMax;
	short   PCX_YMax;
	short   PCX_HRes;
	short   PCX_VRes;
	char    PCXPalette[3][16];
	char    PCXReserved;
	char    PCXPlanes;
	short   PCXBytesPerLine;
	short   PCXPaletteInfo;
	char    filler[58];
	};


struct PCXHeader *PCXHeadBuffer;

char PCXVGAPaletteFlag;


int IsPCX(int hFile)
{
		/* The PCX file check is is done by reading             */
		/* in the header of the file and checking for the       */
		/* PCXManufacturer byte.                                */

	PCXHeadBuffer = AllocateMemory(sizeof (struct PCXHeader) );

	read(hFile, (char *)PCXHeadBuffer, sizeof (struct PCXHeader) );

	if( PCXHeadBuffer->PCXManufacturer == 0x0A ) {

			/* If the file is a possible PCX, then read the */
			/* VGA palette flag byte at the end of the file.*/

		lseek(hFile, -769L, SEEK_END);
		read(hFile, &PCXVGAPaletteFlag, 1);
		if(PCXVGAPaletteFlag == 0x0C) {
			PaletteLevel = 1;
			PaletteSize = 768;
			read(hFile, Palette, 768);
		}
		lseek(hFile, 128L, SEEK_SET);
		return 1;
	} else
		return 0;
}


unsigned long SizePCX(int hFile, int HeaderFlag)
{
	unsigned long PCXFileSize;

		/* The size of a headerless PCX file is the actual file */
		/* file size - the size of the 128 byte header - the 769*/
		/* bytes for a VGA palette and flag if it is present.   */

	PCXFileSize = filelength(hFile);

	if(HeaderFlag)
		return( PCXFileSize );
	else
		return( PCXFileSize - 128 - ( (PCXVGAPaletteFlag == 0xc0)*769) );
}


int DecodePCX(int hInFile, int hOutFile)
{
	int Height, Width, Count, Data, BPL;
	char *pDecodeBuffer, *pBuffer;


	puts("Decoding PCX file.");

	BPL = PCXHeadBuffer->PCXBytesPerLine * PCXHeadBuffer->PCXPlanes;
	Height = (PCXHeadBuffer->PCX_YMax - PCXHeadBuffer->PCX_YMin) + 1;

	pDecodeBuffer = AllocateMemory(BPL);

	InitialiseReadBuffer(hInFile);

	Width = BPL;
	pBuffer = pDecodeBuffer;

	while(Height) {
		Count = 1;
		Data = get_byte();

		if ((Data & 0xc0) == 0xc0) {
			Count = Data & 0x3f;
			Data = get_byte();
		}

		while(Count) {
			*pBuffer = (char)Data;
			pBuffer++;
			Count--;
			Width--;

			if(Width == 0) {
				WriteFile(hOutFile, pDecodeBuffer, BPL);
				Height--;
				Width = BPL;
				pBuffer = pDecodeBuffer;
			}
		}
	}
	free(pDecodeBuffer);
	return 0;
}


void    HeaderPCX()
{
	/* This procedure will print the details in a PCXHeader.        */

	printf("Dimensions     : %ux%ux%u\n", (PCXHeadBuffer->PCX_XMax - PCXHeadBuffer->PCX_XMin+1)
					, (PCXHeadBuffer->PCX_YMax - PCXHeadBuffer->PCX_YMin+1)
					, (0x0001 << (PCXHeadBuffer->PCXBitsPerPixel)));
	printf("Bits per Pixel : %u\n", (short)PCXHeadBuffer->PCXBitsPerPixel);
	printf("Bit Planes     : %u\n", (short)PCXHeadBuffer->PCXPlanes);
	printf("Bytes per Line : %u\n", (short)PCXHeadBuffer->PCXBytesPerLine);
}


void EndPCX()
{
	free(PCXHeadBuffer);
}


/**** GIF file routines ****/


struct GIFHeader {
	char    GIFSignature[6];
	short   GIFWidth;
	short   GIFHeight;
	char    GIFFlags;
	char    GIFBackGround;
	char    GIF0Byte;
	};

struct GIFImageDesc {
	short   GIFX;
	short   GIFY;
	short   GIFWidth;
	short   GIFHeight;
	char    GIFImageFlags;
	};



void    ReadGIFExtension(void);
unsigned long ReadGIFDataBlock(int hFile);
int     out_line(char *pixels, int linelen);


struct GIFHeader *GIFHeadBuffer;


unsigned long GIFRasterDataPos;
unsigned long GIFRasterDataSize;

struct GIFImageDesc ImageDesc;
int bad_code_count;
int hDecodeGIFFile;



int IsGIF(int hFile)
{
	unsigned char    Data;
	int     FoundRasterData;
	int     QuitLoop;
	int ncolours;


	GIFHeadBuffer = AllocateMemory(sizeof (struct GIFHeader) );

	read(hFile, (char *)GIFHeadBuffer, 13 );

	if( (strncmp(GIFHeadBuffer->GIFSignature, "GIF", 3)) ||
	    (GIFHeadBuffer->GIF0Byte != 0) )
		return 0;

		/* Read the global colour palette, if present.          */
	if(GIFHeadBuffer->GIFFlags & 0x80) {
		PaletteLevel = 1;
		PaletteSize = (3 * (0x0001 << ((GIFHeadBuffer->GIFFlags & 0x07)+1)));

		read(hFile, Palette, PaletteSize);
		}


		/* Now loop through each block in the GIF file.         */

	FoundRasterData = 0;
	QuitLoop = 0;

	while(!QuitLoop) {

		read(hFile, &Data, 1);

		switch(Data) {
		case ',':
			if(FoundRasterData == 0) {
				read(hFile, (char *)&ImageDesc, 9 );

				if( ImageDesc.GIFImageFlags & 0x80) {

					ncolours = 1 << ((ImageDesc.GIFImageFlags & 0x7) + 1);
					if(InformationFlag)
						printf("Local Colour Map present: %d colours, Ignored.\n", ncolours);

					lseek(hFile, (long)3*ncolours, SEEK_SET);
				}

				GIFRasterDataPos = tell(hFile);
				GIFRasterDataSize = ReadGIFDataBlock(hFile);

				}
			else {
				struct GIFImageDesc MultiImage;

				puts("Warning: GIF contains multiple images");

				read(hFile, (char *)&MultiImage, 9 );

				if( MultiImage.GIFImageFlags & 0x80) {

					ncolours = 1 << ((MultiImage.GIFImageFlags & 0x7) + 1);
					lseek(hFile, (long)3*ncolours, SEEK_SET);
				}

				ReadGIFDataBlock(hFile);
			}

			FoundRasterData++;
			break;

		case ';':
			if(FoundRasterData == 0)
				Error(E_FATAL, ES_GIF_NO_DATA, NULL, 1);

			QuitLoop = 1;
			break;

		case '!':
			ReadGIFDataBlock(hFile);

			puts("Warning: GIF extension block found, ignored.");
			break;
		default:
			break;
		}
	}

	lseek(hFile, GIFRasterDataPos, SEEK_SET);

	return 1;
}


unsigned long ReadGIFDataBlock(int hFile)
{
	unsigned long BlockSize=0;
	unsigned char Data;

	read(hFile, &Data, 1);

	do {
		read(hFile, (char *)&Data, 1 );
		BlockSize += Data;
		lseek(hFile, (unsigned long)Data, SEEK_CUR);
	} while(Data != 0);

	return BlockSize;
}


unsigned long SizeGIF(int hFile, int HeaderFlag)
{
	if(HeaderFlag)
		return( filelength(hFile) );
	else
		return( GIFRasterDataSize );

}


int DecodeGIF(int hInFile, int hOutFile)
{
	puts("Decoding GIF file.");

	hDecodeGIFFile = hOutFile;

	InitialiseReadBuffer(hInFile);

	switch( decoder(ImageDesc.GIFWidth) ) {

		case(OUT_OF_MEMORY):
			Error(E_FATAL, ES_GIF_NO_MEMORY, NULL, 1);
		case(BAD_CODE_SIZE):
			Error(E_FATAL, ES_GIF_BAD_CODE_SIZE, NULL, 1);
		case(READ_ERROR):
			Error(E_FATAL, ES_GIF_READ_ERROR, NULL, 1);
		case(WRITE_ERROR):
			Error(E_FATAL, ES_GIF_WRITE_ERROR, NULL, 1);
		case(OPEN_ERROR):
			Error(E_FATAL, ES_GIF_OPEN_ERROR, NULL, 1);
		case(CREATE_ERROR):
			Error(E_FATAL, ES_GIF_CREATE_ERROR , NULL, 1);
	}

	if (bad_code_count)
		puts("Warning: GIF file may be corrupt");

	return 0;
}


int out_line(char *pixels, int linelen)
{
	WriteFile(hDecodeGIFFile, pixels, linelen);

	return 0;
}



void EndGIF()
{
	free(GIFHeadBuffer);
}


void    HeaderGIF()
{
	printf("GIF Version    : %6.6s\n", GIFHeadBuffer->GIFSignature);
	printf("Dimensions     : %ux%ux%u %s\n", ImageDesc.GIFWidth,
					      ImageDesc.GIFHeight,
					      (0x0001 << ((GIFHeadBuffer->GIFFlags & 0x07)+1)),
					      (ImageDesc.GIFImageFlags & 0x40 ? "interleaved ":"") );

	if(GIFHeadBuffer->GIFFlags & 0x80)
		puts("Global colour map is present.");

	if( ImageDesc.GIFImageFlags & 0x80 )
		puts("Local colour map is present.");
}



/**** LBM file routines ****/


struct BMHD {
	short   LBMWidth;
	short   LBMHeight;
	short   LBMX;
	short   LBMY;
	char    LBMPlanes;
	char    LBMMasking;
	char    LBMCompression;
	char    LBMPad;
	short   LBMTransparentColour;
	char    LBMXAspect;
	char    LBMYAspect;
	short   LBMPageWidth;
	short   LBMPageHeight;
	};

			/* LBMMasking constants */

#define mskNone                 0
#define mskHasMask              1
#define mskHasTransparentColor  2
#define mskLasso                3


void DoChunks(int hFile);
void UnPackBits(int hInfile, int hOutFile, int Width, int Height);


int seen_ILBM;
int seen_PBM;
int seen_bmhd;
int seen_cmap;
int seen_crng;
int seen_body;
unsigned long LBMPos;
unsigned long FORMSize;
unsigned long BODYPos;
unsigned long BODYSize;

struct BMHD *LBM_BMHD_Buffer;


int IsLBM(int hFile)
{
	char FORMId[4];

	read(hFile, FORMId, 4);

	if( strncmp(FORMId, "FORM", 4) )
		return 0;

	read(hFile, (char *)&FORMSize, sizeof(unsigned long));

	ByteSwapDword(&FORMSize);

	LBMPos = 0;

	seen_bmhd = 0;
	seen_body = 0;
	seen_cmap = 0;
	seen_crng = 0;
	DoChunks(hFile);

	if( !seen_bmhd || !seen_body )
		Error(E_FATAL, ES_LBM_CHUNKS_MISSING, NULL, 1);

	lseek(hFile, BODYPos, SEEK_SET);

	return 1;
}


unsigned long SizeLBM(int hFile, int HeaderFlag)
{
	if(HeaderFlag)
		return( filelength(hFile) );
	else
		return BODYSize;
}


void    HeaderLBM(void)
{
	printf("Dimensions     : %ux%ux%u\n", LBM_BMHD_Buffer->LBMWidth,
					LBM_BMHD_Buffer->LBMHeight,
					(0x0001 << LBM_BMHD_Buffer->LBMPlanes) );
	printf("Bit Planes     : %u\n",LBM_BMHD_Buffer->LBMPlanes);
}


int DecodeLBM(int hInFile, int hOutFile)
{

	puts("Decoding LBM file.");

	if(LBM_BMHD_Buffer->LBMMasking != mskNone) {
		puts("Warning: LBM file has mask defined");
	}

	if (LBM_BMHD_Buffer->LBMCompression != 0 &&
	   LBM_BMHD_Buffer->LBMCompression != 1) {
		Error(E_FATAL, ES_LBM_UNKNOWN_COMPRESS, NULL, 1);
	}

	if( (LBM_BMHD_Buffer->LBMPlanes == 8) && (!seen_PBM) ) {
		puts("Warning: LBM file is 256 ILBM planar");
	}

	if( LBM_BMHD_Buffer->LBMCompression == 1 )
		UnPackBits(hInFile, hOutFile, LBM_BMHD_Buffer->LBMWidth/8*LBM_BMHD_Buffer->LBMPlanes, LBM_BMHD_Buffer->LBMHeight);
	else
		CopyLines(hInFile, hOutFile, LBM_BMHD_Buffer->LBMWidth/8*LBM_BMHD_Buffer->LBMPlanes, LBM_BMHD_Buffer->LBMHeight);

	return 0;
}


void UnPackBits(int hInfile, int hOutFile, int Width, int Height)
{
	char *pDecodeBuffer, *pBuffer;
	int Data, Count, RepeatCount;
	int SaveWidth;

	InitialiseReadBuffer(hInfile);

	pDecodeBuffer = AllocateMemory(Width);
	pBuffer = pDecodeBuffer;
	SaveWidth = Width;

	while(Height) {
		RepeatCount = get_byte();
		Count = 1;

		if(RepeatCount & 0x0080) {
			Count = (-(signed char)RepeatCount)+1;
			RepeatCount = 0;
		}

		RepeatCount++;

		while(RepeatCount) {
			Data = get_byte();

			while(Count) {
				*pBuffer = (char)Data;

				pBuffer ++;
				Count--;
				Width--;
				if(Width == 0) {
					Width = SaveWidth;
					WriteFile(hOutFile, pDecodeBuffer, Width);
					pBuffer = pDecodeBuffer;
					Height--;
				}
			}
		Count = 1;
		RepeatCount--;
		}
	}

	free(pDecodeBuffer);
}


void CopyLines(int hInFile, int hOutFile, int Width, int Height)
{
	char *pBuffer;

	pBuffer = AllocateMemory(Width);

	while(Height) {
		read(hInFile, pBuffer, Width);
		WriteFile(hOutFile, pBuffer, Width);

		Height--;
	}

	free(pBuffer);
}


void EndLBM(void)
{
	free(LBM_BMHD_Buffer);
}


void DoChunks(int hFile)
{
	unsigned long ChunkSize;
	unsigned long t, newpos;
	char ChunkID[4];

	while(FORMSize > LBMPos) {

	LBMPos += read(hFile, ChunkID, 4);

	if( !strncmp(ChunkID, "ILBM", 4) ) {
		seen_ILBM = 1;

	} else
	if( !strncmp(ChunkID, "PBM ", 4) ) {
		seen_PBM = 1;

	} else {                        /* normal chunks */

	LBMPos += read(hFile, (char *)&ChunkSize, sizeof(unsigned long));

	ByteSwapDword( &ChunkSize );

	ChunkSize = (ChunkSize+1) & -2;

	if( !strncmp(ChunkID, "BMHD", 4) ) {
		seen_bmhd = 1;
		if (ChunkSize != sizeof(struct BMHD) )
			Error(E_FATAL, ES_LBM_BAD_BMHD, NULL, 1);

		LBM_BMHD_Buffer = AllocateMemory(sizeof (struct BMHD));

		LBMPos += read(hFile, (char *)LBM_BMHD_Buffer, sizeof(struct BMHD));

		ByteSwapWord( &LBM_BMHD_Buffer->LBMWidth );
		ByteSwapWord( &LBM_BMHD_Buffer->LBMHeight );
		ByteSwapWord( &LBM_BMHD_Buffer->LBMX );
		ByteSwapWord( &LBM_BMHD_Buffer->LBMY );
		ByteSwapWord( &LBM_BMHD_Buffer->LBMTransparentColour );
		ByteSwapWord( &LBM_BMHD_Buffer->LBMPageWidth );
		ByteSwapWord( &LBM_BMHD_Buffer->LBMPageHeight );

	} else
	if( !strncmp(ChunkID, "CMAP", 4) ) {
		seen_cmap=1;

		if (ChunkSize % 6)
			Error(E_FATAL, ES_LBM_BAD_CMAP, NULL, 1);

		PaletteLevel = 1;
		PaletteSize = (unsigned short)ChunkSize;
		read(hFile, Palette, PaletteSize);
		LBMPos += ChunkSize;
	} else
	if(!strncmp(ChunkID, "CRNG", 4) ) {
		if(InformationFlag && !seen_crng) {
			puts("Warning: File contains colour range, ignored");
		}

		seen_crng = 1;
		lseek(hFile, ChunkSize, SEEK_CUR);
		LBMPos += ChunkSize;

	} else
	if(!strncmp(ChunkID, "TINY", 4) ) {
		if(InformationFlag) {
			puts("Warning: File contains preview image, ignored");
		}

		lseek(hFile, ChunkSize, SEEK_CUR);
		LBMPos += ChunkSize;
	} else
	if( !strncmp(ChunkID, "BODY", 4) ) {
		seen_body=1;
		BODYPos = tell(hFile);
		BODYSize = ChunkSize;

		lseek(hFile, ChunkSize, SEEK_CUR);
		LBMPos += ChunkSize;

	} else {
		printf("Warning: Unknown chunk type : %4.4s\n", ChunkID);
		lseek(hFile, ChunkSize, SEEK_CUR);
		LBMPos += ChunkSize;
		}
	}
	}
}


/**** PAL routines ****/


void    ConvertPalette(unsigned char *PaletteInfo, int Count);


int IsPAL(int hFile)
{
	PaletteLevel = 0;

	if ( IsPCX(hFile) ) {
		if(PCXVGAPaletteFlag == 0x0C)
			return 1;
		else {
			puts("PCX doesn't contain VGA colour palette");
			return 0;
		}
	}

	EndPCX();
	lseek(hFile, 0L, SEEK_SET);

	if( IsGIF(hFile) ) {
		if ( GIFHeadBuffer->GIFFlags & 0x80 )
			return 1;
		else {
			puts("GIF doesn't contain global colour palette");
			return 0;
		}
	}

	EndGIF();
	lseek(hFile, 0L, SEEK_SET);

	if( IsLBM(hFile) ) {
		if(seen_cmap)
			return 1;
		else {
			puts("LBM doesn't contain colour palette");
			return( 0 );
		}
	}
	EndLBM();

	lseek(hFile, 0L, SEEK_SET);

	if ( IsCEL(hFile) )
		return 1;

	EndCEL();

	return 0;
}


unsigned long SizePAL(int hFile, int HeaderFlag)
{
	return( PaletteSize );
}


int DecodePAL(int hInFile, int hOutFile)
{
	if(PaletteLevel) {

		if(InformationFlag)
			puts("Converting palette to 64 level VGA palette.");

		ConvertPalette(Palette, PaletteSize);

		WriteFile(hOutFile, Palette, PaletteSize);
	}
	return 0;
}


void    ConvertPalette(unsigned char *PaletteInfo, int Count)
{
	/* This procedure will scale down a VGA palette from 0-255      */
	/* to a 0-63 range.                                             */


	while(Count) {
		*PaletteInfo >>= 2;
		PaletteInfo++;

		Count--;
	}
}


void EndPAL()
{
}



