/* aout4lb.c	-- fiddle with symbols in an a.out format file 
** Copyright (C) 1994	Norman D. Culver	All rights reserved.
**
** COMPILE: (using djcc)
**		gcc -O2 -o aout4lb aout4lb.c
**		aout2exe aout4lb
*/

#include <limits.h>
#include <sys/types.h>
#include <stdlib.h>
#include <stdarg.h>
#include <memory.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>


/* USEFUL DEFINITIONS */
#ifndef __GNUC__
typedef unsigned int u_int;
typedef unsigned char u_char;
typedef unsigned short u_short;
typedef unsigned long u_long;
#endif

typedef union {
  u_short val;
  u_char b[2];
} SVAL;

typedef union {
  u_long a0;
  void *a1;
  struct {
    SVAL offset;
    SVAL seg_addr;
  }a2;
} ADDR;

#define round_up(size,amt) \
((size&(amt-1)) ? size+(amt-(size&(amt-1))) : size)


/* DEFINITIONS for the djcc form of an a.out file */

struct exec
{
  u_long a_info;      /* Use macros N_MAGIC, etc for access */
  u_long a_text;      /* length of text, in bytes */
  u_long a_data;      /* length of data, in bytes */
  u_long a_bss;       /* length of uninitialized data area for file, in bytes */
  u_long a_syms;      /* length of symbol table data in file, in bytes */
  u_long a_entry;     /* start address */
  u_long a_trsize;    /* length of relocation info for text, in bytes */
  u_long a_drsize;    /* length of relocation info for data, in bytes */
};


/* these go in the N_MACHTYPE field  ( djcc sets machine_type to 0 ) */
enum machine_type {

  M_OLDSUN2 = 0,
  M_68010 = 1,
  M_68020 = 2,
  M_SPARC = 3,

  /* skip a bunch so we don't run into any of sun's numbers */
  M_386 = 100,
};

#define N_MAGIC(exec) ((exec).a_info & 0xffff)
#define N_MACHTYPE(exec) ((enum machine_type)(((exec).a_info >> 16) & 0xff))
#define N_FLAGS(exec) (((exec).a_info >> 24) & 0xff)
#define N_SET_INFO(exec, magic, type, flags) \
((exec).a_info = ((magic) & 0xffff) \
         | (((long)(type) & 0xff) << 16) \
         | (((long)(flags) & 0xff) << 24))
#define N_SET_MAGIC(exec, magic) \
((exec).a_info = (((exec).a_info & 0xffff0000) | ((magic) & 0xffff)))

#define N_SET_MACHTYPE(exec, machtype) \
((exec).a_info = \
         ((exec).a_info&0xff00ffff) | ((((long)(machtype))&0xff) << 16))

#define N_SET_FLAGS(exec, flags) \
((exec).a_info = \
         ((exec).a_info&0x00ffffff) | (((long)(flags) & 0xff) << 24))

/* Code indicating object file or impure executable.  */
#define OMAGIC 0407
/* Code indicating pure executable.  */
#define NMAGIC 0410
/* Code indicating demand-paged executable.  */
#define ZMAGIC 0413

#define N_BADMAG(x)                                     \
(N_MAGIC(x) != OMAGIC && N_MAGIC(x) != NMAGIC          \
  && N_MAGIC(x) != ZMAGIC)

#define _N_BADMAG(x)                                    \
(N_MAGIC(x) != OMAGIC && N_MAGIC(x) != NMAGIC          \
  && N_MAGIC(x) != ZMAGIC)

#define _N_HDROFF(x) 0

#define N_TXTOFF(x) \
(N_MAGIC(x) == ZMAGIC ? _N_HDROFF((x)) + sizeof (struct exec) : sizeof (struct exec))

#define N_DATOFF(x) (N_TXTOFF(x) + (x).a_text)

#define N_TRELOFF(x) (N_DATOFF(x) + (x).a_data)

#define N_DRELOFF(x) (N_TRELOFF(x) + (x).a_trsize)

#define N_SYMOFF(x) (N_DRELOFF(x) + (x).a_drsize)

#define N_STROFF(x) (N_SYMOFF(x) + (x).a_syms)

/* Address of text segment in memory after it is loaded.  */

#define N_TXTADDR(x) (sizeof(struct exec)+4096)

/* Address of data segment in memory after it is loaded.
   Note that it is up to you to define SEGMENT_SIZE
   on machines not listed here.  */

#define SEGMENT_SIZE 0x400000

#define N_DATADDR(x) \
(N_MAGIC(x)==OMAGIC? (N_TXTADDR(x)+(x).a_text) \
     : (SEGMENT_SIZE + ((N_TXTADDR(x)+(x).a_text-1) & ~(SEGMENT_SIZE-1))))


/* Address of bss segment in memory after it is loaded.  */
#define N_BSSADDR(x) (N_DATADDR(x) + (x).a_data)


typedef struct nlist {
  union {
    char *n_name;
    struct nlist *n_next;
    long n_strx;
  } n_un;
  u_char n_type;
  char n_other;
  short n_desc;
  u_long n_value;
} NLIST;


#define N_UNDF	0x00
#define N_EXT	0x01
#define N_ABS	0x02
#define N_TEXT	0x04
#define N_DATA	0x06
#define N_BSS	0x08
#define N_COMM	0x12
#define N_FN	0x0f

#define N_TYPE 0x1e
#define N_STAB 0340

/* The following type indicates the definition of a symbol as being
   an indirect reference to another symbol.  The other symbol
   appears as an undefined reference, immediately following this symbol.

   Indirection is asymmetrical.  The other symbol's value will be used
   to satisfy requests for the indirect symbol, but not vice versa.
   If the other symbol does not have a definition, libraries will
   be searched to find a definition.  */

#define N_INDR 0x0a

/* The following symbols refer to set elements.
   All the N_SET[ATDB] symbols with the same name form one set.
   Space is allocated for the set in the text section, and each set
   element's value is stored into one word of the space.
   The first word of the space is the length of the set (number of elements).

   The address of the set is made into an N_SETV symbol
   whose name is the same as the name of the set.
   This symbol acts like a N_DATA global symbol
   in that it can satisfy undefined external references.  */

/* These appear as input to LD, in a .o file.  */
#define N_SETA  0x14            /* Absolute set element symbol */
#define N_SETT  0x16            /* Text set element symbol */
#define N_SETD  0x18            /* Data set element symbol */
#define N_SETB  0x1A            /* Bss set element symbol */

/* This is output from LD.  */
#define N_SETV  0x1C            /* Pointer to set vector in data area.  */

/* Classify stabs symbols */

#define sym_in_text_section(sym) \
(((sym)->n_type  & (N_ABS | N_TEXT | N_DATA | N_BSS))== N_TEXT)

#define sym_in_data_section(sym) \
(((sym)->n_type  & (N_ABS | N_TEXT | N_DATA | N_BSS))== N_DATA)

#define sym_in_bss_section(sym) \
(((sym)->n_type  & (N_ABS | N_TEXT | N_DATA | N_BSS))== N_BSS)

/* Symbol is undefined if type is N_UNDF|N_EXT and if it has
  zero in the "value" field.  Nonzeroes there are fortrancommon
  symbols.  */
#define sym_is_undefined(sym) \
((sym)->n_type == (N_EXT) && (sym)->n_value == 0)

/* Symbol is a global definition if N_EXT is on and if it has
  a nonzero type field.  */
#define sym_is_global_defn(sym) \
(((sym)->n_type & N_EXT) && (sym)->n_type & N_TYPE)

/* Symbol is debugger info if any bits outside N_TYPE or N_EXT
  are on.  */
#define sym_is_debugger_info(sym) \
((sym)->n_type & ~(N_EXT | N_TYPE))

#define sym_is_fortrancommon(sym)       \
(((sym)->n_type == (N_EXT)) && (sym)->n_value != 0)

/* Symbol is absolute if it has N_ABS set */
#define sym_is_absolute(sym) \
(((sym)->n_type  & N_TYPE)== N_ABS)


#define sym_is_indirect(sym) \
(((sym)->n_type & N_TYPE)== N_INDR)


/* This structure describes a single relocation to be performed.
   The text-relocation section of the file is a vector of these structures,
   all of which apply to the text section.
   Likewise, the data-relocation section applies to the data section.  */

typedef struct relinfo
{
  /* Address (within segment) to be relocated.  */
  long r_address;

  /* The meaning of r_symbolnum depends on r_extern.  */
u_long r_symbolnum:
  24;

  /* Nonzero means value is a pc-relative offset
     and it should be relocated for changes in its own address
     as well as for changes in the symbol or section specified.  */
u_long r_pcrel:
  1;

  /* Length (as exponent of 2) of the field to be relocated.
     Thus, a value of 2 indicates 1<<2 bytes.  */
u_long r_length:
  2;

  /* 1 => relocate with value of symbol.
          r_symbolnum is the index of the symbol
          in file's the symbol table.
     0 => relocate with the address of a segment.
          r_symbolnum is N_TEXT, N_DATA, N_BSS or N_ABS
          (the N_EXT bit may be set also, but signifies nothing).  */
u_long r_extern:
  1;

  /* Four bits that aren't used, but when writing an object file
     it is desirable to clear them.  */
u_long r_pad:
  4;
} RELINFO;

typedef struct _exports {
	struct _exports *fptr;
	char *name;
	int len;
} EXPORTS;

typedef struct _renames {
	struct _renames *fptr;
	char *oldname;
	int oldlen;
	char *newname;
	int newlen;
} RENAMES;

/* LOCAL VARIABLES */
static char infile[128];
static char outfile[128];
static char lb4file[128];
static struct exec header;
static FILE *ifd, *lb4fd, *ofd;
static EXPORTS exports;
static RENAMES renames;
static EXPORTS *lexports;
static RENAMES *lrenames;
static int rename_diftot;
static long lowoffset;

/* INPUT DATA FROM THE .o FILE */

static NLIST *symtbl;
static char *strtbl;
static char *nstrtbl;
static long strsize;
static long nstrsize;
static int nsyms;
static char *text;
static RELINFO *text_relocs;
static int n_text_relocs;
static char *data;
static RELINFO *data_relocs;
static int n_data_relocs;


/* start of subroutines */

static void *
xmalloc (long n)
{
void *p;

  if(n > UINT_MAX)
  {
    puts("aout4lb: malloc larger than UINT_MAX");
    exit(1);
  }
  p = calloc (1,(size_t)n);
  if (n > 0 && p == 0)
  {
    puts ("aout4lb: virtual memory exhausted");
    exit (1);
  }
  return p;
}
static int
isglobal(char *str)
{
EXPORTS *l = &exports;

	while(l->fptr)
	{
		l = l->fptr;
		if(!strcmp(str, l->name))
			return 1;
	}
	return 0;
}
static RENAMES *
isrenamed(char *str)
{
RENAMES *r = &renames;

	while(r->fptr)
	{
		r = r->fptr;
		if(!strcmp(str, r->oldname))
			return r;
	}
	return NULL;
}
static void
get_rid_of_fortrancommon(NLIST *sp, int symidx)
{/* move fortrancommon symbols to the bss segment */
u_long bssbase = N_BSSADDR(header);
u_long bssoffset = round_up(header.a_bss, sizeof(int));
u_long symbsize = sp->n_value;
u_long bssaddr = bssbase + bssoffset;
RELINFO *r;
int i;
void *ldata;

	sp->n_value = bssaddr;
	sp->n_type = N_BSS;
	header.a_bss = bssoffset + symbsize;

	for(i = 0, r=text_relocs; i < n_text_relocs; ++i, ++r)
	{
		if(r->r_extern && r->r_symbolnum == symidx)
		{
			ldata = &text[r->r_address];
			r->r_extern = 0;
			r->r_symbolnum = N_BSS;
			if(r->r_length == 2)
				*((long *)ldata) += bssaddr;
			else if(r->r_length == 0)
				*((char *)ldata) += (u_char)bssaddr;
			else if(r->r_length == 1)
				*((short *)ldata) += (u_short)bssaddr;
			else
				*((long long *)ldata) += bssaddr;
		}
	}
	for(i = 0, r=data_relocs; i < n_data_relocs; ++i, ++r)
	{
		if(r->r_extern && r->r_symbolnum == symidx)
		{
			ldata = &data[r->r_address];
			r->r_extern = 0;
			r->r_symbolnum = N_BSS;
			if(r->r_length == 2)
				*((long *)ldata) += bssaddr;
			else if(r->r_length == 0)
				*((char *)ldata) += (u_char)bssaddr;
			else if(r->r_length == 1)
				*((short *)ldata) += (u_short)bssaddr;
			else
				*((long long *)ldata) += bssaddr;
		}
	}
}
static void
adjust_symbol(long oldoffset, long newoffset, int global)
{
NLIST *sp;
int i;

	for(i = 0, sp = symtbl; i < nsyms; i++, sp++)
	{
		if(sp->n_un.n_strx == oldoffset)
		{/* newoffset is set negative to avoid reuse */
			sp->n_un.n_strx = -newoffset;
			if(global)
				sp->n_type |= N_EXT;
			else if(sym_is_fortrancommon(sp))
				get_rid_of_fortrancommon(sp, i);
			else if(!sym_is_undefined(sp))
				sp->n_type &= ~N_EXT;
			return;
		}
	}
}
static void
adjust_strings ()
{/* HERE'S THE BEEF -- WRITE A NEW STRING TABLE AND ADJUST SYMTAB */
long istroff = lowoffset;
long ostroff = lowoffset;
char *ip, *ep, *op;
NLIST *sp;
int i;

	nstrsize = strsize + rename_diftot;
	nstrtbl = xmalloc(nstrsize);
	ip = strtbl+lowoffset;
	op = nstrtbl+lowoffset;
	ep = ip + strsize;

	while(ip < ep)
	{/* scan over each of the strings in the old string table */
	int global;
	int	istrlen = strlen(ip);
	int olen;
	RENAMES *r;
		if((r = isrenamed(ip)))
		{/* change the string on output */
			strcpy(op, r->newname);
			op += r->newlen+1;
			olen = r->newlen+1;
			global = isglobal(r->newname);
		}
		else
		{/* use the old string on output */
			strcpy(op, ip);
			op += istrlen+1;
			olen = istrlen+1;
			global = isglobal(ip);
		}
		adjust_symbol(istroff, ostroff, global);
		ip += istrlen+1;
		istroff += istrlen+1;
		ostroff += olen;
	}
	/* reset the negative values for offsets */
	for(i = 0, sp = symtbl; i < nsyms; i++, sp++)
	  if(sp->n_un.n_strx < 0)
		sp->n_un.n_strx = -sp->n_un.n_strx;
}
static int
read_symbols ()
{
int i;
NLIST *sp;

  nsyms = header.a_syms / sizeof (NLIST);
  if (nsyms == 0)
    return 1;

  symtbl = (NLIST *)xmalloc (header.a_syms);

  fseek (ifd, N_SYMOFF (header), 0);
  fread ((char *)symtbl, sizeof(NLIST), nsyms, ifd);

  fseek (ifd, N_STROFF(header), 0);
  fread ((char *)&strsize, sizeof strsize, 1, ifd);

  strtbl = xmalloc (strsize);
  fseek (ifd, N_STROFF (header), 0);
  fread ((char *)strtbl, (int)strsize, 1, ifd);

  lowoffset = 2000000;
  for (i = 0, sp = symtbl; i < nsyms; i++, sp++)
  {
    if (sp->n_un.n_strx != 0)
		if(sp->n_un.n_strx < lowoffset)
			lowoffset = sp->n_un.n_strx;
  }
  return 0;
}
static void
read_text_and_data()
{
  text = xmalloc(header.a_text);
  data = xmalloc(header.a_data);

  if(header.a_text > 0)
  {
    fseek(ifd, N_TXTOFF(header), 0);
    fread(text, (u_int)header.a_text, 1, ifd);
  }
  if(header.a_data > 0)
  {
    fseek(ifd, N_DATOFF(header), 0);
    fread(data, (u_int)header.a_data, 1, ifd);
  }
}
static int
relcomp(const void *a,const void *b)
{
  return (int)((long)((RELINFO*)a)->r_address-(long)((RELINFO*)b)->r_address);
}
static void
read_relocs()
{
  n_text_relocs = header.a_trsize / sizeof (RELINFO); 
  n_data_relocs = header.a_drsize / sizeof (RELINFO);

  if(n_text_relocs > 0)
  {
    text_relocs = xmalloc(header.a_trsize); 
    fseek(ifd, N_TRELOFF(header), 0);
    fread((char *)text_relocs, (u_int)header.a_trsize, 1, ifd);
    qsort (text_relocs, n_text_relocs, sizeof (RELINFO), relcomp);
  }

  if(n_data_relocs > 0)
  {
    data_relocs = xmalloc(header.a_drsize);	
    fseek(ifd, N_DRELOFF(header), 0);
    fread((char *)data_relocs, (u_int)header.a_drsize, 1, ifd);
    qsort (data_relocs, n_data_relocs, sizeof (RELINFO),relcomp);
  }
}



static void
usage()
{
  puts("Usage: aout4lb infile");
  puts("  infile must be in .o format.");
  puts("  outfile will be the suffix .olb");
  puts("  the specification file should have the suffix .4lb");
  exit(0);
}
static void
lose_backslashes(char *p)
{
  while(*p)
  {
    if(*p == '\\')
      *p = '/';
    ++p;
  }
}


static int
proc_infile(char *name)
{
  ifd = fopen(name, "r+b");
  if(!ifd) {
  	printf("4lb: Cannot open input file %s\n", infile);
	exit(1);
  }
  if(fread((char *)&header, sizeof(struct exec), 1, ifd) != 1)
  {
    printf("4lb: cannot read %s\n", name);
    return 0;
  }
  if(	N_MAGIC(header) != OMAGIC 
  	|| (N_MACHTYPE(header) != 0 && N_MACHTYPE(header) != M_386))
  {
    printf("4lb: incorrect filetype: %s\n", name);
    return 0;
  }

  read_symbols();
  read_text_and_data();
  read_relocs();

  return 1;
}
static void
fix_suffix(char *str, char *suf)
{
char *cp;
	if((cp = strrchr(str, '.')) != NULL)
		 strcpy(cp, suf);
	else strcat(str, suf);
}
static char *
find_sym(char *cp)
{
	while(*cp && (*cp == ':' || isspace(*cp)))
		++cp;

	if(*cp) {
	char *np=cp;	
		
		while(*np && *np != ':' && !isspace(*np))
			++np;
		if(*np)
			*np = '\0';
		return cp;
	}
	return NULL;
}
static int
sym_exists(const char *cp, int len)
{
char *sp = strtbl;
char *ep = strtbl + strsize;

	while(sp < ep)
	{
	int splen = strlen(sp);
		if(splen == len)
			if(!strcmp(sp,cp))
				return 1;
		sp += splen+1;
	}
	return 0;
}
static void
proc_4lb()
{/* Process the specification file */
char buf[200];
char *cp;

	lexports = &exports;
	lrenames = &renames;
	while(fgets(buf, 190, lb4fd))
	{
		if((cp = strchr(buf, ':')) == NULL)
			continue;
		if(!strncmp(buf, "export", 6))
		{
			if((cp = find_sym(cp)) != NULL)
			{
			int len;
			char *strloc;
				len = strlen(cp);
				if(sym_exists(cp, len))
				{
					strloc = xmalloc(len+1);
					strcpy(strloc, cp);
					lexports->fptr = xmalloc(sizeof(EXPORTS));
					lexports = lexports->fptr;
					lexports->name = strloc;
					lexports->len = len;
				}
			}
		}
		else if(!strncmp(buf, "rename", 6))
		{
			if((cp = find_sym(cp)) != NULL)
			{
			int oldlen;
			char *oldstr;

				oldlen = strlen(cp);
				if(sym_exists(cp, oldlen))
				{
				  oldstr = cp;
				  cp += oldlen+1;
				  if((cp = find_sym(cp)) != NULL)
				  {
				  int newlen = strlen(cp);
				  char *newstr = cp;
				  char *oldloc = xmalloc(oldlen+1);
				  char *newloc = xmalloc(newlen+1);
					strcpy(oldloc, oldstr);
					strcpy(newloc, newstr);
					lrenames->fptr = xmalloc(sizeof(RENAMES));
					lrenames = lrenames->fptr;
					lrenames->oldname = oldloc;
					lrenames->oldlen = oldlen;
					lrenames->newname = newloc;
					lrenames->newlen = newlen; 
					rename_diftot += newlen - oldlen;
				  }
				}
			}
		}
	}
}
static void
proc_outfile(FILE *fd)
{
	/* rename symbols and set exported symbols to global */
	adjust_strings();

	fseek(fd, 0, 0);

	/* header */
	fwrite((char *)&header, sizeof(struct exec), 1, fd);

	/* text */
	if(header.a_text)
		fwrite(text, (u_int)header.a_text, 1, fd);

	/* data */
	if(header.a_data)
		fwrite(data, (u_int)header.a_data, 1, fd);

	/* text relocs */
	if(header.a_trsize)
		fwrite((char *)text_relocs, (u_int)header.a_trsize, 1, fd);

	/* data relocs */
	if(header.a_drsize);
		fwrite((char *)data_relocs, (u_int)header.a_drsize, 1, fd);

	/* symbol table */
	if(header.a_syms)
		fwrite((char *)symtbl, (u_int)header.a_syms, 1, fd);

	/* strings */
	fwrite((char *)&nstrsize, sizeof nstrsize, 1, fd);
	if(nstrsize)
	  fwrite((char *)nstrtbl+sizeof(nstrsize),nstrsize-sizeof(nstrsize),1,fd);
}
int
main(int argc, char **argv)
{
  if (argc != 2) usage();
  strcpy(infile,argv[1]);
  lose_backslashes(infile);
  strcpy(outfile,infile);
  strcpy(lb4file,infile);
  fix_suffix(outfile, ".olb");
  fix_suffix(lb4file, ".4lb");

  /* process the input file */
  if(proc_infile(infile))
  {
	  fclose(ifd);
	  /* process the specification file */
	  lb4fd = fopen(lb4file, "r"); 
	  if(lb4fd) {
	  	proc_4lb();
		fclose(lb4fd);
	  }
	  ofd = fopen(outfile, "wb");
	  proc_outfile(ofd);
	  fclose(ofd);
  }
  return 1;
}

