/* *******************************************************************

The Oracle SQL Class Library Version 1.0,
Author: Sergei Kuchin
Copyright (C) Mosakin Corp., 1995
This library is free software.  Permission to use, copy,
modify and redistribute the Oracle SQL class library for any
purpose is hereby granted without fee, provided that the
above copyright notice appear in all copies.


******************************************************************* */

#ifndef __CSQL_REQ_H
#define __CSQL_REQ_H

/* ***************** C-interface with Oracle **************** */

int sql_open(char* Connect);/* Connect to Oracle, e.g. "prime/prime" */
int sql_close(void);        /* Close connection                      */
void  sql_commit(void);     /* Commit transaction                    */
void  sql_roll_back(void);  /* Roll back transaction                 */
int   sql_exec(char* Stm,int ignore_error);
  /* Execute a SQL or PL/SQL statement having no parameters.

     Stm - the statement text

     ignore_error - ignore error flag. When the flag is set then any
                    error will be ignored.

     The function returns  0 on success  and 1 when any  errors occurs
     and the ingnore error  flag is set. When  the flag is not set and
     an error occurs, exception is raised and the user defined handler
     is invoked. See below how to set your own handler.

  */
 
 typedef void (*ERR_HANDLER)(char*);

 void sql_set_handler(ERR_HANDLER ahandler);/*  set user defined handler */
 ERR_HANDLER sql_get_handler();/* get pointer to the currentl handler */

/* User defined handlers must functions with the one parameter:

   void handler(char* err_msg);

When  an error occurs  the  handler is invoked.  The err_msg parameter
contains the  error message  recevied  from the  Oracle   server.  The
handler is invoked  and then  the program  execution is terminated  by
invoking exit(1). 

By default, no handler is defined and the sql_get_handler function
returns NULL.

*/

/*******************  SQL stream functions **********************/
/*

  The SQL  streams  are intended for  SQL or  PL/SQL statements having
  input and/or output bind variables.  Any statement can be treated as
  a  functional   element with  input/output  parameters.    There are
  functions to put objects to the stream, that is  to assign values to
  input variables. Also, there are  functions to get objects from  the
  stream, that is to get the values from the output variables.

  The SQL statement  is executed automatically  when all the values of
  its input  variables  are defined.   Right after the  execution  the
  corresponding stream is ready for reading. If the statement has no
  input variables then it is executed immediately.

*/

 int sql_stream_open(char* Stm,short BufSize);
  /* 
     Open the SQL stream and attach the internal buffer of the size
     BufSize. The function returns the stream id.

     Example 1
     =========

     int u=sql_stream_open(
	     "insert into state values(:state<char[33]>,:state_code<int>)",
	     20
	   );

     Example 2
     =========

     int u=sql_stream_open(
	     "select * from state where state like :state<char[33]>",
	     20
          );

The following data types for bind variables are available:

    int
    unsigned
    short
    long           -- (integer)
    float
    double
    char[<length>]


For PL/SQL blocks having both input and output variables, the output
variables must be qualified, e.g.:

    begin
      :A<int,out> := :B<int>;
    end;
	   
*/

 void sql_stream_close(int sd);/* close stream */

 int  sql_eof(int sd);
  /* test if all the output data was fetched */

 int  sql_is_null(int sd);
  /* test if Oracle NULL was fetched in the last operation */

 void sql_flush(int sd);
  /* 
     "flush" the  stream if it has input  variables. The SQL statement
     is executed  even    if the buffer  of   input   varables  is not
     filled. By  default,   after the statement  is executed,  current
     transaction is committed.
  */

 void sql_set_commit(int sd,int commit_when_flush);
  /* 
    Inform the flush  function not to commit transaction
    after flushing the input data. It is your responsibility to commit
    or roll back transactions.
  */

/* Put objects to the stream */

 void sql_putc(int sd,char c);
 void sql_putuc(int sd,unsigned char c);
 void sql_puts(int sd,const char* s);
 void sql_putus(int sd,const unsigned char* s);
 void sql_puti(int sd,int n);
 void sql_putu(int sd,unsigned u);
 void sql_putsh(int sd,short sh);
 void sql_putl(int sd,long int l);
 void sql_putf(int sd,float f);
 void sql_putd(int sd,double d);
 void sql_putnull(int sd);

/* Get objects from the stream */

 void sql_getc(int sd,char* c);
 void sql_getuc(int sd,unsigned char* c);
 void sql_gets(int sd,char* s);
 void sql_getus(int sd,unsigned char* s);
 void sql_geti(int sd,int* n);
 void sql_getu(int sd,unsigned* u);
 void sql_getsh(int sd,short* sh);
 void sql_getl(int sd,long int* l);
 void sql_getf(int sd,float* f);
 void sql_getd(int sd,double* d);


/* printf/scanf-like functions */

 void sql_printf(int sd,char* fmt,...);
 void sql_scanf(int sd,char* fmt,...);

/* The following format specifiers are provided:
    %d  - int
    %u  - unsigned 
    %ld - long int
    %f  - float
    %lf - double
    %c  - char
    %s  - string

  for sql_printf the %N  specifier is provided, in order to put NULL
  into stream:

  sql_printf(sd,"%N");

*/

/* 

   Notes
   =====

The main advantage of using this C-interface is that it is very simple
and has  built-in transparent  Oracle  array  interface. The  size  of
internal  arrays  you define  in the   sql_open function call  by  the
BufSize parameter.

You don't need to link Oracle libraries to your application.

*/
#endif
